import 'dart:math';

import 'package:collection/collection.dart';
import 'package:flutter/material.dart';
import 'package:miniature_painting_companion/main.dart';
import 'package:miniature_painting_companion/views/design/app_paddings.dart';
import 'package:miniature_painting_companion/views/design/paint_computed_color.dart';

import '../../Models/hive_models.dart';
import '../design/app_sizes.dart';
import 'package:miniature_painting_companion/services/image_asset_service.dart';

int clampHue(double h, int segments) {
  return (h * segments).floor();
}

// biased luminosity algorithm for better sorting
// see https://devforum.roblox.com/t/a-journey-of-color-sorting/29876
double luminosity(HSVColor hsvColor) {
  final color = hsvColor.toColor();
  return sqrt(pow(color.r * 0.299, 2) +
      pow(color.g * 0.587, 2) +
      pow(color.b * 0.114, 2));
}

// Arbitrary number of segments
const int SEGMENTS = 8;

// Sort function
int sortLuma(HSVColor hsvA, HSVColor hsvB) {
  final clampedHA = clampHue(hsvA.hue, SEGMENTS);
  final clampedHB = clampHue(hsvB.hue, SEGMENTS);

  final lumA = luminosity(hsvA);
  final lumB = luminosity(hsvB);

  if (clampedHA == clampedHB) {
    return lumA.compareTo(lumB);
  } else {
    return clampedHA.compareTo(clampedHB);
  }
}

class TabbedPaintsList<T extends BasePaint> extends StatelessWidget {
  final void Function(T selected) onTap;
  final Iterable<T> paintsList;

  const TabbedPaintsList(
      {super.key, required this.onTap, required this.paintsList});

  @override
  Widget build(BuildContext context) {
    var paints = groupBy(paintsList, (paint) => paint.type);
    var paintTypes = paints.keys.sortedBy((type) => type.toLowerCase());

    return paints.isNotEmpty
        ? DefaultTabController(
            length: paintTypes.length,
            child: SizedBox(
              height: 400,
              width: 350,
              child: Scaffold(
                appBar: TabBar(
                  dividerColor: Colors.transparent,
                  isScrollable: true,
                  tabs: paintTypes
                      .map((type) => Tab(
                            text: type,
                          ))
                      .toList(),
                ),
                body: Padding(
                  padding: AppPaddings.top16,
                  child: TabBarView(
                    children: paintTypes.map((typeAct) {
                      var typePaints = paints[typeAct]!.sorted((act, other) {
                        if (act is! Paint ||
                            other is! Paint ||
                            act.getHsvColor() == null ||
                            other.getHsvColor() == null) {
                          return act.name.compareTo(other.name);
                        }

                        var actHsv = act.getHsvColor()!;
                        var otherHsv = other.getHsvColor()!;
                        return sortLuma(actHsv, otherHsv);
                      });
                      return GridView.builder(
                          gridDelegate:
                              SliverGridDelegateWithFixedCrossAxisCount(
                                  mainAxisExtent: 128,
                                  crossAxisSpacing: 8,
                                  mainAxisSpacing: 8,
                                  crossAxisCount: 4),
                          itemCount: typePaints.length,
                          itemBuilder: (context, index) {
                            var actual = typePaints[index];
                            return Material(
                              clipBehavior: Clip.antiAlias,
                              borderRadius:
                                  BorderRadius.all(Radius.circular(8)),
                              child: InkWell(
                                onTap: () {
                                  onTap(actual);
                                },
                                child: Column(
                                  mainAxisSize: MainAxisSize.min,
                                  children: [
                                    SizedBox(
                                      height: 60,
                                      child: switch (actual) {
                                        Paint() => actual.image != null
                                            ? ImageAssetService.getSvgIconOrPlaceholder(
                                                actual.image)
                                            : actual.getHsvColor() != null
                                                ? SizedBox(
                                                    height: AppSizes.size50,
                                                    width: AppSizes.size50,
                                                    child: CircleAvatar(
                                                      backgroundColor: actual
                                                          .getHsvColor()!
                                                          .toColor(),
                                                    ),
                                                  )
                                                : ImageAssetService.getPlaceHolderSvg(),
                                        UserPaint() =>
                                          PaintComputedColor(paint: actual),
                                      },
                                    ),
                                    Expanded(
                                        child: Text.rich(
                                      textAlign: TextAlign.center,
                                      maxLines: 3,
                                      overflow: TextOverflow.ellipsis,
                                      TextSpan(text: actual.name),
                                    ))
                                  ],
                                ),
                              ),
                            );
                          });
                    }).toList(),
                  ),
                ),
              ),
            ),
          )
        : SizedBox(
            height: 400,
            width: 350,
            child: Center(
              child: Text(localizations.noPaintFound),
            ),
          );
  }
}
