import 'dart:async';

import 'package:flutter/material.dart';
import 'package:fluttertoast/fluttertoast.dart';
import 'package:hive_ce_flutter/adapters.dart';
import 'package:miniature_painting_companion/services/image_asset_service.dart';
import 'package:miniature_painting_companion/views/design/app_paddings.dart';
import 'package:receive_sharing_intent/receive_sharing_intent.dart';

import '../../Models/hive_models.dart';
import '../../injector.dart';
import '../../main.dart';
import '../../utils/app_icons.dart';
import '../design/nothing_created_widget.dart';
import '../receivedShare/received_share_dialog_content.dart';
import 'miniature_widget.dart';
import 'new_miniature_dialog.dart';

class MiniaturesScreen extends StatefulWidget {
  const MiniaturesScreen({super.key, required this.title});

  final String title;

  @override
  State<MiniaturesScreen> createState() => _MiniaturesScreenState();
}

class _MiniaturesScreenState extends State<MiniaturesScreen> {
  late StreamSubscription _intentSub;
  final _sharedFiles = <SharedMediaFile>[];

  @override
  void dispose() {
    _intentSub.cancel();
    super.dispose();
  }

  @override
  void initState() {
    super.initState();

    _intentSub = ReceiveSharingIntent.instance.getMediaStream().listen((value) {
      setState(() {
        _sharedFiles.clear();
        _sharedFiles.addAll(value);
        if (_sharedFiles.isNotEmpty) {
          showAddIntentDialog();
        }
      });
    }, onError: (err) {
      Fluttertoast.showToast(
          msg: "Error while importing images", toastLength: Toast.LENGTH_LONG);
    });

    // Get the media sharing coming from outside the app while the app is closed.
    ReceiveSharingIntent.instance.getInitialMedia().then((value) {
      setState(() {
        _sharedFiles.clear();
        _sharedFiles.addAll(value);

        if (_sharedFiles.isNotEmpty) {
          showAddIntentDialog();
        }

        // Tell the library that we are done processing the intent.
        ReceiveSharingIntent.instance.reset();
      });
    });
  }

  void showAddIntentDialog() {
    showDialog<String>(
      context: context,
      builder: (BuildContext context) => Dialog(
        child: Padding(
          padding: AppPaddings.all16,
          child: SharingIntentContent(
            sharedFiles: _sharedFiles,
          ),
        ),
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    return MainScreenContent(
      title: widget.title,
    );
  }
}

class MainScreenContent extends StatefulWidget {
  final String title;

  const MainScreenContent({super.key, required this.title});

  @override
  State<MainScreenContent> createState() => _MainScreenContentState();
}

class _MainScreenContentState extends State<MainScreenContent> {
  bool sortByName = false;

  List<Miniature> getSortedContent() {
    var toSort = miniatureRepo.values().toList();
    toSort.sort((a, b) {
      if (a.favorite == true && (b.favorite == false || b.favorite == null)) {
        return -1;
      } else if ((a.favorite == false || a.favorite == null) &&
          b.favorite == true) {
        return 1;
      } else {
        if (sortByName) {
          return (a.name ?? "z").compareTo(b.name ?? "z");
        } else {
          // Sort by import time by default
          return a.importTime.compareTo(b.importTime);
        }
      }
    });
    return toSort;
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: Text(widget.title),
        actions: [
          IconButton(
            onPressed: () => setState(() {
              sortByName = !sortByName;
            }),
            icon: sortByName
                ? const Icon(Icons.sort_by_alpha)
                : const Icon(Icons.sort),
          )
        ],
      ),
      body: ValueListenableBuilder(
          valueListenable: miniatureRepo.listenable(),
          builder: (context, Box box, _) {
            if (box.length > 0) {
              return GridView.builder(
                padding: AppPaddings.bottom200,
                itemCount: box.length,
                shrinkWrap: true,
                itemBuilder: (context, listIndex) {
                  return MiniatureWidget(
                      miniatures: getSortedContent(), index: listIndex);
                },
                gridDelegate: const SliverGridDelegateWithFixedCrossAxisCount(
                  crossAxisCount: 2,
                ),
              );
            } else {
              return NothingCreatedWidget(
                title: localizations.noMiniatureCreated,
                onCreateClick: () {
                  NewMiniatureDialog.show(context, null);
                },
              );
            }
          }),
      floatingActionButton: FloatingActionButton(
        onPressed: () async => await NewMiniatureDialog.show(context, null),
        tooltip: localizations.addMiniature,
        child: ImageAssetService.getSvgIconSized(40, 40, AppIcons.plus),
      ),
    );
  }
}
