import 'dart:developer';

import 'package:get_it/get_it.dart';
import 'package:miniature_painting_companion/Models/hive_models.dart';
import 'package:miniature_painting_companion/Models/layers_model.dart';
import 'package:miniature_painting_companion/Models/paints_loader.dart';
import 'package:miniature_painting_companion/services/file_storage_service.dart';
import 'package:miniature_painting_companion/services/image_picker_service.dart';
import 'package:shared_preferences/shared_preferences.dart';

import 'repositories/paint_manufacturer_repository.dart';

final getIt = GetIt.instance;

Future<void> setupInjector() async {
  final sharedPreferences = await SharedPreferences.getInstance();
  getIt.registerSingleton<SharedPreferences>(sharedPreferences);

  getIt.registerLazySingleton<PaintManufacturerSettingsRepo>(
    () => PaintManufacturerSettingsRepo(getIt<SharedPreferences>()),
  );

  getIt.registerSingleton<FileStorageService>(FileStorageService());
  getIt.registerSingleton<ImagePickerService>(ImagePickerService());

  final paintRepo = HiveRepository<BasePaint>("Paint");
  await paintRepo.init();
  getIt.registerSingleton<HiveRepository<BasePaint>>(paintRepo);

  final iDrawableRepo = HiveRepository<IDrawableLayer>("IDrawableLayer");
  await iDrawableRepo.init();
  getIt.registerSingleton<HiveRepository<IDrawableLayer>>(iDrawableRepo);

  final positionedRepo = HiveRepository<PositionedLayer>("PositionedLayer");
  await positionedRepo.init();
  getIt.registerSingleton<HiveRepository<PositionedLayer>>(positionedRepo);

  try {
    await PaintLoader.loadCitadelPaintList(paintRepo);
    await PaintLoader.loadTamiyaPaintList(paintRepo);
  } catch (e) {
    log("Warn could not load any paint", error: e);
  }

  final paintJobRepo = HiveRepository<PaintJob>("PaintJob");
  await paintJobRepo.init();
  getIt.registerSingleton<HiveRepository<PaintJob>>(paintJobRepo);

  final miniatureRepo = HiveRepository<Miniature>("Miniature");
  await miniatureRepo.init();
  getIt.registerSingleton<HiveRepository<Miniature>>(miniatureRepo);

  final historyImagesRepo = HiveRepository<HistoryImage>("HistoryImage");
  await historyImagesRepo.init();
  getIt.registerSingleton<HiveRepository<HistoryImage>>(historyImagesRepo);
}

// Global repository getters for cleaner code
HiveRepository<BasePaint> get paintRepo => getIt<HiveRepository<BasePaint>>();

HiveRepository<IDrawableLayer> get iDrawableRepo =>
    getIt<HiveRepository<IDrawableLayer>>();

HiveRepository<PositionedLayer> get positionedRepo =>
    getIt<HiveRepository<PositionedLayer>>();

HiveRepository<PaintJob> get paintJobRepo => getIt<HiveRepository<PaintJob>>();

HiveRepository<Miniature> get miniatureRepo =>
    getIt<HiveRepository<Miniature>>();

HiveRepository<HistoryImage> get historyImagesRepo =>
    getIt<HiveRepository<HistoryImage>>();

PaintManufacturerSettingsRepo get paintManufacturerSettingsRepo =>
    getIt<PaintManufacturerSettingsRepo>();

FileStorageService get fileStorageService => getIt<FileStorageService>();

ImagePickerService get imagePickerService => getIt<ImagePickerService>();
