import 'dart:io';

import 'package:copy_with_extension/copy_with_extension.dart';
import 'package:flutter/material.dart';
import 'package:flutter/widgets.dart';
import 'package:hive_ce/hive.dart';
import 'package:miniature_painting_companion/injector.dart';

import '../views/Layers/combined/combined_layer_widget.dart';
import '../views/Layers/paint_layer/paint_layer_widget.dart';
import '../views/Layers/separator/separator_layer_widget.dart';
import '../views/Layers/sticker/sticker_layer_widget.dart';
import 'hive_models.dart';

part 'layers_model.g.dart';

sealed class IDrawableLayer extends HiveObject implements Cloneable {
  Widget getWidget(Key? key);

  Map<String, dynamic> toJson();

  static Future<IDrawableLayer> genericLayerFromJson(
      Map<String, dynamic> layerJson) async {
    final String type = layerJson['type'];
    final Map<String, dynamic> data =
        Map<String, dynamic>.from(layerJson['data']);

    IDrawableLayer layer;
    switch (type) {
      case 'PaintLayer':
        layer = await PaintLayer.fromJson(data);
        break;
      case 'CombinedLayer':
        layer = await CombinedLayer.fromJson(data);
        break;
      case 'LayerSeparator':
        layer = await LayerSeparator.fromJson(data);
        break;
      case 'StickerLayer':
        layer = await StickerLayer.fromJson(data);
        break;
      default:
        throw Exception("Unsupported layer type: $type");
    }
    return layer;
  }
}

// After update launch: dart pub run build_runner build --delete-conflicting-outputs
@CopyWith()
class PaintLayer extends IDrawableLayer {
  String name;
  BasePaint paint;
  String note;
  ApplicationType applicationType;
  int dilution;

  PaintLayer(
      {required this.name,
      required this.paint,
      required this.note,
      required this.applicationType,
      required this.dilution});

  @override
  Widget getWidget(Key? key) {
    return PaintLayerWidget(key: key, layer: this);
  }

  @override
  Future<PaintLayer> clone() async {
    return await PaintLayer.fromJson(toJson());
  }

  @override
  Map<String, dynamic> toJson() {
    return {
      'name': name,
      'paint': paint.name,
      'paintId': paint.id,
      'note': note,
      'applicationType': applicationType.name, // or .index if preferred
      'dilution': dilution,
    };
  }

  static Future<PaintLayer> fromJson(Map<String, dynamic> json) async {
    final layer = PaintLayer(
      name: json['name'],
      paint: paintRepo.values().firstWhere(
            (act) => act.id == (json['paintId'] as String?),
            // fallback for exports made before paintId existed
            orElse: () => paintRepo.values().whereType<Paint>().firstWhere(
                  (act) => act.name == (json['paint'] as String),
                  // fallback if paint really does not exists
                  orElse: () => Paint(
                    type: "LostType",
                    name: json['paint'],
                  ),
                ),
          ),
      note: json['note'],
      applicationType: ApplicationType.values.firstWhere(
        (e) => e.name == json['applicationType'], // or use .index
      ),
      dilution: json['dilution'],
    );

    await iDrawableRepo.add(layer);
    return layer;
  }
}

@CopyWith()
class LayerSeparator extends IDrawableLayer {
  String text;

  LayerSeparator(this.text);

  @override
  Widget getWidget(Key? key) {
    return SeparatorLayerWidget(key: key, separator: this);
  }

  @override
  Future<LayerSeparator> clone() async {
    return await LayerSeparator.fromJson(toJson());
  }

  @override
  Map<String, dynamic> toJson() {
    return {
      'text': text,
    };
  }

  static Future<LayerSeparator> fromJson(Map<String, dynamic> json) async {
    final separator = LayerSeparator(json['text']);
    await iDrawableRepo.add(separator);
    return separator;
  }
}

@CopyWith()
class CombinedLayer extends HiveObject implements IDrawableLayer {
  String name;
  HiveList<IDrawableLayer> layers;

  CombinedLayer({required this.name, required this.layers});

  @override
  Widget getWidget(Key? key) {
    return CombinedLayerWidget(
      key: key,
      layer: this,
    );
  }

  Future<void> addLayer(IDrawableLayer layer) async {
    await iDrawableRepo.add(layer);
    layers.add(layer);
    await iDrawableRepo.save(this);
  }

  @override
  Future<CombinedLayer> clone() async {
    return await CombinedLayer.fromJson(toJson());
  }

  @override
  Map<String, dynamic> toJson() {
    return {
      'name': name,
      'layers': layers.map((layer) => layer.toJson()).toList(),
    };
  }

  static Future<CombinedLayer> fromJson(Map<String, dynamic> json) async {
    final combined = CombinedLayer(
      name: json['name'],
      layers: iDrawableRepo.createList(),
    );
    await iDrawableRepo
        .add(combined); // or wherever you're storing CombinedLayers

    for (var layerJson in json['layers'] ?? []) {
      final layer =
          await PaintLayer.fromJson(Map<String, dynamic>.from(layerJson));
      combined.layers.add(layer);
    }

    await iDrawableRepo.save(combined);
    return combined;
  }

  @override
  Future<void> delete() async {
    for (var act in layers) {
      await act.delete();
    }
    await super.delete();
  }
}

@CopyWith()
class StickerLayer extends IDrawableLayer {
  String name;
  String? imagePath;

  StickerLayer(this.name, this.imagePath);

  @override
  Widget getWidget(Key? key) {
    return StickerLayerWidget(key: key, sticker: this);
  }

  @override
  Future<StickerLayer> clone() async {
    return await StickerLayer.fromJson(toJson());
  }

  @override
  Map<String, dynamic> toJson() {
    return {
      'name': name,
      'imagePath': imagePath,
      'imageBase64': fileStorageService.filePathToBase64(imagePath),
    };
  }

  static Future<StickerLayer> fromJson(Map<String, dynamic> json) async {
    final layer = StickerLayer(
      json['name'],
      fileStorageService.base64toFile(json['imagePath'], json['imageBase64']),
    );

    await iDrawableRepo.add(layer);
    return layer;
  }

  @override
  Future<void> delete() async {
    if (imagePath != null) {
      File(imagePath!).deleteSync();
    }

    await super.delete();
  }
}
