import 'package:collection/collection.dart';
import 'package:flutter/material.dart';
import 'package:flutter_svg/svg.dart';
import 'package:miniature_painting_companion/extensions/string_extension.dart';
import 'package:miniature_painting_companion/views/paints/tabbed_paints_list.dart';

import '../../Models/hive_models.dart';
import '../../injector.dart';
import '../../main.dart';
import '../../repositories/paint_manufacturer_repository.dart';
import '../design/app_paddings.dart';
import '../design/app_sizes.dart';

class PaintPickerWidget<T extends BasePaint> extends StatefulWidget {
  final TextEditingController paintSearchController;
  final Future<void> Function(T picked) onPicked;

  const PaintPickerWidget(
      {super.key, required this.onPicked, required this.paintSearchController});

  @override
  State<PaintPickerWidget> createState() =>
      _PaintPickerWidgetState<T>(onPicked);
}

class _PaintPickerWidgetState<S extends BasePaint>
    extends State<PaintPickerWidget> {
  final Future<void> Function(S picked) onPicked;
  final PaintManufacturerSettingsRepo _repository =
      getIt<PaintManufacturerSettingsRepo>();

  String _selectedManufacturer = PaintManufacturer.citadel.name.toLowerCase();
  List<PaintManufacturer> _enabledManufacturers = [];

  _PaintPickerWidgetState(this.onPicked);

  @override
  void initState() {
    super.initState();
    _loadManufacturers();
  }

  Future<void> _loadManufacturers() async {
    final manufacturers = await _repository.loadManufacturers();
    final enabled = manufacturers.entries
        .where((entry) => entry.value)
        .map((entry) => entry.key)
        .toList();
    enabled.add(PaintManufacturer.custom);

    setState(() {
      _enabledManufacturers = enabled;
      // Ensure selected manufacturer is valid
      if (!_enabledManufacturers.any(
          (m) => m.name.toLowerCase() == _selectedManufacturer.toLowerCase())) {
        if (_enabledManufacturers.isNotEmpty) {
          _selectedManufacturer =
              _enabledManufacturers.first.name.toLowerCase();
        } else {
          _selectedManufacturer =
              'None'; // Handle case with no enabled manufacturers
        }
      }
    });
  }

  @override
  Widget build(BuildContext context) {
    final paintsByManufacturer = paintRepo
        .values()
        .where(
          (paintAct) =>
              widget.paintSearchController.text.isEmpty ||
              paintAct.name
                  .toLowerCase()
                  .contains(widget.paintSearchController.text.toLowerCase()),
        )
        .whereType<S>()
        .groupListsBy((paint) => paint.manufacturer.name.toLowerCase());

    final manufacturerWithPaintsInSearch = paintsByManufacturer.keys.toList();

    final selectedManufIfAvailable = manufacturerWithPaintsInSearch.firstWhere(
        (act) => act.toLowerCase() == _selectedManufacturer.toLowerCase(),
        orElse: () => manufacturerWithPaintsInSearch.firstOrNull ?? 'None');

    final filteredPaints = paintsByManufacturer[selectedManufIfAvailable]
            ?.toList()
            .where(
              (paintAct) =>
                  widget.paintSearchController.text.isEmpty ||
                  paintAct.name.toLowerCase().contains(
                      widget.paintSearchController.text.toLowerCase()),
            )
            .whereType<S>() ??
        [];

    return Column(
      children: [
        Row(
          spacing: AppSizes.size8,
          children: [
            Expanded(
              child: SearchBar(
                padding: const WidgetStatePropertyAll(
                    AppPaddings.symmetricHorizontal16),
                hintText: localizations
                    .search_paints(_selectedManufacturer.capitalize()),
                shadowColor: const WidgetStatePropertyAll(Colors.transparent),
                controller: widget.paintSearchController,
                trailing: widget.paintSearchController.text.isNotEmpty
                    ? [
                        Material(
                          color: Colors.transparent,
                          borderRadius: BorderRadius.circular(50),
                          child: InkWell(
                            onTap: () {
                              widget.paintSearchController.clear();
                              setState(() {});
                            },
                            child: SvgPicture.asset(
                              "assets/close.svg",
                              colorFilter: const ColorFilter.mode(
                                  Colors.grey, BlendMode.srcIn),
                              width: AppSizes.size30,
                              height: AppSizes.size30,
                            ),
                          ),
                        )
                      ]
                    : [],
                onChanged: (_) => setState(() {}),
              ),
            ),
            if (manufacturerWithPaintsInSearch.isNotEmpty &&
                // Kind of a hack but had a crash when only use the isNotEmpty
                selectedManufIfAvailable != 'None')
              DropdownButton<String>(
                value: selectedManufIfAvailable,
                items: manufacturerWithPaintsInSearch.map((manufacturer) {
                  return DropdownMenuItem(
                    value: manufacturer.toLowerCase(),
                    child: Text(manufacturer.capitalize()),
                  );
                }).toList(),
                onChanged: (value) {
                  if (value != null) {
                    setState(() {
                      _selectedManufacturer = value;
                    });
                  }
                },
              ),
          ],
        ),
        TabbedPaintsList<S>(
          paintsList: filteredPaints,
          onTap: (actual) {
            onPicked(actual);
          },
        ),
      ],
    );
  }
}
