import 'package:flutter/material.dart';
import 'package:flutter_svg/svg.dart';
import 'package:hive_ce/hive.dart';
import 'package:miniature_painting_companion/Models/hive_models.dart';
import 'package:miniature_painting_companion/extensions/string_extension.dart';
import 'package:miniature_painting_companion/main.dart';
import 'package:miniature_painting_companion/utils.dart';
import 'package:miniature_painting_companion/utils/app_icons.dart';
import 'package:miniature_painting_companion/views/design/app_sizes.dart';
import 'package:miniature_painting_companion/views/paints/paint_picker_widget.dart';
import 'package:rflutter_alert/rflutter_alert.dart';

import '../../injector.dart';
import '../design/app_paddings.dart';
import '../design/paint_computed_color.dart';
import '../design/slidable_card.dart';
import 'custom_paint_create_dialog.dart';
import 'package:miniature_painting_companion/services/image_asset_service.dart';

Widget _getPaintImage(String? path) {
  if (path == null) {
    return ImageAssetService.getSvgIconSized(
        AppSizes.size50, AppSizes.size50, AppIcons.placeholderFilled);
  } else {
    return SvgPicture.string(path);
  }
}

class CustomPaintWidget extends StatefulWidget {
  final UserPaint paint;

  const CustomPaintWidget({super.key, required this.paint});

  @override
  State<CustomPaintWidget> createState() => _CustomPaintWidgetState();
}

class _CustomPaintWidgetState extends State<CustomPaintWidget> {
  final DILUTION = "dilution";
  late final TextEditingController paintSearchController =
      TextEditingController();

  @override
  Widget build(BuildContext context) {
    var cardContent = Column(
      children: [
        CustomPaintHeader(widget: widget),
        Padding(
          padding: AppPaddings.h16b16,
          child: Column(
            children: [
              ValueListenableBuilder(
                valueListenable:
                    iDrawableRepo.listenable(),
                builder: (context, Box box, _) => Container(
                  decoration: BoxDecoration(
                      color: Colors.grey.withAlpha(50),
                      borderRadius: BorderRadius.circular(16)),
                  child: Column(
                    children: [
                      ListView.builder(
                        shrinkWrap: true,
                        itemCount: widget.paint.components.length,
                        physics: const ClampingScrollPhysics(),
                        itemBuilder: (_, index) {
                          return ComponentWidget(
                              holder: widget.paint, index: index);
                        },
                        padding: AppPaddings.all8,
                      ),
                      AddComponentButton(
                          widget: widget,
                          paintSearchController: paintSearchController),
                      const SizedBox(height: AppSizes.size16),
                    ],
                  ),
                ),
              ),
            ],
          ),
        )
      ],
    );

    return SlidableCard(
        startAction: (context) async =>
            {await paintRepo.delete(widget.paint)},
        endAction: (otherContext) =>
            {CustomPaintCreateDialog.show(context, widget.paint)},
        child: cardContent);
  }
}

class ComponentWidget extends StatelessWidget {
  const ComponentWidget({super.key, required this.holder, required this.index});

  final UserPaint holder;
  final int index;

  @override
  Widget build(BuildContext context) {
    return SlidableCard(
      startAction: (context) async => {
        holder.components.removeAt(index),
        paintRepo.save(holder)
      },
      endAction: (otherContext) async => {
        showPaintPicker(context, TextEditingController(), (Paint picked) async {
          await holder.setComponent(
              UserPaintComponent(paint: picked, concentration: 100), index);
          Navigator.pop(context);
        })
      },
      child: Padding(
        padding: AppPaddings.all16,
        child: Column(
          children: [
            Center(
              child: Utils.gothicText(holder.components[index].paint.name),
            ),
            Row(
              children: [
                _getPaintImage(holder.components[index].paint.image),
                Expanded(
                  child: Slider(
                    min: 0,
                    max: 100,
                    divisions: 10,
                    onChanged: (double value) {
                      holder.components[index].concentration = value;
                      paintRepo.save(holder);
                    },
                    label: holder.components[index].concentration.toString(),
                    value: holder.components[index].concentration,
                  ),
                )
              ],
            ),
          ],
        ),
      ),
    );
  }
}

class AddComponentButton extends StatelessWidget {
  const AddComponentButton({
    super.key,
    required this.widget,
    required this.paintSearchController,
  });

  final CustomPaintWidget widget;
  final TextEditingController paintSearchController;

  @override
  Widget build(BuildContext context) {
    return Ink(
      decoration: const ShapeDecoration(
        color: Colors.blueGrey,
        shape: CircleBorder(),
      ),
      child: IconButton(
        icon: ImageAssetService.getSvgIconSized(
            AppSizes.size30, AppSizes.size30, AppIcons.plus),
        onPressed: () {
          showPaintPicker(context, paintSearchController, (Paint picked) async {
            await widget.paint.addComponent(
                UserPaintComponent(paint: picked, concentration: 100));
          });
        },
      ),
    );
  }
}

void showPaintPicker(
    BuildContext context,
    TextEditingController paintSearchController,
    Future<void> Function(Paint picked) onPicked) {
  Alert(
      useRootNavigator: false,
      closeIcon: CircleAvatar(
        backgroundColor: Colors.grey.withOpacity(0.30),
        child: Icon(
          Icons.close_rounded,
        ),
      ),
      // removes the default theme of the extension which doesn't handle dark theme
      style: AlertStyle(
        titleStyle: TextStyle(fontFamily: "Gothic"),
        titlePadding: AppPaddings.bottom24,
        alertPadding: AppPaddings.symmetricHorizontal16,
      ),
      context: context,
      title: localizations.pickPaint,
      content: Column(
        children: [
          PaintPickerWidget(
            onPicked: (Paint picked) {
              Navigator.pop(context);
              return onPicked(picked);
            },
            paintSearchController: paintSearchController,
          )
        ],
      ),
      buttons: []).show();
}

class CustomPaintHeader extends StatelessWidget {
  const CustomPaintHeader({
    super.key,
    required this.widget,
  });

  final CustomPaintWidget widget;

  @override
  Widget build(BuildContext context) {
    return Row(
      children: [
        const SizedBox(width: AppSizes.size24),
        Center(
          child: PaintComputedColor(
            paint: widget.paint,
          ),
        ),
        const SizedBox(width: AppSizes.size8),
        Expanded(
            child: Container(
          alignment: Alignment.center,
          child: Column(
            children: [
              const SizedBox(height: AppSizes.size8),
              Utils.gothicText(widget.paint.name.capitalize()),
              const SizedBox(height: AppSizes.size8),
              Visibility(
                visible: widget.paint.type.isNotEmpty,
                child: Utils.gothicText(widget.paint.type),
              ),
              const SizedBox(height: AppSizes.size8),
            ],
          ),
        )),
        const SizedBox(width: AppSizes.size24),
      ],
    );
  }
}
