import 'package:flutter/material.dart';
import 'package:flutter_svg/svg.dart';
import 'package:miniature_painting_companion/Models/hive_models.dart';
import 'package:miniature_painting_companion/Models/layers_model.dart';
import 'package:miniature_painting_companion/extensions/string_extension.dart';
import 'package:miniature_painting_companion/views/design/app_paddings.dart';
import 'package:miniature_painting_companion/views/paints/paint_picker_widget.dart';
import 'package:reactive_forms/reactive_forms.dart';
import 'package:uuid/uuid.dart';

import '../../../main.dart';
import 'package:miniature_painting_companion/injector.dart';
import '../../design/app_sizes.dart';
import '../../design/reactive_crud_dialog.dart';

final PAINT_NAME = "paintName";
final PAINT_KEY = "paintKey";
final PAINT_TYPE = "paintType";
final PAINT_IMAGE = "paintImage";
final NOTE = "note";
final APPLICATION_TYPE = "applicationType";
final DILUTION = "dilution";

class PaintLayerCreationWidget extends StatefulWidget {
  final PaintLayer? layerToEdit;
  final FormGroup form;
  bool showColorPicker;

  PaintLayerCreationWidget(
      {super.key,
      this.layerToEdit,
      required this.showColorPicker,
      required this.form});

  @override
  State<StatefulWidget> createState() {
    return PaintLayerCreationState();
  }
}

class PaintLayerCreationState extends State<PaintLayerCreationWidget> {
  late final TextEditingController paintSearchController =
      TextEditingController();

  @override
  Widget build(BuildContext context) {
    return widget.showColorPicker
        ? PaintPickerWidget<BasePaint>(
            paintSearchController: paintSearchController,
            onPicked: (BasePaint actual) async {
              setState(() {
                widget.showColorPicker = false;
                widget.form.control(PAINT_KEY).value = actual.key;
                widget.form.control(PAINT_NAME).value = actual.name;
                widget.form.control(PAINT_TYPE).value = actual.type;
                if (actual is Paint) {
                  widget.form.control(PAINT_IMAGE).value =
                      actual.image == null ? "" : actual.image!;
                }
              });
            })
        : PaintCustomisationWidget(
            form: widget.form,
            onActionButton: () {
              setState(() {
                widget.showColorPicker = true;
              });
            },
          );
  }
}

class PaintCustomisationWidget extends StatelessWidget {
  const PaintCustomisationWidget(
      {super.key, required this.form, required this.onActionButton});

  final FormGroup form;
  final void Function() onActionButton;

  @override
  Widget build(BuildContext context) {
    return Column(
      children: [
        Container(
          decoration: BoxDecoration(
            color: Colors.blueAccent.withAlpha(60),
            borderRadius: BorderRadius.circular(20),
          ),
          child: Padding(
            padding: AppPaddings.all16,
            child: Column(
              children: [
                Row(
                  children: [
                    Expanded(
                      child: Text(form.control(PAINT_NAME).value),
                    ),
                    SizedBox(
                      width: AppSizes.size8,
                    ),
                    IconButton.filled(
                      onPressed: onActionButton,
                      icon: SvgPicture.asset(
                          colorFilter:
                              ColorFilter.mode(Colors.white, BlendMode.srcIn),
                          height: AppSizes.size24,
                          "assets/color_picker.svg"),
                    )
                  ],
                ),
                Row(
                  children: [
                    Padding(
                        padding: AppPaddings.symmetricVertical8,
                        child: Text(
                          style: TextStyle(
                            color: Colors.black54,
                          ),
                          form.control(PAINT_TYPE).value,
                        )),
                  ],
                )
              ],
            ),
          ),
        ),
        SizedBox(
          height: AppSizes.size24,
        ),
        Padding(
          padding: AppPaddings.symmetricVertical8,
          child: Row(
            children: [
              Text.rich(
                TextSpan(text: "Application"),
                textAlign: TextAlign.start,
              ),
              SizedBox(
                width: AppSizes.size24,
              ),
              Expanded(
                  child: Align(
                alignment: Alignment.centerRight,
                // child: PaintDropdown(paintNotifier),
                child: ReactiveDropdownField(
                    decoration: InputDecoration(border: InputBorder.none),
                    formControlName: APPLICATION_TYPE,
                    items: ApplicationType.values
                        .map((e) => DropdownMenuItem<ApplicationType>(
                            value: e,
                            child: Text(e.name.camelToSentence().capitalize())))
                        .toList()),
              ))
            ],
          ),
        ),
        Padding(
          padding: AppPaddings.symmetricVertical8,
          child: Row(
            children: [
              Text.rich(
                TextSpan(text: localizations.dilution),
                textAlign: TextAlign.start,
              ),
              ReactiveSlider(
                formControlName: DILUTION,
                min: 0,
                max: 100,
                divisions: 10,
                labelBuilder: (value) => value.toString(),
              )
            ],
          ),
        ),
        Padding(
          padding: AppPaddings.symmetricVertical8,
          child: ReactiveTextField(
            formControlName: NOTE,
            decoration: InputDecoration(
              labelText: localizations.note,
              hintText: localizations.note,
            ),
          ),
        ),
      ],
    );
  }
}

class NewPaintLayerDialog {
  /// [saveFunction] is used to handle both saving normal and nested paintLayer (ie. in combined layers)
  static Future<bool?> show(dialogContext, startOnColorPicker,
      PaintLayer? layerToEdit, Function(PaintLayer) saveFunction) {
    var key = GlobalKey<PaintLayerCreationState>();

    final form = FormGroup({
      PAINT_KEY: FormControl<dynamic>(
          value: layerToEdit?.paint.key, validators: [Validators.required]),
      PAINT_TYPE: FormControl<String>(
          value: layerToEdit?.paint.type, validators: [Validators.required]),
      PAINT_NAME: FormControl<String>(
          value: layerToEdit?.paint.name, validators: [Validators.required]),
      PAINT_IMAGE: FormControl<String?>(
          value: (layerToEdit?.paint is Paint)
              ? (layerToEdit?.paint as Paint).image
              : null),
      NOTE: FormControl<String>(value: layerToEdit?.note),
      APPLICATION_TYPE: FormControl<ApplicationType>(
          value: ApplicationType.normal, validators: [Validators.required]),
      DILUTION: FormControl<int>(value: 100, validators: [Validators.required]),
    });

    var content = [
      PaintLayerCreationWidget(
        key: key,
        showColorPicker: startOnColorPicker,
        layerToEdit: layerToEdit,
        form: form,
      )
    ];

    Future<void> onUpdate(BuildContext context) async {
      var paint = paintRepo.get(
          form.control(PAINT_KEY).value,
          defaultValue: UserPaint(
              id: Uuid().v4(),
              type: form.control(PAINT_TYPE).value,
              name: form.control(PAINT_NAME).value,
              components: []))!;

      layerToEdit?.name = form.control(PAINT_NAME).value;
      layerToEdit?.paint = paint;
      layerToEdit?.note = (form.control(NOTE).value as String?) ?? "";
      layerToEdit?.applicationType = form.control(APPLICATION_TYPE).value;
      layerToEdit?.dilution = form.control(DILUTION).value;
      await iDrawableRepo.save(layerToEdit);
    }

    void onSave(BuildContext context) {
      var paint = paintRepo.get(
          form.control(PAINT_KEY).value,
          defaultValue: UserPaint(
              id: Uuid().v4(),
              type: form.control(PAINT_TYPE).value,
              name: form.control(PAINT_NAME).value,
              components: []))!;

      var createdLayer = PaintLayer(
          name: form.control(PAINT_NAME).value,
          paint: paint,
          note: (form.control(NOTE).value as String?) ?? "",
          applicationType: form.control(APPLICATION_TYPE).value,
          dilution: form.control(DILUTION).value);
      saveFunction(createdLayer);
    }

    return ReactiveCRUDDialog.show(
        dialogContext,
        layerToEdit,
        content,
        form,
        localizations.createPaintJob,
        localizations.updatePaintJob,
        onSave,
        onUpdate);
  }
}
