import 'dart:io';

import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:hive_ce_flutter/adapters.dart';
import 'package:miniature_painting_companion/Models/hive_models.dart';
import 'package:miniature_painting_companion/views/Layers/positioned_layers_screen.dart';
import 'package:miniature_painting_companion/views/Layers/zoom_circle.dart';
import 'package:miniature_painting_companion/views/design/app_sizes.dart';

import '../../injector.dart';
import '../../utils/app_icons.dart';
import 'package:miniature_painting_companion/services/image_asset_service.dart';

final TAG_SIZE = AppSizes.size48;
final HALF_SIZE = TAG_SIZE / 2.0;
final ZOOM_BOX_SIZE = AppSizes.size60;

class ImageTagger extends StatefulWidget {
  final String imagePath;
  final PaintJob holder;

  const ImageTagger({super.key, required this.imagePath, required this.holder});

  @override
  _ImageTaggerState createState() => _ImageTaggerState();
}

class _ImageTaggerState extends State<ImageTagger> {
  int? _draggingIndex;
  late RenderBox _imageBox;

  @override
  void didChangeDependencies() {
    super.didChangeDependencies();
    WidgetsBinding.instance.addPostFrameCallback((_) {
      setState(() {
        _imageBox = context.findRenderObject() as RenderBox;
      });
    });
  }

  Offset clampToBox(Offset offset, RenderBox box) {
    return Offset(offset.dx.clamp(0, box.size.width),
        offset.dy.clamp(0, box.size.height));
  }

  Future<PositionedLayer> _addTag(Offset position, BuildContext context) async {
    final RenderBox box = context.findRenderObject() as RenderBox;
    final Offset localPosition = box.globalToLocal(position);
    final Offset clampedPosition = clampToBox(localPosition, box);

    var created = PositionedLayer(
        dx: clampedPosition.dx,
        dy: clampedPosition.dy,
        layers: iDrawableRepo.createList());
    widget.holder.addPositioned(created);
    setState(() {});

    return created;
  }

  void navigateToTag(PositionedLayer created) {
    Navigator.push(
      context,
      MaterialPageRoute(
          builder: (context) => PositionedLayerScreen(
                originalImageSize: _imageBox.size,
                zoomPosition: getZoomPosition(
                    clampToBox(Offset(created.dx, created.dy), _imageBox)),
                paintJob: widget.holder,
                layer: created,
              )),
    );
  }

  Offset getZoomPosition(Offset clampedPosition) {
    return Offset(((clampedPosition.dx) / _imageBox.size.width) * 2 - 1,
        ((clampedPosition.dy) / _imageBox.size.height) * 2 - 1);
  }

  void _startDragging(int index) {
    HapticFeedback.lightImpact();
    setState(() {
      _draggingIndex = index;
    });
  }

  void _updateTagPosition(LongPressMoveUpdateDetails details) {
    if (_draggingIndex != null) {
      setState(() {
        final RenderBox box = context.findRenderObject() as RenderBox;
        final Offset localPosition = box.globalToLocal(details.globalPosition);
        final Offset clampedPosition = clampToBox(localPosition, box);

        widget.holder.positioned[_draggingIndex!].dx = clampedPosition.dx;
        widget.holder.positioned[_draggingIndex!].dy = clampedPosition.dy;
      });
    }
  }

  Future<void> _stopDragging() async {
    await positionedRepo.save(widget.holder.positioned[_draggingIndex!]);
    setState(() {
      _draggingIndex = null;
    });
  }

  @override
  Widget build(BuildContext context) {
    return ValueListenableBuilder(
      valueListenable: positionedRepo.listenable(),
      builder: (BuildContext context, Box value, Widget? child) =>
          GestureDetector(
        onTapUp: (details) async {
          final created = await _addTag(details.globalPosition, context);
          navigateToTag(created);
        },

        onLongPressStart: (details) async {
          await _addTag(details.globalPosition, context);
          _startDragging(widget.holder.positioned.length);
        },
        // also handle move here since for newly created layers we drag on the image not on the layer
        onLongPressMoveUpdate: (newPosition) => _updateTagPosition(newPosition),
        onLongPressEnd: (_) {
          _stopDragging();
          navigateToTag(
              widget.holder.positioned[widget.holder.positioned.length - 1]);
        },

        behavior: HitTestBehavior.opaque,
        child: Stack(
          clipBehavior: Clip.none,
          children: [
            Card(
                child: Image.file(File(widget.imagePath), fit: BoxFit.contain)),
            ...widget.holder.positioned.asMap().entries.map((entry) {
              int index = entry.key;
              Offset tag =
                  clampToBox(Offset(entry.value.dx, entry.value.dy), _imageBox);
              bool isDragging = _draggingIndex == index;

              return Positioned(
                left: tag.dx - HALF_SIZE,
                top: tag.dy - HALF_SIZE,
                child: GestureDetector(
                  onTap: () async {
                    navigateToTag(entry.value);
                    setState(() {});
                  },
                  behavior: HitTestBehavior.opaque,
                  onLongPressStart: (_) => _startDragging(index),
                  onLongPressMoveUpdate: (newPosition) =>
                      _updateTagPosition(newPosition),
                  onLongPressEnd: (_) => _stopDragging(),
                  child: Stack(
                    clipBehavior: Clip.none,
                    children: [
                      ImageAssetService.getSvgIconSized(TAG_SIZE, TAG_SIZE, AppIcons.loupe),
                      if (isDragging)
                        Positioned(
                          left: -HALF_SIZE / 4,
                          top: -ZOOM_BOX_SIZE - AppSizes.size10,
                          child: ZoomCircle(
                            border: Border.all(color: Colors.black, width: 2),
                            imagePath: widget.imagePath,
                            originalSize: _imageBox.size,
                            zoomCenter: getZoomPosition(tag),
                            radius: ZOOM_BOX_SIZE,
                            zoomFactor: 2.0,
                          ),
                        ),
                    ],
                  ),
                ),
              );
            }),
          ],
        ),
      ),
    );
  }
}
