import 'dart:typed_data';
import 'dart:io';

import 'package:flutter/material.dart';
import 'package:fluttertoast/fluttertoast.dart';
import 'package:image_picker/image_picker.dart';
import 'package:permission_handler/permission_handler.dart';
import 'package:pro_image_editor/models/editor_callbacks/pro_image_editor_callbacks.dart';
import 'package:pro_image_editor/models/editor_configs/pro_image_editor_configs.dart';
import 'package:pro_image_editor/modules/main_editor/main_editor.dart';

class ImagePickerService {
  Future<XFile?> pickImage(ImageSource source) async {
    if (source != ImageSource.camera ||
        await Permission.camera.request().isGranted) {
      final ImagePicker picker = ImagePicker();
      return await picker.pickImage(source: source, requestFullMetadata: true);
    }

    Fluttertoast.showToast(
        msg: "Camera permission denied, can't take a picture",
        toastLength: Toast.LENGTH_LONG);
    return null;
  }

  Future<List<XFile>?> pickImages(ImageSource source) async {
    if (source != ImageSource.camera ||
        await Permission.camera.request().isGranted) {
      final ImagePicker picker = ImagePicker();
      return await picker.pickMultiImage(requestFullMetadata: true);
    }

    Fluttertoast.showToast(
        msg: "Camera permission denied, can't take a picture",
        toastLength: Toast.LENGTH_LONG);
    return null;
  }

  void showImageEditor(String imagePath, BuildContext context,
      {VoidCallback? additionalCallback}) {
    var imageFile = File(imagePath);
    Navigator.push(
      context,
      MaterialPageRoute(
        builder: (context) => ProImageEditor.file(
          imageFile,
          configs: ProImageEditorConfigs(
              filterEditorConfigs: FilterEditorConfigs(enabled: false),
              blurEditorConfigs: BlurEditorConfigs(enabled: false),
              emojiEditorConfigs: EmojiEditorConfigs(enabled: false)),
          callbacks: ProImageEditorCallbacks(
              onImageEditingComplete: (Uint8List bytes) async {
            imageFile.writeAsBytesSync(bytes);
            await FileImage(imageFile).evict();
            if (additionalCallback != null) {
              additionalCallback();
            }
            Navigator.pop(context);
          }),
        ),
      ),
    );
  }
}
