import 'dart:convert';
import 'dart:io';

import 'package:copy_with_extension/copy_with_extension.dart';
import 'package:crypto/crypto.dart';
import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:hive_ce_flutter/adapters.dart';
import 'package:json_annotation/json_annotation.dart';
import 'package:miniature_painting_companion/injector.dart';

import 'layers_model.dart';

part 'hive_models.g.dart';

// After update launch: dart run build_runner build --delete-conflicting-outputs

class HiveRepository<T extends HiveObjectMixin> {
  final String boxName;
  late final Box<T> _box;

  HiveRepository(this.boxName);

  init() async {
    _box = await Hive.openBox<T>(boxName);
  }

  HiveList<T> createList({List<T>? objects}) {
    return HiveList(_box, objects: objects);
  }

  ValueListenable<Box<T>> listenable() {
    return _box.listenable();
  }

  Future<int> add(T value) async {
    return await _box.add(value);
  }

  Iterable<T> values() {
    return _box.values;
  }

  T? get(dynamic key, {T? defaultValue}) {
    return _box.get(key, defaultValue: defaultValue);
  }

  Future<void> save(T? value) async {
    return await value?.save();
  }

  Future<void> delete(T? value) async {
    return await value?.delete();
  }

  dynamic getKey(T value) {
    return value.key;
  }

  Future<void> clear() async {
    // don't want to do this accidentally
    if (kDebugMode) {
      await _box.clear();
    }
  }

  Future<void> put(dynamic key, T value) async {
    _box.put(key, value);
  }

  bool containsKey(dynamic key) {
    return _box.containsKey(key);
  }
}

abstract class Cloneable<T> {
  T clone();
}

@CopyWith()
class Miniature extends HiveObject implements Cloneable {
  String? name;
  String? imagePath;
  HiveList<PaintJob> paintJobs;
  HiveList<HistoryImage> historyImages;
  DateTime importTime;
  bool? favorite;

  Miniature(
      {this.imagePath,
      required this.name,
      required this.paintJobs,
      required this.historyImages,
      required this.importTime,
      this.favorite});

  Future<void> addPaintJob(PaintJob paintJob) async {
    await paintJobRepo.add(paintJob);
    paintJobs.add(paintJob);
    await miniatureRepo.save(this);
  }

  Future<void> addHistoryImage(HistoryImage historyImage) async {
    await historyImagesRepo.add(historyImage);
    historyImages.add(historyImage);
    await miniatureRepo.save(this);
  }

  Map<String, dynamic> toJson() {
    return {
      'name': name,
      'imagePath': imagePath,
      'imageBase64': fileStorageService.filePathToBase64(imagePath),
      'paintJobs': paintJobs.map((job) => job.toJson()).toList(),
      'historyImages': historyImages.map((img) => img.toJson()).toList(),
      'importTime': importTime.toIso8601String(),
      'favorite': favorite?.toString(),
    };
  }

  static Future<Miniature> fromJson(Map<String, dynamic> json) async {
    final miniature = Miniature(
        name: json['name'],
        imagePath: fileStorageService.base64toFile(
            json['imagePath'], json['imageBase64']),
        paintJobs: paintJobRepo.createList(),
        historyImages: historyImagesRepo.createList(),
        favorite: json['favorite'] as bool?,
        importTime: DateTime.parse(json['importTime']));

    await miniatureRepo.add(miniature);

    for (var jobJson in json['paintJobs'] ?? []) {
      final paintJob = await PaintJob.fromJson(jobJson);
      miniature.paintJobs.add(paintJob);
    }

    for (var imgJson in json['historyImages'] ?? []) {
      final historyImage = await HistoryImage.fromJson(imgJson);
      miniature.historyImages.add(historyImage);
    }

    await miniatureRepo.save(miniature);
    return miniature;
  }

  @override
  Future<Miniature> clone() async {
    return await Miniature.fromJson(toJson());
  }

  @override
  Future<void> delete() async {
    await super.delete();
    if (imagePath != null) {
      File(imagePath!).deleteSync();
    }
  }
}

@CopyWith()
class PaintJob extends HiveObject implements Cloneable {
  String? imagePath;
  String? name;
  HiveList<IDrawableLayer> paintLayers;
  HiveList<PositionedLayer> positioned;
  DateTime importTime;

  PaintJob(
      {this.imagePath,
      required this.name,
      required this.paintLayers,
      required this.positioned,
      required this.importTime});

  @override
  Future<PaintJob> clone() async {
    return await PaintJob.fromJson(toJson());
  }

  Future<void> addPaintLayer(IDrawableLayer layer) async {
    await iDrawableRepo.add(layer);
    paintLayers.add(layer);
    await paintJobRepo.save(this);
  }

  Future<void> addPositioned(PositionedLayer layer) async {
    await positionedRepo.add(layer);
    positioned.add(layer);
    await paintJobRepo.save(this);
  }

  Map<String, dynamic> toJson() {
    return {
      'imagePath': imagePath,
      'imageBase64': fileStorageService.filePathToBase64(imagePath),
      'name': name,
      'paintLayers': iDrawableLayerListToJson(paintLayers),
      'positioned': positioned.map((layer) => layer.toJson()).toList(),
      'importTime': importTime.toIso8601String(),
    };
  }

  static Future<PaintJob> fromJson(Map<String, dynamic> json) async {
    final paintJob = PaintJob(
        imagePath: fileStorageService.base64toFile(
            json['imagePath'], json['imageBase64']),
        name: json['name'],
        paintLayers: iDrawableRepo.createList(),
        positioned: positionedRepo.createList(),
        importTime: DateTime.parse(json['importTime']));

    await paintJobRepo.add(paintJob);

    for (var layerJson in json['paintLayers'] ?? []) {
      IDrawableLayer layer =
          await IDrawableLayer.genericLayerFromJson(layerJson);
      paintJob.paintLayers.add(layer);
    }

    for (var posJson in json['positioned'] ?? []) {
      PositionedLayer layer =
          await PositionedLayer.fromJson(Map<String, dynamic>.from(posJson));
      paintJob.positioned.add(layer);
    }

    await paintJobRepo.save(paintJob);
    return paintJob;
  }

  @override
  Future<void> delete() async {
    await super.delete();
    if (imagePath != null) {
      File(imagePath!).deleteSync();
    }
  }
}

enum ApplicationType { normal, dryBrush, lacy }

sealed class BasePaint extends HiveObject {
  String id;
  String type;
  String name;
  PaintManufacturer manufacturer;

  BasePaint(
      {required this.id,
      required this.type,
      required this.name,
      required this.manufacturer});
}

enum PaintManufacturer {
  citadel,
  tamiya,
  custom,
}

@JsonSerializable()
class Paint extends BasePaint {
  String? image;
  double? h;
  double? s;
  double? v;

  Paint({
    required super.type,
    required super.name,
    super.manufacturer = PaintManufacturer.citadel,
    String? id,
    this.image,
    this.h,
    this.s,
    this.v,
  }) : super(id: id ?? md5.convert(utf8.encode(name)).toString());

  factory Paint.fromJson(Map<String, dynamic> json) => _$PaintFromJson(json);

  Map<String, dynamic> toJson() => _$PaintToJson(this);

  HSVColor? getHsvColor() {
    if (h != null) {
      try {
        return HSVColor.fromAHSV(1.0, h!, s!, v!);
      } catch (e) {
        return HSVColor.fromColor(Colors.black);
      }
    }
    return null;
  }
}

class UserPaintComponent {
  Paint paint;
  double concentration;

  UserPaintComponent({required this.paint, required this.concentration});

  Map<String, dynamic> toJson() => {
        'concentration': concentration,
        'name': paint.name,
        'id': paint.id,
      };

  factory UserPaintComponent.fromJson(Map<String, dynamic> json) {
    return UserPaintComponent(
      concentration: json['concentration'] as double,
      paint: paintRepo.values().whereType<Paint>().firstWhere(
          (act) => act.id == (json['id'] as String),
          orElse: () => Paint(
              type: "LostType",
              name: json['name'],
              manufacturer: PaintManufacturer.custom)),
    );
  }
}

class UserPaint extends BasePaint {
  /// Between 0 and 100, represents the concentration of paint if it's used as a component (unused otherwise)
  List<UserPaintComponent> components;

  UserPaint(
      {required super.id,
      required super.type,
      required super.name,
      required this.components})
      : super(manufacturer: PaintManufacturer.custom);

  Future<void> addComponent(UserPaintComponent component) async {
    components.add(component);
    await paintRepo.save(this);
  }

  Future<void> setComponent(UserPaintComponent component, int index) async {
    components[index] = (component);
    await paintRepo.save(this);
  }

  static Future<UserPaint> fromJson(Map<String, dynamic> json) async {
    var created = UserPaint(
      id: json['id'] as String,
      type: json['type'] as String,
      name: json['name'] as String,
      components: (json['components'] as List<dynamic>? ?? [])
          .map(
              (act) => UserPaintComponent.fromJson(act as Map<String, dynamic>))
          .toList(),
    );

    await paintRepo.add(created);
    return created;
  }

  UserPaint copyWith({
    String? id,
    String? type,
    String? name,
    double? concentration,
    List<UserPaintComponent>? components,
  }) {
    return UserPaint(
      id: id ?? this.id,
      type: type ?? this.type,
      name: name ?? this.name,
      components: components ?? List<UserPaintComponent>.from(this.components),
    );
  }

  Map<String, dynamic> toJson() => {
        'id': id,
        'type': type,
        'name': name,
        'components':
            components.map((components) => components.toJson()).toList(),
      };
}

@CopyWith()
class HistoryImage extends HiveObject implements Cloneable {
  String imagePath;
  DateTime importTime;

  HistoryImage(this.imagePath, this.importTime);

  @override
  Future<HistoryImage> clone() async {
    return await HistoryImage.fromJson(toJson());
  }

  Map<String, dynamic> toJson() {
    return {
      'imagePath': imagePath,
      'imageBase64': fileStorageService.filePathToBase64(imagePath),
      'importTime': importTime.toIso8601String(),
    };
  }

  static Future<HistoryImage> fromJson(Map<String, dynamic> json) async {
    final historyImage = HistoryImage(
      fileStorageService.base64toFile(json['imagePath'], json['imageBase64']),
      DateTime.parse(json['importTime']),
    );

    await historyImagesRepo.add(historyImage);
    return historyImage;
  }

  @override
  Future<void> delete() async {
    await super.delete();
    File(imagePath).deleteSync();
  }
}

@CopyWith()
class PositionedLayer extends HiveObject implements Cloneable {
  double dx;
  double dy;
  HiveList<IDrawableLayer> layers;

  PositionedLayer({required this.dx, required this.dy, required this.layers});

  Future<void> addLayer(IDrawableLayer layer) async {
    await iDrawableRepo.add(layer);
    layers.add(layer);
    await positionedRepo.save(this);
  }

  @override
  Future<PositionedLayer> clone() async {
    return await PositionedLayer.fromJson(toJson());
  }

  Map<String, dynamic> toJson() {
    return {
      'dx': dx,
      'dy': dy,
      'layers': iDrawableLayerListToJson(layers),
    };
  }

  static Future<PositionedLayer> fromJson(Map<String, dynamic> json) async {
    final positioned = PositionedLayer(
      dx: json['dx'],
      dy: json['dy'],
      layers: iDrawableRepo.createList(),
    );
    positionedRepo.add(positioned);

    for (var layerJson in json['layers'] ?? []) {
      IDrawableLayer layer =
          await IDrawableLayer.genericLayerFromJson(layerJson);
      positioned.layers.add(layer);
    }

    await positionedRepo.save(positioned);
    return positioned;
  }
}

List<Map<String, Object>> iDrawableLayerListToJson(
    HiveList<IDrawableLayer> layers) {
  return layers
      .map((layer) => {
            'type': layer.runtimeType.toString(),
            'data': layer.toJson(),
          })
      .toList();
}
