// SPDX-License-Identifier: GPL-3.0-or-later
// Copyright (C) 2019-2025 The Sanmill developers (see AUTHORS file)

// automated_move_single_test.dart

// ignore_for_file: avoid_print, always_specify_types

import 'package:flutter_test/flutter_test.dart';
import 'package:integration_test/integration_test.dart';
import 'package:sanmill/main.dart' as app;

import 'automated_move_test_data.dart';
import 'automated_move_test_runner.dart';
import 'backup_service.dart';

/// Integration test for running a SINGLE test case for debugging
///
/// Usage:
///   flutter test integration_test/automated_move_single_test.dart -d linux
void main() {
  IntegrationTestWidgetsFlutterBinding.ensureInitialized();

  group('Single Test - First Failed Case', () {
    testWidgets('Run first failed test case', (WidgetTester tester) async {
      // Launch the app
      print('[IntegrationTest] Launching Sanmill app...');
      app.main();
      await tester.pumpAndSettle();

      // Wait for app initialization
      await Future<void>.delayed(const Duration(seconds: 2));

      // Backup the database AFTER app initialization
      final Map<String, dynamic> dbBackup = await backupDatabase();
      addTearDown(() async => restoreDatabase(dbBackup));

      print('[IntegrationTest] App initialized, starting single test...');

      // Execute ONLY the first failed test case from integration test
      final singleTestConfig = AutomatedMoveTestData.createCustomConfig(
        configName: 'Single Test - First Failed Case',
        batchDescription: 'Run only first failed test case for debugging',
        testCases: [AutomatedMoveTestData.advancedMultipleCaptures],
        stopOnFirstFailure: true,
      );

      final result = await AutomatedMoveTestRunner.runTestBatch(
        singleTestConfig,
      );

      // Print summary
      print('[IntegrationTest] =====================================');
      print('[IntegrationTest] Single Test Completed');
      print(
        '[IntegrationTest] Test ID: ${result.testResults.first.testCase.id}',
      );
      print('[IntegrationTest] Passed: ${result.passedCount}');
      print('[IntegrationTest] Failed: ${result.failedCount}');
      print('[IntegrationTest] =====================================');
    });
  });
}
