package com.byagowi.persiancalendar.ui.astronomy

import android.content.res.Resources
import com.byagowi.persiancalendar.R
import com.byagowi.persiancalendar.entities.Jdn
import com.byagowi.persiancalendar.entities.Language
import com.byagowi.persiancalendar.global.coordinates
import com.byagowi.persiancalendar.global.spacedColon
import com.byagowi.persiancalendar.utils.formatDateAndTime
import com.byagowi.persiancalendar.utils.generateYearName
import com.byagowi.persiancalendar.utils.sunlitSideMoonTiltAngle
import com.byagowi.persiancalendar.utils.toGregorianCalendar
import com.byagowi.persiancalendar.utils.toObserver
import io.github.cosinekitty.astronomy.Aberration
import io.github.cosinekitty.astronomy.Body
import io.github.cosinekitty.astronomy.EquatorEpoch
import io.github.cosinekitty.astronomy.Refraction
import io.github.cosinekitty.astronomy.Time
import io.github.cosinekitty.astronomy.eclipticGeoMoon
import io.github.cosinekitty.astronomy.equator
import io.github.cosinekitty.astronomy.equatorialToEcliptic
import io.github.cosinekitty.astronomy.geoVector
import io.github.cosinekitty.astronomy.helioVector
import io.github.cosinekitty.astronomy.horizon
import io.github.cosinekitty.astronomy.searchGlobalSolarEclipse
import io.github.cosinekitty.astronomy.searchLocalSolarEclipse
import io.github.cosinekitty.astronomy.searchLunarEclipse
import io.github.cosinekitty.astronomy.sunPosition
import java.util.Date
import java.util.GregorianCalendar

class AstronomyState(val date: GregorianCalendar) {
    private val time = Time.fromMillisecondsSince1970(date.time.time)
    val sun = sunPosition(time)
    val moon = eclipticGeoMoon(time)
    private val observer by lazy(LazyThreadSafetyMode.NONE) { coordinates?.toObserver() }
    val moonTilt by lazy(LazyThreadSafetyMode.NONE) {
        observer?.let { observer -> sunlitSideMoonTiltAngle(time, observer).toFloat() }
    }
    val sunAltitude by lazy(LazyThreadSafetyMode.NONE) {
        val observer = observer ?: return@lazy null
        val sunEquator =
            equator(Body.Sun, time, observer, EquatorEpoch.OfDate, Aberration.Corrected)
        horizon(time, observer, sunEquator.ra, sunEquator.dec, Refraction.Normal).altitude
    }
    val moonAltitude by lazy(LazyThreadSafetyMode.NONE) {
        val observer = observer ?: return@lazy null
        val moonEquator =
            equator(Body.Moon, time, observer, EquatorEpoch.OfDate, Aberration.Corrected)
        horizon(time, observer, moonEquator.ra, moonEquator.dec, Refraction.Normal).altitude
    }
    val heliocentricPlanets by lazy(LazyThreadSafetyMode.NONE) {
        heliocentricPlanetsList.map { equatorialToEcliptic(helioVector(it, time)) }
    }
    val geocentricPlanets by lazy(LazyThreadSafetyMode.NONE) {
        geocentricPlanetsList.map {
            equatorialToEcliptic(geoVector(it, time, Aberration.Corrected))
        }
    }

    fun generateHeader(resources: Resources, language: Language, jdn: Jdn): List<String> {
        val observer = coordinates?.toObserver()
        return listOf(
            if (observer != null) searchLocalSolarEclipse(time, observer).run {
                Triple(R.string.solar_eclipse, kind, peak.time)
            } else searchGlobalSolarEclipse(time).run {
                Triple(R.string.solar_eclipse, kind, peak)
            },
            searchLunarEclipse(time).run {
                Triple(R.string.lunar_eclipse, kind, peak)
            },
        ).sortedBy { (_, _, peak) -> peak }.map { (title, kind, peak) ->
            val formattedDate =
                Date(peak.toMillisecondsSince1970()).toGregorianCalendar().formatDateAndTime()
            val isSolar = title == R.string.solar_eclipse
            val type = language.tryTranslateEclipseType(isSolar, kind) ?: resources.getString(title)
            when {
                language.isPersianOrDari -> "$type بعدی"
                else -> resources.getString(R.string.next_x, type)
            } + spacedColon + formattedDate
        } + generateYearName(
            resources = resources,
            jdn = jdn,
            withOldEraName = language.isPersianOrDari,
            withEmoji = true,
            time = date,
        )
    }

    companion object {
        val heliocentricPlanetsList = listOf(
            Body.Mercury,
            Body.Venus,
            Body.Earth,
            Body.Mars,
            Body.Jupiter,
            Body.Saturn,
            Body.Uranus,
            Body.Neptune,
        )
    }
}
