package com.byagowi.persiancalendar.ui.astronomy

import android.content.res.Configuration
import androidx.compose.animation.AnimatedContent
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.SharedTransitionScope
import androidx.compose.animation.animateContentSize
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.combinedClickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.BoxWithConstraints
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.WindowInsetsSides
import androidx.compose.foundation.layout.aspectRatio
import androidx.compose.foundation.layout.displayCutout
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.only
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.windowInsetsPadding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.text.TextAutoSize
import androidx.compose.foundation.text.selection.SelectionContainer
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.KeyboardArrowLeft
import androidx.compose.material.icons.automirrored.filled.KeyboardArrowRight
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.LocalTextStyle
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.NavigationRailItem
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.material3.ripple
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.Stable
import androidx.compose.runtime.derivedStateOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableLongStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.drawBehind
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.drawscope.drawIntoCanvas
import androidx.compose.ui.graphics.nativeCanvas
import androidx.compose.ui.graphics.toArgb
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.compose.ui.layout.Layout
import androidx.compose.ui.platform.LocalConfiguration
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.platform.LocalHapticFeedback
import androidx.compose.ui.platform.LocalLayoutDirection
import androidx.compose.ui.platform.LocalResources
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.res.vectorResource
import androidx.compose.ui.semantics.clearAndSetSemantics
import androidx.compose.ui.semantics.contentDescription
import androidx.compose.ui.semantics.semantics
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.LayoutDirection
import androidx.compose.ui.unit.coerceAtMost
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.compose.ui.util.fastForEach
import androidx.compose.ui.viewinterop.AndroidView
import androidx.core.util.lruCache
import androidx.navigation3.ui.LocalNavAnimatedContentScope
import com.byagowi.persiancalendar.R
import com.byagowi.persiancalendar.SHARED_CONTENT_KEY_MAP
import com.byagowi.persiancalendar.SHARED_CONTENT_KEY_MOON
import com.byagowi.persiancalendar.SHARED_CONTENT_KEY_TIME_BAR
import com.byagowi.persiancalendar.entities.Jdn
import com.byagowi.persiancalendar.entities.Season
import com.byagowi.persiancalendar.global.coordinates
import com.byagowi.persiancalendar.global.language
import com.byagowi.persiancalendar.global.spacedColon
import com.byagowi.persiancalendar.global.spacedComma
import com.byagowi.persiancalendar.ui.common.AppDropdownMenuItem
import com.byagowi.persiancalendar.ui.common.DatePickerDialog
import com.byagowi.persiancalendar.ui.common.NavigationNavigateUpIcon
import com.byagowi.persiancalendar.ui.common.NavigationOpenNavigationRailIcon
import com.byagowi.persiancalendar.ui.common.ScreenSurface
import com.byagowi.persiancalendar.ui.common.SolarDraw
import com.byagowi.persiancalendar.ui.common.SwitchWithLabel
import com.byagowi.persiancalendar.ui.common.ThreeDotsDropdownMenu
import com.byagowi.persiancalendar.ui.common.TodayActionButton
import com.byagowi.persiancalendar.ui.theme.appCrossfadeSpec
import com.byagowi.persiancalendar.ui.theme.appTopAppBarColors
import com.byagowi.persiancalendar.ui.theme.isDynamicGrayscale
import com.byagowi.persiancalendar.ui.theme.resolveAndroidCustomTypeface
import com.byagowi.persiancalendar.ui.utils.appBoundsTransform
import com.byagowi.persiancalendar.ui.utils.appContentSizeAnimationSpec
import com.byagowi.persiancalendar.ui.utils.performHapticFeedbackVirtualKey
import com.byagowi.persiancalendar.ui.utils.performLongPress
import com.byagowi.persiancalendar.utils.formatDateAndTime
import com.byagowi.persiancalendar.utils.isSouthernHemisphere
import com.byagowi.persiancalendar.utils.toCivilDate
import com.byagowi.persiancalendar.utils.toGregorianCalendar
import io.github.cosinekitty.astronomy.seasons
import io.github.persiancalendar.calendar.CivilDate
import io.github.persiancalendar.calendar.PersianDate
import kotlinx.coroutines.delay
import java.util.Date
import kotlin.math.abs
import kotlin.math.max
import kotlin.time.Duration.Companion.seconds

@Composable
fun SharedTransitionScope.AstronomyScreen(
    openNavigationRail: () -> Unit,
    navigateToMap: () -> Unit,
    viewModel: AstronomyViewModel,
    noBackStackAction: (() -> Unit)?,
) {
    LaunchedEffect(Unit) {
        // Default animation screen enter, only if minutes offset is at it's default
        if (viewModel.minutesOffset == AstronomyViewModel.DEFAULT_TIME) {
            viewModel.animateToAbsoluteMinutesOffset(0)
        }

        while (true) {
            delay(10.seconds)
            // Ugly, just to make the offset
            viewModel.addMinutesOffset(1)
            viewModel.addMinutesOffset(-1)
        }
    }

    // Port SliderView to Compose maybe sometime
    var slider by remember { mutableStateOf<SliderView?>(null) }

    val isLandscape = LocalConfiguration.current.orientation == Configuration.ORIENTATION_LANDSCAPE

    Scaffold(
        containerColor = Color.Transparent,
        topBar = {
            @OptIn(ExperimentalMaterial3Api::class) TopAppBar(
                title = {
                    Text(
                        stringResource(R.string.astronomy),
                        maxLines = 1,
                        autoSize = TextAutoSize.StepBased(
                            maxFontSize = LocalTextStyle.current.fontSize,
                            minFontSize = 10.sp,
                        ),
                    )
                },
                colors = appTopAppBarColors(),
                navigationIcon = {
                    if (noBackStackAction != null) NavigationNavigateUpIcon(noBackStackAction)
                    else NavigationOpenNavigationRailIcon(openNavigationRail)
                },
                actions = {
                    TodayActionButton(visible = viewModel.minutesOffset != 0) {
                        viewModel.animateToAbsoluteMinutesOffset(0)
                    }
                    this.AnimatedVisibility(visible = viewModel.mode == AstronomyMode.EARTH) {
                        SwitchWithLabel(
                            label = stringResource(R.string.tropical),
                            checked = viewModel.isTropical,
                            labelBeforeSwitch = true,
                            onValueChange = { viewModel.isTropical = it },
                        )
                    }

                    var showHoroscopeDialog by rememberSaveable { mutableStateOf(false) }
                    if (showHoroscopeDialog) HoroscopeDialog(viewModel.astronomyState.date.time) {
                        showHoroscopeDialog = false
                    }
                    var showYearHoroscopeDialog by rememberSaveable { mutableStateOf(false) }
                    if (showYearHoroscopeDialog) {
                        YearHoroscopeDialog(PersianDate(viewModel.astronomyState.date.toCivilDate()).year) {
                            showYearHoroscopeDialog = false
                        }
                    }

                    var showPlanetaryHoursDialog by rememberSaveable { mutableStateOf(false) }
                    if (showPlanetaryHoursDialog) coordinates?.also {
                        PlanetaryHoursDialog(it, viewModel.astronomyState.date.timeInMillis) {
                            showPlanetaryHoursDialog = false
                        }
                    }

                    var showMoonInScorpioDialog by rememberSaveable { mutableStateOf(false) }
                    if (showMoonInScorpioDialog) MoonInScorpioDialog(viewModel.astronomyState.date) {
                        showMoonInScorpioDialog = false
                    }

                    ThreeDotsDropdownMenu { closeMenu ->
                        AppDropdownMenuItem({ Text(stringResource(R.string.select_date)) }) {
                            closeMenu()
                            viewModel.isDatePickerDialogShown = true
                        }
                        AppDropdownMenuItem({ Text(stringResource(R.string.map)) }) {
                            closeMenu()
                            navigateToMap()
                        }
                        AppDropdownMenuItem({ Text(stringResource(R.string.horoscope)) }) {
                            showHoroscopeDialog = true
                            closeMenu()
                        }
                        AppDropdownMenuItem(
                            {
                                Text(
                                    stringResource(R.string.horoscope) + spacedComma + stringResource(
                                        R.string.year,
                                    ),
                                )
                            },
                        ) {
                            showYearHoroscopeDialog = true
                            closeMenu()
                        }
                        if (coordinates != null) AppDropdownMenuItem(
                            {
                                Text(stringResource(R.string.planetary_hours))
                            },
                        ) {
                            showPlanetaryHoursDialog = true
                            closeMenu()
                        }
                        AppDropdownMenuItem(
                            {
                                Text(stringResource(R.string.moon_in_scorpio))
                            },
                        ) {
                            showMoonInScorpioDialog = true
                            closeMenu()
                        }
                    }
                },
            )
        },
    ) { paddingValues ->
        Box(Modifier.padding(top = paddingValues.calculateTopPadding())) {
            ScreenSurface {
                val bottomPadding = paddingValues.calculateBottomPadding()
                if (isLandscape) BoxWithConstraints(
                    Modifier
                        .fillMaxSize()
                        .windowInsetsPadding(WindowInsets.displayCutout.only(WindowInsetsSides.Horizontal)),
                ) {
                    val maxHeight = this.maxHeight
                    val maxWidth = this.maxWidth
                    Row(Modifier.fillMaxWidth()) {
                        Column(
                            Modifier
                                .width((maxWidth / 2).coerceAtMost(480.dp))
                                .fillMaxHeight()
                                .padding(
                                    top = 24.dp,
                                    start = 24.dp,
                                    bottom = bottomPadding + 16.dp,
                                ),
                        ) {
                            Header(Modifier, viewModel)
                            Spacer(Modifier.weight(1f))
                            SliderBar(slider, viewModel) { slider = it }
                        }
                        SolarDisplay(
                            Modifier
                                .weight(1f)
                                .padding(top = 16.dp, bottom = bottomPadding + 16.dp)
                                .height(maxHeight - bottomPadding),
                            viewModel, slider, navigateToMap,
                        )
                    }
                } else Column {
                    BoxWithConstraints(Modifier.weight(1f, fill = false)) {
                        val maxHeight = this.maxHeight
                        val maxWidth = this.maxWidth
                        var needsScroll by remember { mutableStateOf(false) }
                        // Puts content in middle of available space after the measured header
                        Layout(
                            modifier = if (needsScroll) Modifier.verticalScroll(rememberScrollState()) else Modifier,
                            content = {
                                // Header
                                Header(
                                    Modifier.padding(start = 24.dp, end = 24.dp, top = 24.dp),
                                    viewModel,
                                )
                                // Content
                                SolarDisplay(
                                    Modifier
                                        .fillMaxWidth()
                                        .height(maxWidth - (56 * 2 + 8).dp),
                                    viewModel, slider, navigateToMap,
                                )
                            },
                        ) { (header, content), constraints ->
                            val placeableHeader = header.measure(constraints)
                            val placeableContent = content.measure(constraints)
                            layout(
                                width = constraints.maxWidth,
                                height = max(
                                    placeableHeader.height + placeableContent.height,
                                    maxHeight.roundToPx(),
                                ),
                            ) {
                                // Put the header at top
                                placeableHeader.placeRelative(0, 0)

                                val availableHeight = maxHeight.roundToPx() - placeableHeader.height
                                val space = availableHeight / 2 - placeableContent.height / 2
                                needsScroll = space <= 0
                                placeableContent.placeRelative(
                                    0, placeableHeader.height + space.coerceAtLeast(0),
                                )
                            }
                        }
                    }
                    SliderBar(slider, viewModel) { slider = it }
                    Spacer(Modifier.height(16.dp + bottomPadding))
                }
            }
        }
    }

    if (viewModel.isDatePickerDialogShown) DatePickerDialog(
        initialJdn = Jdn(viewModel.astronomyState.date.toCivilDate()),
        onDismissRequest = { viewModel.isDatePickerDialogShown = false },
    ) { jdn -> viewModel.animateToAbsoluteDayOffset(jdn - Jdn.today()) }
}

@Composable
private fun SharedTransitionScope.SliderBar(
    slider: SliderView?,
    viewModel: AstronomyViewModel,
    setSlider: (SliderView) -> Unit,
) {
    var lastButtonClickTimestamp by remember { mutableLongStateOf(System.currentTimeMillis()) }
    val isRtl = LocalLayoutDirection.current == LayoutDirection.Rtl
    fun buttonScrollSlider(days: Int) {
        lastButtonClickTimestamp = System.currentTimeMillis()
        slider?.smoothScrollBy(250f * days * if (isRtl) 1 else -1, 0f)
        viewModel.animateToRelativeDayOffset(days)
    }

    Column(Modifier.fillMaxWidth()) {
        Text(
            viewModel.astronomyState.date.formatDateAndTime(),
            textAlign = TextAlign.Center,
            modifier = Modifier
                .fillMaxWidth()
                .combinedClickable(
                    onClick = { viewModel.isDatePickerDialogShown = true },
                    onClickLabel = stringResource(R.string.select_date),
                    onLongClick = { viewModel.animateToAbsoluteMinutesOffset(0) },
                    onLongClickLabel = stringResource(R.string.today),
                )
                .sharedElement(
                    rememberSharedContentState(key = SHARED_CONTENT_KEY_TIME_BAR),
                    animatedVisibilityScope = LocalNavAnimatedContentScope.current,
                    boundsTransform = appBoundsTransform,
                ),
            color = MaterialTheme.colorScheme.onSurface,
        )
        Spacer(Modifier.height(8.dp))
        Row(
            verticalAlignment = Alignment.CenterVertically,
            modifier = Modifier.padding(horizontal = 16.dp),
        ) {
            TimeArrow(::buttonScrollSlider, isPrevious = true)
            val primary = MaterialTheme.colorScheme.primary
            AndroidView(
                factory = { context ->
                    val root = SliderView(context)
                    root.setBarsColor(primary.toArgb())
                    setSlider(root)
                    var latestVibration = 0L
                    root.smoothScrollBy(250f * if (isRtl) 1 else -1, 0f)
                    root.onScrollListener = { dx, _ ->
                        if (dx != 0f) {
                            val current = System.currentTimeMillis()
                            if (current - lastButtonClickTimestamp > 2000) {
                                if (current >= latestVibration + 25_000_000 / abs(dx)) {
                                    root.performHapticFeedbackVirtualKey()
                                    latestVibration = current
                                }
                                viewModel.addMinutesOffset(
                                    (dx * if (isRtl) 1 else -1).toInt(),
                                )
                            }
                        }
                    }
                    root
                },
                modifier = Modifier
                    .padding(horizontal = 16.dp)
                    .height(46.dp)
                    .weight(1f, fill = false),
            )
            TimeArrow(::buttonScrollSlider, isPrevious = false)
        }
    }
}

@Composable
private fun TimeArrow(buttonScrollSlider: (Int) -> Unit, isPrevious: Boolean) {
    val hapticFeedback = LocalHapticFeedback.current
    Icon(
        if (isPrevious) Icons.AutoMirrored.Default.KeyboardArrowLeft
        else Icons.AutoMirrored.Default.KeyboardArrowRight,
        contentDescription = stringResource(
            if (isPrevious) R.string.previous_x else R.string.next_x,
            stringResource(R.string.day),
        ),
        Modifier.combinedClickable(
            indication = ripple(bounded = false),
            interactionSource = null,
            onClick = {
                hapticFeedback.performLongPress()
                buttonScrollSlider(if (isPrevious) -1 else 1)
            },
            onClickLabel = stringResource(R.string.select_day),
            onLongClick = { buttonScrollSlider(if (isPrevious) -365 else 365) },
            onLongClickLabel = stringResource(
                if (isPrevious) R.string.previous_x else R.string.next_x,
                stringResource(R.string.year),
            ),
        ),
        tint = MaterialTheme.colorScheme.primary,
    )
}

@Composable
private fun SharedTransitionScope.SolarDisplay(
    modifier: Modifier,
    viewModel: AstronomyViewModel,
    slider: SliderView?,
    navigateToMap: () -> Unit,
) {
    Box(modifier) {
        Column(Modifier.align(Alignment.CenterStart)) {
            AstronomyMode.entries.forEach {
                NavigationRailItem(
                    modifier = Modifier.size(56.dp),
                    selected = viewModel.mode == it,
                    onClick = { viewModel.mode = it },
                    icon = {
                        if (it == AstronomyMode.MOON) MoonIcon(viewModel.astronomyState) else Icon(
                            ImageVector.vectorResource(it.icon),
                            modifier = Modifier.size(24.dp),
                            contentDescription = null,
                            tint = Color.Unspecified,
                        )
                    },
                )
            }
        }
        val map = stringResource(R.string.map)
        NavigationRailItem(
            modifier = Modifier
                .size(56.dp)
                .align(Alignment.CenterEnd)
                .sharedBounds(
                    rememberSharedContentState(key = SHARED_CONTENT_KEY_MAP),
                    animatedVisibilityScope = LocalNavAnimatedContentScope.current,
                    boundsTransform = appBoundsTransform,
                ),
            selected = false,
            onClick = navigateToMap,
            icon = { Text("🗺", modifier = Modifier.semantics { this.contentDescription = map }) },
        )
        val surfaceColor = MaterialTheme.colorScheme.surface
        val contentColor = LocalContentColor.current
        val typeface = resolveAndroidCustomTypeface()
        AndroidView(
            factory = {
                val solarView = SolarView(it)
                solarView.rotationalMinutesChange = { offset ->
                    viewModel.addMinutesOffset(offset)
                    slider?.manualScrollBy(offset / 200f, 0f)
                }
                solarView
            },
            modifier = Modifier
                .padding(horizontal = 56.dp)
                .aspectRatio(1f)
                .align(Alignment.Center),
            update = {
                it.setSurfaceColor(surfaceColor.toArgb())
                it.setContentColor(contentColor.toArgb())
                it.isTropicalDegree = viewModel.isTropical
                it.setTime(viewModel.astronomyState)
                it.setFont(typeface)
                it.mode = viewModel.mode
            },
        )
    }
}

@Composable
private fun Header(modifier: Modifier, viewModel: AstronomyViewModel) {
    val sunZodiac = if (viewModel.isTropical) Zodiac.fromTropical(viewModel.astronomyState.sun.elon)
    else Zodiac.fromIau(viewModel.astronomyState.sun.elon)
    val moonZodiac =
        if (viewModel.isTropical) Zodiac.fromTropical(viewModel.astronomyState.moon.lon)
        else Zodiac.fromIau(viewModel.astronomyState.moon.lon)

    val resources = LocalResources.current
    val headerCache = remember(resources, language) {
        lruCache(
            1024,
            create = { jdn: Jdn ->
                viewModel.astronomyState.generateHeader(resources, language, jdn)
            },
        )
    }

    Column(modifier) {
        val jdn by remember { derivedStateOf { Jdn(viewModel.astronomyState.date.toCivilDate()) } }
        headerCache[jdn].fastForEach { line ->
            AnimatedContent(targetState = line, label = "line", transitionSpec = appCrossfadeSpec) {
                SelectionContainer {
                    Text(
                        text = it,
                        style = MaterialTheme.typography.bodyMedium,
                        maxLines = 1,
                        softWrap = false,
                        autoSize = TextAutoSize.StepBased(
                            minFontSize = 9.sp,
                            maxFontSize = MaterialTheme.typography.bodyMedium.fontSize,
                        ),
                    )
                }
            }
        }
        Seasons(jdn, viewModel)
        this.AnimatedVisibility(visible = viewModel.mode == AstronomyMode.EARTH) {
            Row(Modifier.padding(top = 8.dp)) {
                listOf(
                    // ☉☀️
                    Triple(sunZodiac, R.string.sun, Color(0xcceaaa00)),
                    // ☽it.moonPhaseEmoji
                    Triple(moonZodiac, R.string.moon, Color(0xcc606060)),
                ).forEach { (zodiac, titleId, color) ->
                    Box(Modifier.weight(1f)) {
                        val title = stringResource(titleId)
                        val value = stringResource(zodiac.titleId)
                        Cell(
                            Modifier
                                .semantics(mergeDescendants = true) {
                                    this.contentDescription = title + spacedColon + value
                                }
                                .clearAndSetSemantics {}
                                .align(Alignment.Center),
                            color = color,
                            title = title,
                            value = zodiac.symbol + " " + value,
                        )
                    }
                }
            }
        }
    }
}

@Composable
private fun Seasons(jdn: Jdn, viewModel: AstronomyViewModel) {
    val seasonsCache = remember { lruCache(1024, create = ::seasons) }
    val seasonsOrder = remember {
        if (coordinates?.isSouthernHemisphere == true) {
            listOf(Season.WINTER, Season.SPRING, Season.SUMMER, Season.AUTUMN)
        } else listOf(Season.SUMMER, Season.AUTUMN, Season.WINTER, Season.SPRING)
    }
    val equinoxes = (1..4).map { i ->
        Date(
            seasonsCache[
                CivilDate(
                    PersianDate(jdn.toPersianDate().year, i * 3, 29),
                ).year,
            ].let {
                when (i) {
                    1 -> it.juneSolstice
                    2 -> it.septemberEquinox
                    3 -> it.decemberSolstice
                    else -> it.marchEquinox
                }
            }.toMillisecondsSince1970(),
        ).let { it.time to it.toGregorianCalendar().formatDateAndTime() }
    }
    repeat(2) { row ->
        Row(Modifier.padding(top = 8.dp)) {
            repeat(2) { cell ->
                Box(Modifier.weight(1f)) {
                    val title = stringResource(seasonsOrder[cell + row * 2].nameStringId)
                    val (time, formattedTime) = equinoxes[cell + row * 2]
                    Cell(
                        Modifier
                            .clip(MaterialTheme.shapes.small)
                            .semantics(true) {
                                this.contentDescription = title + spacedComma + formattedTime
                            }
                            .clickable(onClickLabel = stringResource(R.string.select_date)) {
                                viewModel.animateToTime(time)
                            }
                            .clearAndSetSemantics {},
                        color = seasonsOrder[cell + row * 2].color,
                        title = title,
                        value = formattedTime,
                    )
                }
            }
        }
    }
}

@Stable
@Composable
private fun SharedTransitionScope.MoonIcon(astronomyState: AstronomyState) {
    val resources = LocalResources.current
    val solarDraw = remember(resources) { SolarDraw(resources) }
    Box(
        modifier = Modifier
            .size(24.dp)
            .sharedBounds(
                rememberSharedContentState(key = SHARED_CONTENT_KEY_MOON),
                animatedVisibilityScope = LocalNavAnimatedContentScope.current,
                boundsTransform = appBoundsTransform,
            )
            .drawBehind {
                drawIntoCanvas {
                    val radius = size.minDimension / 2f
                    val sun = astronomyState.sun
                    val moon = astronomyState.moon
                    solarDraw.moon(it.nativeCanvas, sun, moon, radius, radius, radius)
                }
            },
    )
}

@Stable
@Composable
private fun Cell(
    modifier: Modifier,
    color: Color,
    title: String,
    value: String,
) {
    Row(
        modifier
            .heightIn(max = 64.dp)
            .animateContentSize(appContentSizeAnimationSpec),
        verticalAlignment = Alignment.CenterVertically,
    ) {
        Text(
            title,
            modifier = Modifier
                .background(
                    if (isDynamicGrayscale()) Color(0xcc808080) else color,
                    MaterialTheme.shapes.small,
                )
                .align(alignment = Alignment.CenterVertically)
                .padding(vertical = 4.dp, horizontal = 8.dp),
            style = MaterialTheme.typography.bodyMedium,
            color = Color.White,
            fontSize = with(LocalDensity.current) {
                MaterialTheme.typography.bodyMedium.fontSize.toDp().coerceAtMost(18.dp).toSp()
            },
        )
        SelectionContainer {
            Text(
                value,
                style = LocalTextStyle.current.copy(
                    lineHeight = with(LocalDensity.current) {
                        LocalTextStyle.current.lineHeight.toDp().coerceAtMost(28.dp).toSp()
                    },
                ),
                modifier = Modifier.padding(start = 8.dp, end = 4.dp),
                maxLines = if (LocalDensity.current.fontScale > 1) 2 else 1,
                autoSize = TextAutoSize.StepBased(
                    minFontSize = 9.sp,
                    maxFontSize = MaterialTheme.typography.bodyMedium.fontSize,
                ),
            )
        }
    }
}
