package com.byagowi.persiancalendar.entities

import com.byagowi.persiancalendar.IRAN_TIMEZONE_ID
import com.byagowi.persiancalendar.utils.toCivilDate
import com.byagowi.persiancalendar.utils.toGregorianCalendar
import io.github.cosinekitty.astronomy.Time
import io.github.persiancalendar.calendar.AbstractDate
import io.github.persiancalendar.calendar.CivilDate
import io.github.persiancalendar.calendar.IslamicDate
import io.github.persiancalendar.calendar.NepaliDate
import io.github.persiancalendar.calendar.PersianDate
import kotlinx.serialization.Serializable
import java.util.Date
import java.util.GregorianCalendar
import java.util.TimeZone
import kotlin.math.ceil

// Julian day number, basically a day counter starting from some day in concept
// https://en.wikipedia.org/wiki/Julian_day
@Serializable
@JvmInline
value class Jdn(val value: Long) {
    constructor(value: AbstractDate) : this(value.toJdn())
    constructor(
        calendar: Calendar,
        year: Int,
        month: Int,
        day: Int,
    ) : this(calendar.createDate(year, month, day))

    val weekDay: WeekDay get() = WeekDay.entries[((value + 2L) % 7L).toInt()]

    infix fun on(calendar: Calendar): AbstractDate = when (calendar) {
        Calendar.ISLAMIC -> toIslamicDate()
        Calendar.GREGORIAN -> toCivilDate()
        Calendar.SHAMSI -> toPersianDate()
        Calendar.NEPALI -> toNepaliDate()
    }

    fun toIslamicDate() = IslamicDate(value)
    fun toCivilDate() = CivilDate(value)
    fun toPersianDate() = PersianDate(value)
    fun toNepaliDate() = NepaliDate(value)

    operator fun compareTo(other: Jdn) = value compareTo other.value
    operator fun plus(other: Int): Jdn = Jdn(value + other)
    operator fun minus(other: Int): Jdn = Jdn(value - other)

    // Difference of two Jdn values in days
    operator fun minus(other: Jdn): Int = (value - other.value).toInt()

    fun toGregorianCalendar(): GregorianCalendar = GregorianCalendar().also {
        val gregorian = this.toCivilDate()
        it.set(gregorian.year, gregorian.month - 1, gregorian.dayOfMonth)
    }

    fun toAstronomyTime(hourOfDay: Int, setIranTime: Boolean = false): Time {
        val date = toGregorianCalendar()
        if (setIranTime) date.timeZone = TimeZone.getTimeZone(IRAN_TIMEZONE_ID)
        date[GregorianCalendar.HOUR_OF_DAY] = hourOfDay
        date[GregorianCalendar.MINUTE] = 0
        date[GregorianCalendar.SECOND] = 0
        date[GregorianCalendar.MILLISECOND] = 0
        return Time.fromMillisecondsSince1970(date.timeInMillis)
    }

    fun getWeekOfYear(startOfYear: Jdn, weekStart: WeekDay): Int {
        val dayOfYear = this - startOfYear
        return ceil(1 + (dayOfYear - (this.weekDay - weekStart)) / 7.0).toInt()
    }

    // Days passed in a season and total days available in the season
    // The result is a (passedDaysInSeason, totalSeasonDays)
    fun getPositionInSeason(): Pair<Int, Int> {
        val persianDate = this.toPersianDate()
        val season = (persianDate.month - 1) / 3
        val seasonBeginning = PersianDate(persianDate.year, season * 3 + 1, 1)
        val seasonBeginningJdn = Jdn(seasonBeginning)
        return this - seasonBeginningJdn + 1 to Jdn(seasonBeginning.monthStartOfMonthsDistance(3)) - seasonBeginningJdn
    }

    operator fun rangeTo(that: Jdn): Sequence<Jdn> =
        (this.value..that.value).asSequence().map(::Jdn)

    operator fun rangeUntil(that: Jdn): Sequence<Jdn> =
        (this.value..<that.value).asSequence().map(::Jdn)

    companion object {
        fun today() = Jdn(Date().toGregorianCalendar().toCivilDate())
    }
}
