package connect

import (
	"bytes"
	"context"
	"crypto/tls"
	"fmt"
	mathrand "math/rand"
	"net"
	"net/http"
	"net/url"
	"slices"
	"strings"
	"sync"
	"sync/atomic"
	"time"
	// "runtime/debug"

	"golang.org/x/exp/maps"

	"github.com/gorilla/websocket"
	quic "github.com/quic-go/quic-go"

	"github.com/urnetwork/glog/v2026"

	"github.com/urnetwork/connect/v2026/protocol"
)

// note that it is possible to have multiple transports for the same client destination
// e.g. platform, p2p, and a bunch of extenders

// extenders are identified and credited with the platform by ip address
// they forward to a special port, 8443, that whitelists their ip without rate limiting
// when an extender gets an http message from a client, it always connects tcp to connect.bringyour.com:8443
// appends the proxy protocol headers, and then forwards the bytes from the client
// https://docs.nginx.com/nginx/admin-guide/load-balancer/using-proxy-protocol/
// rate limit using $proxy_protocol_addr https://www.nginx.com/blog/rate-limiting-nginx/
// add the source ip as the X-Extender header

// the transport attempts to upgrade from http1 to http3
// versus the h1 transport, h3 is:
// - more cpu efficient.
//   The quic stream does not need to mask/unmask each byte before TLS.
// - better throughput on poor networks.
//   quic optimizes congestion control to better handle poor network conditions.
// However, h3 is not available in all locations due to dpi/filtering.
// When available, it takes precedence over the default transport.

// packet translation mode gives options for how udp packets are formed on the wire
// We include options here that are known to help with availability

// When packet translation is set, the upgrade mode must be h3 only

// 1: initial version
// 2: latency and speed test support
const TransportVersion = 2

// turn this on to be extra careful about returning all messages
// note we don't run this because it's most efficient to let the gc handle some infrequent orphaned messages
const DebugCloseSend = false

type TransportControl = byte

const (
	TransportControlSpeedStart TransportControl = 1
	TransportControlSpeedStop  TransportControl = 2
)

type TransportMode string

// in order of increasing preference
const (
	// start all modes in skewed parallel and choose the best one
	TransportModeAuto      TransportMode = "auto"
	TransportModeH3DnsPump TransportMode = "h3dnspump"
	TransportModeH3Dns     TransportMode = "h3dns"
	TransportModeH1        TransportMode = "h1"
	TransportModeH3        TransportMode = "h3"
	TransportModeNone      TransportMode = ""
)

type ClientAuth struct {
	ByJwt string
	// ClientId Id
	InstanceId Id
	AppVersion string
}

func (self *ClientAuth) ClientId() (Id, error) {
	byJwt, err := ParseByJwtUnverified(self.ByJwt)
	if err != nil {
		return Id{}, err
	}
	return byJwt.ClientId, nil
}

// (ctx, network, address)
// type DialContextFunc func(ctx context.Context, network string, address string) (net.Conn, error)

type PlatformTransportSettings struct {
	HttpConnectTimeout   time.Duration
	WsHandshakeTimeout   time.Duration
	QuicConnectTimeout   time.Duration
	QuicHandshakeTimeout time.Duration
	QuicTlsConfig        *tls.Config
	AuthTimeout          time.Duration
	ReconnectTimeout     time.Duration
	PingTimeout          time.Duration
	WriteTimeout         time.Duration
	ReadTimeout          time.Duration
	TransportGenerator   func() (sendTransport Transport, receiveTransport Transport)
	TransportBufferSize  int
	InactiveDrainTimeout time.Duration
	// it smoothes out the h3 transition to not start/stop h1 if h3 connects in this time
	ModeInitialDelay time.Duration

	// MinConnectDelay time.Duration
	// MaxConnectDelay time.Duration

	ProtocolVersion int

	H3Port  int
	DnsPort int

	// FIXME
	DnsTlds        [][]byte
	V2H1Auth       bool
	FramerSettings *FramerSettings

	PtDnsSlowMultiple int
}

func DefaultPlatformTransportSettings() *PlatformTransportSettings {
	tlsConfig, err := DefaultTlsConfig()
	if err != nil {
		panic(err)
	}
	return &PlatformTransportSettings{
		HttpConnectTimeout:   15 * time.Second,
		WsHandshakeTimeout:   15 * time.Second,
		QuicConnectTimeout:   15 * time.Second,
		QuicHandshakeTimeout: 15 * time.Second,
		QuicTlsConfig:        tlsConfig,
		AuthTimeout:          5 * time.Second,
		ReconnectTimeout:     5 * time.Second,
		PingTimeout:          5 * time.Second,
		WriteTimeout:         10 * time.Second,
		ReadTimeout:          30 * time.Second,
		TransportBufferSize:  1,
		InactiveDrainTimeout: 30 * time.Second,
		ModeInitialDelay:     2 * time.Second,
		// MinConnectDelay:      0,
		// MaxConnectDelay:      1 * time.Second,
		ProtocolVersion: DefaultProtocolVersion,
		H3Port:          443,
		DnsPort:         53,
		// FIXME
		DnsTlds: [][]byte{[]byte("ur.xyz.")},
		// servers are migrated on 2025-06-12. We can remove this and always use true.
		V2H1Auth:          true,
		FramerSettings:    DefaultFramerSettings(),
		PtDnsSlowMultiple: 4,
	}
}

type PlatformTransport struct {
	ctx    context.Context
	cancel context.CancelFunc

	clientStrategy *ClientStrategy
	routeManager   *RouteManager

	platformUrl string
	auth        *ClientAuth

	settings *PlatformTransportSettings

	stateLock      sync.Mutex
	modeMonitor    *Monitor
	availableModes map[TransportMode]bool
	targetMode     TransportMode
	mode           TransportMode
}

func NewPlatformTransportWithDefaults(
	ctx context.Context,
	clientStrategy *ClientStrategy,
	routeManager *RouteManager,
	platformUrl string,
	auth *ClientAuth,
) *PlatformTransport {
	return NewPlatformTransport(
		ctx,
		clientStrategy,
		routeManager,
		platformUrl,
		auth,
		DefaultPlatformTransportSettings(),
	)
}

func NewPlatformTransport(
	ctx context.Context,
	clientStrategy *ClientStrategy,
	routeManager *RouteManager,
	platformUrl string,
	auth *ClientAuth,
	settings *PlatformTransportSettings,
) *PlatformTransport {
	return NewPlatformTransportWithTargetMode(
		ctx,
		clientStrategy,
		routeManager,
		platformUrl,
		auth,
		TransportModeAuto,
		settings,
	)
}

func NewPlatformTransportWithTargetMode(
	ctx context.Context,
	clientStrategy *ClientStrategy,
	routeManager *RouteManager,
	platformUrl string,
	auth *ClientAuth,
	targetMode TransportMode,
	settings *PlatformTransportSettings,
) *PlatformTransport {
	cancelCtx, cancel := context.WithCancel(ctx)
	transport := &PlatformTransport{
		ctx:    cancelCtx,
		cancel: cancel,
		// cancel: func() {
		// 	select {
		// 	case <- ctx.Done():
		// 	default:
		// 		debug.PrintStack()
		// 		cancel()
		// 	}
		// },
		clientStrategy: clientStrategy,
		routeManager:   routeManager,
		platformUrl:    platformUrl,
		auth:           auth,
		settings:       settings,
		modeMonitor:    NewMonitor(),
		availableModes: map[TransportMode]bool{},
		targetMode:     targetMode,
		mode:           TransportModeNone,
	}
	go HandleError(transport.run, cancel)
	return transport
}

// the auth is used on future connections
func (self *PlatformTransport) SetAuth(auth *ClientAuth) {
	self.stateLock.Lock()
	defer self.stateLock.Unlock()

	self.auth = auth
}

func (self *PlatformTransport) setModeAvailable(mode TransportMode, available bool) {
	self.stateLock.Lock()
	defer self.stateLock.Unlock()

	self.availableModes[mode] = available
}

func (self *PlatformTransport) modeAvailable(mode TransportMode) (bool, chan struct{}) {
	self.stateLock.Lock()
	defer self.stateLock.Unlock()

	return self.availableModes[mode], self.modeMonitor.NotifyChannel()
}

func (self *PlatformTransport) modesAvailable() (map[TransportMode]bool, chan struct{}) {
	self.stateLock.Lock()
	defer self.stateLock.Unlock()

	return maps.Clone(self.availableModes), self.modeMonitor.NotifyChannel()
}

func (self *PlatformTransport) setActiveMode(mode TransportMode) {
	self.stateLock.Lock()
	defer self.stateLock.Unlock()

	self.mode = mode
}

func (self *PlatformTransport) activeMode() (TransportMode, chan struct{}) {
	self.stateLock.Lock()
	defer self.stateLock.Unlock()

	return self.mode, self.modeMonitor.NotifyChannel()
}

var transportModePreferences = map[TransportMode]int{
	TransportModeH3DnsPump: 1,
	TransportModeH3Dns:     2,
	TransportModeH3:        3,
	TransportModeH1:        3,
}

func (self *PlatformTransport) run() {
	defer self.cancel()

	switch self.targetMode {
	case TransportModeAuto:
		go HandleError(func() {
			self.runH1(0)
		}, self.cancel)
		// go HandleError(func() {
		// 	self.runH3(TransportModeH3, 0, 1)
		// }, self.cancel)
		// go HandleError(func() {
		// 	self.runH3(TransportModeH3Dns, self.settings.ModeInitialDelay, self.settings.PtDnsSlowMultiple)
		// }, self.cancel)
		// go HandleError(func() {
		// 	self.runH3(TransportModeH3DnsPump, self.settings.ModeInitialDelay*2, self.settings.PtDnsSlowMultiple)
		// }, self.cancel)
	case TransportModeH3:
		go HandleError(func() {
			self.runH3(TransportModeH3, 0, 1)
		}, self.cancel)
	case TransportModeH1:
		go HandleError(func() {
			self.runH1(0)
		}, self.cancel)
	case TransportModeH3Dns:
		go HandleError(func() {
			self.runH3(TransportModeH3Dns, 0, self.settings.PtDnsSlowMultiple)
		}, self.cancel)
	case TransportModeH3DnsPump:
		go HandleError(func() {
			self.runH3(TransportModeH3DnsPump, 0, self.settings.PtDnsSlowMultiple)
		}, self.cancel)
	}

	for {
		available, notify := self.modesAvailable()

		// descending preference
		orderedModes := maps.Keys(transportModePreferences)
		slices.SortFunc(orderedModes, func(a TransportMode, b TransportMode) int {
			if a == b {
				return 0
			} else if isBetterMode(a, b) {
				return -1
			} else {
				return 1
			}
		})
		if 0 < len(orderedModes) {
			for _, mode := range orderedModes {
				if available[mode] {
					self.setActiveMode(mode)
					break
				}
			}
		} else {
			self.setActiveMode(TransportModeNone)
		}

		select {
		case <-notify:
		case <-self.ctx.Done():
			return
		}
	}
}

// returns true is other is better than current
func isBetterMode(current TransportMode, other TransportMode) bool {
	return transportModePreferences[current] < transportModePreferences[other]
}

func (self *PlatformTransport) runH1(initialTimeout time.Duration) {
	// connect and update route manager for this transport
	defer self.cancel()

	clientId, _ := self.auth.ClientId()

	if 0 < initialTimeout {
		select {
		case <-self.ctx.Done():
			return
		case <-time.After(initialTimeout):
		}
	}

	for {
		// wait until we are back in h1 or worse
		func() {
			for {
				mode, notify := self.activeMode()
				if isBetterMode(TransportModeH1, mode) {
					select {
					case <-self.ctx.Done():
						return
					case <-notify:
					}
				} else {
					return
				}
			}
		}()

		reconnect := NewReconnect(self.settings.ReconnectTimeout)
		connect := func() (*websocket.Conn, error) {
			header := http.Header{}
			if self.settings.V2H1Auth {
				header.Add("Authorization", fmt.Sprintf("Bearer %s", self.auth.ByJwt))
				header.Add("X-UR-AppVersion", self.auth.AppVersion)
				header.Add("X-UR-InstanceId", self.auth.InstanceId.String())
				header.Add("X-UR-TransportVersion", fmt.Sprintf("%d", TransportVersion))
			}

			ws, _, err := self.clientStrategy.WsDialContext(self.ctx, self.platformUrl, header)
			if err != nil {
				return nil, err
			}

			success := false
			defer func() {
				if !success {
					ws.Close()
				}
			}()

			if !self.settings.V2H1Auth {
				authBytes, err := EncodeFrame(&protocol.Auth{
					ByJwt:      self.auth.ByJwt,
					AppVersion: self.auth.AppVersion,
					InstanceId: self.auth.InstanceId.Bytes(),
				}, self.settings.ProtocolVersion)
				if err != nil {
					return nil, err
				}
				defer MessagePoolReturn(authBytes)

				ws.SetWriteDeadline(time.Now().Add(self.settings.AuthTimeout))
				if err := ws.WriteMessage(websocket.BinaryMessage, authBytes); err != nil {
					return nil, err
				}
				ws.SetReadDeadline(time.Now().Add(self.settings.AuthTimeout))
				if messageType, message, err := ws.ReadMessage(); err != nil {
					return nil, err
				} else {
					// verify the auth echo
					switch messageType {
					case websocket.BinaryMessage:
						if !bytes.Equal(authBytes, message) {
							return nil, fmt.Errorf("Auth response error: bad bytes.")
						}
					default:
						return nil, fmt.Errorf("Auth response error.")
					}
				}
			}

			success = true
			return ws, nil
		}

		if connectDelay := self.clientStrategy.NextConnectTime().Sub(time.Now()); 0 < connectDelay {
			select {
			case <-self.ctx.Done():
				return
			case <-time.After(connectDelay):
			}
		}

		var ws *websocket.Conn
		var err error
		if glog.V(2) {
			ws, err = TraceWithReturnError(fmt.Sprintf("[t]connect %s", clientId), connect)
		} else {
			ws, err = connect()
		}
		if err != nil {
			glog.Infof("[t]auth error %s = %s\n", clientId, err)
			select {
			case <-self.ctx.Done():
				return
			case <-reconnect.After():
				continue
			}
		}

		c := func() {
			defer ws.Close()

			self.setModeAvailable(TransportModeH1, true)
			defer self.setModeAvailable(TransportModeH1, false)

			handleCtx, handleCancel := context.WithCancel(self.ctx)
			defer handleCancel()

			readCounter := atomic.Uint64{}
			writeCounter := atomic.Uint64{}

			send := make(chan []byte, self.settings.TransportBufferSize)
			receive := make(chan []byte, self.settings.TransportBufferSize)
			controlSend := make(chan []byte, self.settings.TransportBufferSize)

			drain := func(c chan []byte) {
				for {
					select {
					case message, ok := <-c:
						if !ok {
							return
						}
						MessagePoolReturn(message)
					default:
						return
					}
				}
			}

			var exportedSend chan []byte
			// note: this should be false in production
			//       it seems better to potentially leak messages than to
			//       have an extra inefficiency on the packet path
			if DebugCloseSend {
				// use zero buffer here so that the transport can stop accepting and not drop messages
				exportedSend = make(chan []byte)
				go HandleError(func() {
					defer func() {
						handleCancel()
						close(send)
						drain(send)
					}()
					for {
						select {
						case <-handleCtx.Done():
							return
						case message, ok := <-exportedSend:
							if !ok {
								return
							}
							select {
							case <-handleCtx.Done():
								MessagePoolReturn(message)
								return
							case send <- message:
							}
						}
					}
				}, func() {
					handleCancel()
					close(send)
					drain(send)
				})
			} else {
				exportedSend = send
			}

			// the platform can route any destination,
			// since every client has a platform transport
			var sendTransport Transport
			var receiveTransport Transport
			if self.settings.TransportGenerator != nil {
				sendTransport, receiveTransport = self.settings.TransportGenerator()
			} else {
				sendTransport = NewSendGatewayTransport()
				receiveTransport = NewReceiveGatewayTransport()
			}

			self.routeManager.UpdateTransport(sendTransport, []Route{exportedSend})
			self.routeManager.UpdateTransport(receiveTransport, []Route{receive})

			defer func() {
				self.routeManager.RemoveTransport(sendTransport)
				self.routeManager.RemoveTransport(receiveTransport)
			}()

			go HandleError(func() {
				defer handleCancel()

				for {
					mode, notify := self.activeMode()
					if mode != TransportModeH1 {
						startReadCount := readCounter.Load()
						startWriteCount := writeCounter.Load()
						select {
						case <-handleCtx.Done():
							return
						case <-time.After(self.settings.InactiveDrainTimeout):
							// no activity after cool down, shut down this transport
							if readCounter.Load() == startReadCount && writeCounter.Load() == startWriteCount {
								handleCancel()
							}
						case <-notify:
						}
					} else {
						select {
						case <-handleCtx.Done():
							return
						case <-notify:
						}
					}
				}
			}, handleCancel)

			go HandleError(func() {
				defer handleCancel()

				speedTest := false

				write := func(message []byte) error {
					ws.SetWriteDeadline(time.Now().Add(self.settings.WriteTimeout))
					err := ws.WriteMessage(websocket.BinaryMessage, message)
					MessagePoolReturn(message)
					if err != nil {
						// note that for websocket a dealine timeout cannot be recovered
						glog.Infof("[ts]%s-> error = %s\n", clientId, err)
						return err
					}
					glog.V(2).Infof("[ts]%s->\n", clientId)

					writeCounter.Add(1)
					return nil
				}

				for {
					if speedTest {
						select {
						case <-handleCtx.Done():
							return
						case <-WakeupAfter(self.settings.PingTimeout, self.settings.PingTimeout):
							ws.SetWriteDeadline(time.Now().Add(self.settings.WriteTimeout))
							if err := ws.WriteMessage(websocket.BinaryMessage, make([]byte, 0)); err != nil {
								// note that for websocket a dealine timeout cannot be recovered
								return
							}
						case message, ok := <-controlSend:
							if !ok {
								return
							}
							if len(message) == 5 {
								switch message[0] {
								case TransportControlSpeedStop:
									speedTest = false
								}
							}
							if write(message) != nil {
								return
							}
						}
					} else {
						select {
						case <-handleCtx.Done():
							return
						case message, ok := <-send:
							if !ok {
								return
							}
							// if !MessagePoolCheckShared(message) {
							// 	panic("[t]shared should be set")
							// }

							if len(message) <= 16 {
								glog.Infof("[ts]send message must be >16 bytes (%s)\n", len(message))
								MessagePoolReturn(message)
							} else if write(message) != nil {
								return
							}
						case <-WakeupAfter(self.settings.PingTimeout, self.settings.PingTimeout):
							ws.SetWriteDeadline(time.Now().Add(self.settings.WriteTimeout))
							if err := ws.WriteMessage(websocket.BinaryMessage, make([]byte, 0)); err != nil {
								// note that for websocket a dealine timeout cannot be recovered
								return
							}
						case message, ok := <-controlSend:
							if !ok {
								return
							}
							if len(message) == 5 {
								switch message[0] {
								case TransportControlSpeedStart:
									speedTest = true
								}
							}
							if write(message) != nil {
								return
							}
						}
					}
				}
			}, handleCancel)

			go HandleError(func() {
				defer func() {
					handleCancel()
					close(receive)
					close(controlSend)

					drain(receive)
					drain(controlSend)
				}()

				speedTest := false

				for {
					select {
					case <-handleCtx.Done():
						return
					default:
					}

					ws.SetReadDeadline(time.Now().Add(self.settings.ReadTimeout))
					messageType, r, err := ws.NextReader()
					if err != nil {
						glog.V(2).Infof("[tr]%s<- error = %s\n", clientId, err)
						return
					}

					switch messageType {
					case websocket.BinaryMessage:

						message, err := MessagePoolReadAll(r)
						if err != nil {
							glog.V(2).Infof("[tr]%s<- error = %s\n", clientId, err)
							return
						}

						readCounter.Add(1)

						if len(message) <= 16 {
							if len(message) == 0 {
								// ping
								glog.V(2).Infof("[tr]ping %s<-\n", clientId)
								MessagePoolReturn(message)
							} else if len(message) == 5 {
								switch message[0] {
								case TransportControlSpeedStart:
									speedTest = true
									// echo
									select {
									case <-self.ctx.Done():
										MessagePoolReturn(message)
									case controlSend <- message:
									}
								case TransportControlSpeedStop:
									speedTest = false
									// echo
									select {
									case <-self.ctx.Done():
										MessagePoolReturn(message)
									case controlSend <- message:
									}
								default:
									MessagePoolReturn(message)
								}
							} else if len(message) == 16 {
								// latency test echo
								select {
								case <-self.ctx.Done():
									MessagePoolReturn(message)
								case controlSend <- message:
								}
							} else {
								MessagePoolReturn(message)
							}
							continue
						}
						if speedTest {
							// speed test echo
							select {
							case <-self.ctx.Done():
								MessagePoolReturn(message)
							case controlSend <- message:
							}
							continue
						}

						select {
						case <-handleCtx.Done():
							MessagePoolReturn(message)
							return
						case receive <- message:
							glog.V(2).Infof("[tr]%s<-\n", clientId)
						case <-time.After(self.settings.ReadTimeout):
							glog.Infof("[tr]drop %s<-\n", clientId)
							MessagePoolReturn(message)
						}
					default:
						glog.V(2).Infof("[tr]other=%s %s<-\n", messageType, clientId)
					}

					// messageType, message, err := ws.ReadMessage()
					// if err != nil {
					// 	glog.Infof("[tr]%s<- error = %s\n", clientId, err)
					// 	return
					// }

				}
			}, func() {
				handleCancel()
				close(receive)
				close(controlSend)

				drain(receive)
				drain(controlSend)
			})

			select {
			case <-handleCtx.Done():
			}
		}

		reconnect = NewReconnect(self.settings.ReconnectTimeout)
		if glog.V(2) {
			Trace(fmt.Sprintf("[t]connect run %s", clientId), c)
		} else {
			c()
		}

		select {
		case <-self.ctx.Done():
			return
		case <-reconnect.After():
		}
	}
}

func (self *PlatformTransport) runH3(ptMode TransportMode, initialTimeout time.Duration, slowMultiple int) {
	// connect and update route manager for this transport
	defer self.cancel()

	if slowMultiple < 1 {
		panic(fmt.Errorf("Bad slow multiple: %d", slowMultiple))
	}

	clientId, _ := self.auth.ClientId()

	authBytes, err := EncodeFrame(&protocol.Auth{
		ByJwt:      self.auth.ByJwt,
		AppVersion: self.auth.AppVersion,
		InstanceId: self.auth.InstanceId.Bytes(),
	}, self.settings.ProtocolVersion)
	if err != nil {
		return
	}

	if 0 < initialTimeout {
		select {
		case <-self.ctx.Done():
			return
		case <-time.After(initialTimeout):
		}
	}

	for {
		// wait until we are back in the specific pt mode or auto mode
		func() {
			for {
				mode, notify := self.activeMode()
				if isBetterMode(ptMode, mode) {
					select {
					case <-self.ctx.Done():
						return
					case <-notify:
					}
				} else {
					return
				}
			}
		}()

		reconnect := NewReconnect(self.settings.ReconnectTimeout)

		type ConnStream struct {
			conn   *quic.Conn
			stream *quic.Stream
		}

		connect := func() (*ConnStream, error) {
			// quicConfig := &quic.Config{
			// 	HandshakeIdleTimeout: self.settings.QuicConnectTimeout + self.settings.QuicHandshakeTimeout,
			// }

			success := false

			quicConfig := &quic.Config{
				HandshakeIdleTimeout:    time.Duration(slowMultiple) * (self.settings.QuicConnectTimeout + self.settings.QuicHandshakeTimeout),
				MaxIdleTimeout:          self.settings.PingTimeout * 4,
				KeepAlivePeriod:         0,
				Allow0RTT:               true,
				DisablePathMTUDiscovery: true,
				InitialPacketSize:       1400,
			}
			var tlsConfig *tls.Config
			if self.settings.QuicTlsConfig != nil {
				// copy
				tlsConfigCopy := *self.settings.QuicTlsConfig
				tlsConfig = &tlsConfigCopy
			} else {
				tlsConfig = &tls.Config{}
			}

			var packetConn net.PacketConn

			udpConn, err := net.ListenUDP("udp", &net.UDPAddr{IP: net.IPv4zero, Port: 0})
			if err != nil {
				return nil, err
			}
			defer func() {
				if !success {
					udpConn.Close()
				}
			}()

			// udpAddr, err := net.ResolveUDPAddr("udp", addr)
			// if err != nil {
			// 	return nil, err
			// }

			serverName, err := connectHost(self.platformUrl)
			if err != nil {
				return nil, err
			}
			var udpAddr *net.UDPAddr
			switch ptMode {
			case TransportModeH3Dns:
				tld := self.settings.DnsTlds[mathrand.Intn(len(self.settings.DnsTlds))]
				udpAddr, err = net.ResolveUDPAddr("udp", fmt.Sprintf("%s:%d", serverName, self.settings.DnsPort))
				ptSettings := DefaultPacketTranslationSettings()
				ptSettings.DnsTlds = [][]byte{tld}
				packetConn, err = NewPacketTranslation(self.ctx, PacketTranslationModeDns, udpConn, ptSettings)
				if err != nil {
					return nil, err
				}
			case TransportModeH3DnsPump:
				tld := self.settings.DnsTlds[mathrand.Intn(len(self.settings.DnsTlds))]
				pumpServerName, err := pumpHost(self.platformUrl, tld)
				if err != nil {
					return nil, err
				}
				udpAddr, err = net.ResolveUDPAddr("udp", fmt.Sprintf("%s:%d", pumpServerName, self.settings.DnsPort))
				ptSettings := DefaultPacketTranslationSettings()
				ptSettings.DnsTlds = [][]byte{tld}
				packetConn, err = NewPacketTranslation(self.ctx, PacketTranslationModeDnsPump, udpConn, ptSettings)
				if err != nil {
					return nil, err
				}
			default:
				udpAddr, err = net.ResolveUDPAddr("udp", fmt.Sprintf("%s:%d", serverName, self.settings.H3Port))
				if err != nil {
					return nil, err
				}
				packetConn = udpConn
			}

			defer func() {
				if !success {
					packetConn.Close()
				}
			}()

			glog.Infof("[c]h3 connect to %v (%s)\n", udpAddr, serverName)

			tlsConfig.ServerName = serverName
			quicTransport := &quic.Transport{
				Conn: packetConn,
				// createdConn: true,
				// isSingleUse: true,
			}
			conn, err := quicTransport.DialEarly(self.ctx, udpAddr, tlsConfig, quicConfig)

			// conn, err := quic.Dial(self.ctx, packetConn, packetConn.ConnectedAddr(), self.settings.QuicTlsConfig, quicConfig)
			if err != nil {
				glog.Infof("[c]h3 connect err = %s\n", err)
				return nil, err
			}
			defer func() {
				if !success {
					conn.CloseWithError(0, "")
				}
			}()

			stream, err := conn.OpenStreamSync(self.ctx)
			if err != nil {
				glog.Infof("[c]h3 open stream err = %s\n", err)
				return nil, err
			}

			framer := NewFramer(self.settings.FramerSettings)

			stream.SetWriteDeadline(time.Now().Add(time.Duration(slowMultiple) * self.settings.AuthTimeout))
			if err := framer.Write(stream, authBytes); err != nil {
				return nil, err
			}
			stream.SetReadDeadline(time.Now().Add(time.Duration(slowMultiple) * self.settings.AuthTimeout))
			if message, err := framer.Read(stream); err != nil {
				return nil, err
			} else {
				// verify the auth echo
				if !bytes.Equal(authBytes, message) {
					return nil, fmt.Errorf("Auth response error: bad bytes.")
				}
			}

			success = true
			return &ConnStream{
				conn:   conn,
				stream: stream,
			}, nil
		}

		var connStream *ConnStream
		var err error
		if glog.V(2) {
			connStream, err = TraceWithReturnError(fmt.Sprintf("[t]connect %s", clientId), connect)
		} else {
			connStream, err = connect()
		}
		if err != nil {
			glog.Infof("[t]auth error %s = %s\n", clientId, err)
			select {
			case <-self.ctx.Done():
				return
			case <-reconnect.After():
				continue
			}
		}
		conn := connStream.conn
		stream := connStream.stream

		c := func() {
			defer conn.CloseWithError(0, "")

			self.setModeAvailable(ptMode, true)
			defer self.setModeAvailable(ptMode, false)

			handleCtx, handleCancel := context.WithCancel(self.ctx)
			defer handleCancel()

			framer := NewFramer(self.settings.FramerSettings)

			readCounter := atomic.Uint64{}
			writeCounter := atomic.Uint64{}

			send := make(chan []byte, self.settings.TransportBufferSize)
			receive := make(chan []byte, self.settings.TransportBufferSize)

			// the platform can route any destination,
			// since every client has a platform transport
			var sendTransport Transport
			var receiveTransport Transport
			if self.settings.TransportGenerator != nil {
				sendTransport, receiveTransport = self.settings.TransportGenerator()
			} else {
				sendTransport = NewPrioritySendGatewayTransport(TransportMaxPriority, TransportMaxWeight)
				receiveTransport = NewPriorityReceiveGatewayTransport(TransportMaxPriority, TransportMaxWeight)
			}

			self.routeManager.UpdateTransport(sendTransport, []Route{send})
			self.routeManager.UpdateTransport(receiveTransport, []Route{receive})

			defer func() {
				self.routeManager.RemoveTransport(sendTransport)
				self.routeManager.RemoveTransport(receiveTransport)

				// note `send` is not closed. This channel is left open.
				// it used to be closed after a delay, but it is not needed to close it.
			}()

			go HandleError(func() {
				defer handleCancel()

				for {
					mode, notify := self.activeMode()
					if mode != ptMode {
						startReadCount := readCounter.Load()
						startWriteCount := writeCounter.Load()
						select {
						case <-handleCtx.Done():
							return
						case <-time.After(time.Duration(slowMultiple) * self.settings.InactiveDrainTimeout):
							// no activity after cool down, shut down this transport
							if readCounter.Load() == startReadCount && writeCounter.Load() == startWriteCount {
								handleCancel()
							}
						case <-notify:
						}
					} else {
						select {
						case <-handleCtx.Done():
							return
						case <-notify:
						}
					}
				}
			}, handleCancel)

			go HandleError(func() {
				defer handleCancel()

				for {
					select {
					case <-handleCtx.Done():
						return
					case message, ok := <-send:
						if !ok {
							return
						}
						// if !MessagePoolCheckShared(message) {
						// 	panic("[t]shared should be set")
						// }
						stream.SetWriteDeadline(time.Now().Add(time.Duration(slowMultiple) * self.settings.WriteTimeout))
						err := framer.Write(stream, message)
						MessagePoolReturn(message)
						if err != nil {
							// note that for websocket a dealine timeout cannot be recovered
							glog.Infof("[ts]%s-> error = %s\n", clientId, err)
							return
						}
						glog.V(2).Infof("[ts]%s->\n", clientId)
					case <-WakeupAfter(self.settings.PingTimeout, self.settings.PingTimeout):
						stream.SetWriteDeadline(time.Now().Add(time.Duration(slowMultiple) * self.settings.WriteTimeout))
						if err := framer.Write(stream, make([]byte, 0)); err != nil {
							// note that for websocket a dealine timeout cannot be recovered
							return
						}
					}
				}
			}, handleCancel)

			go HandleError(func() {
				defer func() {
					handleCancel()
					close(receive)
				}()

				for {
					select {
					case <-handleCtx.Done():
						return
					default:
					}

					stream.SetReadDeadline(time.Now().Add(time.Duration(slowMultiple) * self.settings.ReadTimeout))
					message, err := framer.Read(stream)
					if err != nil {
						glog.Infof("[tr]%s<- error = %s\n", clientId, err)
						return
					}

					if 0 == len(message) {
						// ping
						glog.V(2).Infof("[tr]ping %s<-\n", clientId)
						MessagePoolReturn(message)
						continue
					}

					select {
					case <-handleCtx.Done():
						MessagePoolReturn(message)
						return
					case receive <- message:
						glog.V(2).Infof("[tr]%s<-\n", clientId)
					case <-time.After(time.Duration(slowMultiple) * self.settings.ReadTimeout):
						glog.Infof("[tr]drop %s<-\n", clientId)
						MessagePoolReturn(message)
					}
				}
			}, func() {
				handleCancel()
				close(receive)
			})

			select {
			case <-handleCtx.Done():
			}
		}
		reconnect = NewReconnect(self.settings.ReconnectTimeout)
		if glog.V(2) {
			Trace(fmt.Sprintf("[t]connect run %s", clientId), c)
		} else {
			c()
		}

		select {
		case <-self.ctx.Done():
			return
		case <-reconnect.After():
		}
	}
}

func (self *PlatformTransport) Close() {
	self.cancel()
}

func connectHost(platformUrl string) (string, error) {
	u, err := url.Parse(platformUrl)
	if err != nil {
		return "", err
	}
	return u.Hostname(), nil
}

// this host should resolve in dns to the root zone ips for the tld
func pumpHost(platformUrl string, tld []byte) (string, error) {
	u, err := url.Parse(platformUrl)
	if err != nil {
		return "", err
	}
	host := u.Hostname()
	if net.ParseIP(host) != nil {
		return host, nil
	}
	// tld replace . with -
	// zone-<tld>.<base>
	baseHost := strings.SplitN(host, ".", 2)[1]
	pumpHost := fmt.Sprintf("zone-%s.%s", strings.ReplaceAll(string(tld), ".", "-"), baseHost)
	return pumpHost, nil
}
