package com.bringyour.network.ui.feedback

import android.app.Activity
import android.content.res.Configuration
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowForward
import androidx.compose.material.icons.filled.Star
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.painter.Painter
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.compose.ui.graphics.vector.rememberVectorPainter
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.platform.LocalConfiguration
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalSoftwareKeyboardController
import androidx.compose.ui.platform.LocalUriHandler
import androidx.compose.ui.platform.SoftwareKeyboardController
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.res.vectorResource
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.TextLayoutResult
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.input.ImeAction
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.input.TextFieldValue
import androidx.compose.ui.text.withStyle
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import com.bringyour.network.R
import com.bringyour.network.ui.components.ExportLogButton
import com.bringyour.network.ui.components.PromptSolanaDAppStoreReview
import com.bringyour.network.ui.components.ShareLogFileButton
import com.bringyour.network.ui.components.URButton
import com.bringyour.network.ui.components.URSwitch
import com.bringyour.network.ui.components.URTextInput
import com.bringyour.network.ui.components.URTextInputLabel
import com.bringyour.network.ui.components.overlays.OverlayMode
import com.bringyour.network.ui.shared.managers.rememberReviewManager
import com.bringyour.network.ui.shared.models.BundleStore
import com.bringyour.network.ui.shared.viewmodels.OverlayViewModel
import com.bringyour.network.ui.theme.Black
import com.bringyour.network.ui.theme.Pink
import com.bringyour.network.ui.theme.URNetworkTheme
import com.bringyour.network.utils.isTablet
import com.bringyour.sdk.Sdk
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

@Composable
fun FeedbackScreen(
    overlayViewModel: OverlayViewModel,
    bundleStore: BundleStore?,
    feedbackViewModel: FeedbackViewModel = hiltViewModel(),
) {

    FeedbackScreen(
        feedbackMsg = feedbackViewModel.feedbackMsg,
        setFeedbackMsg = feedbackViewModel.setFeedbackMsg,
        sendFeedback = feedbackViewModel.sendFeedback,
        launchOverlay = overlayViewModel.launch,
        isSendEnabled = feedbackViewModel.isSendEnabled,
        starCount = feedbackViewModel.starCount,
        setStarCount = feedbackViewModel.setStarCount,
        bundleStore = bundleStore,
        promptSolanaReview = feedbackViewModel.promptSolanaReview,
        setPromptSolanaReview = feedbackViewModel.setPromptSolanaReview,
        includeLogs = feedbackViewModel.includeLogs.collectAsState().value,
        toggleIncludeLogs = feedbackViewModel.toggleIncludeLogs
    )

}

@Composable
fun FeedbackScreen(
    feedbackMsg: TextFieldValue,
    setFeedbackMsg: (TextFieldValue) -> Unit,
    sendFeedback: () -> Unit,
    launchOverlay: (OverlayMode) -> Unit,
    isSendEnabled: Boolean,
    starCount: Int,
    setStarCount: (Int) -> Unit,
    bundleStore: BundleStore?,
    promptSolanaReview: Boolean,
    setPromptSolanaReview: (Boolean) -> Unit,
    includeLogs: Boolean,
    toggleIncludeLogs: () -> Unit
) {

    val configuration = LocalConfiguration.current
    val isLandscape = configuration.orientation == Configuration.ORIENTATION_LANDSCAPE
    val reviewManagerRequest = rememberReviewManager()
    val context = LocalContext.current
    val scope = rememberCoroutineScope()
    val keyboardController = LocalSoftwareKeyboardController.current

    val promptReview = {
        val activity = context as? Activity
        activity?.let {
            reviewManagerRequest.launchReviewFlow(
                activity = it,
            )
        }
    }


    val submitFeedback = {

        if (feedbackMsg.text.isNotEmpty() || starCount > 0) {

            sendFeedback()

            launchOverlay(OverlayMode.FeedbackSubmitted)

            setFeedbackMsg(TextFieldValue())

            if (starCount == 5) {
                scope.launch {
                    delay(1000)

                    if (bundleStore == BundleStore.SOLANA_DAPP) {
                        // prompt dialog to navigate to review
                        setPromptSolanaReview(true)
                    } else {
                        // PLAY - launch native review prompt
                        promptReview()
                    }

                }
            }

            setStarCount(0)
        }
    }
    Scaffold(
        bottomBar = {

            Box(
                modifier = Modifier
                    .background(Black)
                    .imePadding()
                    .padding(16.dp)
            ) {

                URButton(
                    onClick = {
                        submitFeedback()
                        keyboardController?.hide()
                    },
                    enabled = isSendEnabled
                ) { buttonTextStyle ->
                    Row {
                        Text(
                            stringResource(id = R.string.send),
                            style = buttonTextStyle
                        )
                        Spacer(modifier = Modifier.width(4.dp))
                        Icon(
                            imageVector = Icons.AutoMirrored.Filled.ArrowForward,
                            contentDescription = "Right Arrow",
                            modifier = Modifier.size(16.dp),
                            tint = if (isSendEnabled) Color.White else Color.Gray
                        )
                    }
                }

            }

        }
    ) { innerPadding ->

        if (isTablet() && !isLandscape) {
            Column(
                modifier = Modifier
                    .fillMaxSize()
                    .verticalScroll(rememberScrollState())
                    .padding(innerPadding)
                    .padding(16.dp)
                    .pointerInput(Unit) {
                        detectTapGestures(onTap = {
                            keyboardController?.hide()
                        })
                    },
            ) {

                FeedbackForm(
                    feedbackMsg = feedbackMsg,
                    setFeedbackMsg = setFeedbackMsg,
                    sendFeedback = {
                        if (feedbackMsg.text.isNotEmpty()) {
                            submitFeedback()
                        }
                    },
                    starCount = starCount,
                    setStarCount = setStarCount,
                    includeLogs = includeLogs,
                    toggleIncludeLogs = toggleIncludeLogs,
                    keyboardController = keyboardController
                )

            }
        } else if (isTablet() && isLandscape) {
            Column(
                modifier = Modifier
                    .pointerInput(Unit) {
                        detectTapGestures(onTap = {
                            keyboardController?.hide()
                        })
                    }
                    .width(512.dp)
                    .verticalScroll(rememberScrollState())
                    .padding(innerPadding)
                    .padding(16.dp),
            ) {

                FeedbackForm(
                    feedbackMsg = feedbackMsg,
                    setFeedbackMsg = setFeedbackMsg,
                    sendFeedback = {
                        if (feedbackMsg.text.isNotEmpty()) {
                            submitFeedback()
                        }
                    },
                    starCount = starCount,
                    setStarCount = setStarCount,
                    includeLogs = includeLogs,
                    toggleIncludeLogs = toggleIncludeLogs,
                    keyboardController = keyboardController
                )

            }
        } else {
            /**
             * phone
             */

            Column(
                modifier = Modifier
                    .width(512.dp)
                    .fillMaxHeight()
                    .verticalScroll(rememberScrollState())
                    .padding(16.dp)
                    .padding(innerPadding)
                    .pointerInput(Unit) {
                        detectTapGestures(onTap = {
                            keyboardController?.hide()
                        })
                    }
                ,
                verticalArrangement = Arrangement.SpaceBetween
            ) {

                FeedbackForm(
                    feedbackMsg = feedbackMsg,
                    setFeedbackMsg = setFeedbackMsg,
                    sendFeedback = { submitFeedback() },
                    starCount = starCount,
                    setStarCount = setStarCount,
                    includeLogs = includeLogs,
                    toggleIncludeLogs = toggleIncludeLogs,
                    keyboardController = keyboardController
                )
            }
        }

    }
    
    if (promptSolanaReview) {
        PromptSolanaDAppStoreReview(
            promptReview = {
                promptReview()
            },
            dismiss = {
                setPromptSolanaReview(false)
            }
        )
    }

}

@Composable
private fun FeedbackForm(
    feedbackMsg: TextFieldValue,
    setFeedbackMsg: (TextFieldValue) -> Unit,
    sendFeedback: () -> Unit,
    starCount: Int,
    setStarCount: (Int) -> Unit,
    includeLogs: Boolean,
    toggleIncludeLogs: () -> Unit,
    keyboardController: SoftwareKeyboardController?
) {

    val supportUrl = "https://discord.com/invite/RUNZXMwPRK"

    val uriHandler = LocalUriHandler.current
    
    val discordText = "Discord"
    val feedbackFull = stringResource(id = R.string.send_feedback, discordText)

    val startIndex = feedbackFull.indexOf(discordText)
    val endIndex = startIndex + discordText.length

    val feedbackAnnotatedString = buildAnnotatedString {
        withStyle(style = MaterialTheme.typography.bodyLarge.toSpanStyle().copy(color = Color.White)) {
            append(feedbackFull)
            addStyle(SpanStyle(color = Pink), startIndex, endIndex)
            addStringAnnotation("URL", supportUrl, startIndex, endIndex)
        }
    }

//    val keyboardController = LocalSoftwareKeyboardController.current

    var layoutResult by remember { mutableStateOf<TextLayoutResult?>(null) }

    Column {
        Text(
            stringResource(id = R.string.feedback_header),
            style = MaterialTheme.typography.headlineSmall
        )

        Spacer(modifier = Modifier.height(64.dp))

        Text(
            text = feedbackAnnotatedString,
            style = MaterialTheme.typography.bodyLarge,
            modifier = Modifier.pointerInput(Unit) {
                detectTapGestures { offset ->
                    layoutResult?.let { layoutResult ->
                        val position = layoutResult.getOffsetForPosition(offset)
                        val annotation = feedbackAnnotatedString
                            .getStringAnnotations(position, position)
                            .firstOrNull()

                        if (annotation?.tag == "URL") {
                            uriHandler.openUri(annotation.item)
                        }
                    }
                }
            },
            onTextLayout = { layoutResult = it }
        )

        Spacer(modifier = Modifier.height(32.dp))

        URTextInput(
            value = feedbackMsg,
            onValueChange = { newValue ->
                setFeedbackMsg(newValue)
            },
            label = stringResource(id = R.string.feedback_label),
            placeholder = stringResource(id = R.string.feedback_placeholder),
            maxLines = 5,
            keyboardOptions = KeyboardOptions(
                keyboardType = KeyboardType.Text,
                imeAction = ImeAction.Send
            ),
            onSend = {
                sendFeedback()
                keyboardController?.hide()
            },
            keyboardController = keyboardController
        )

        Spacer(modifier = Modifier.height(16.dp))

        Row(
            modifier = Modifier
                .fillMaxWidth(),
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.CenterVertically
        ) {

            Text(stringResource(id = R.string.feedback_include_logs))

            URSwitch(
                checked = includeLogs,
                toggle = toggleIncludeLogs
            )
        }

        Row() {

            // share
            ShareLogFileButton(logDir = Sdk.getLogDir())

            // save to device
            ExportLogButton(logDir = Sdk.getLogDir())
        }

        Spacer(modifier = Modifier.height(16.dp))

        URTextInputLabel(stringResource(id = R.string.how_are_we_doing))

        Spacer(modifier = Modifier.height(8.dp))

        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.Center
        ) {
            for (index in 1..5) {
                Spacer(modifier = Modifier.width(8.dp))

                val starIcon: Painter = if (index <= starCount) {
                    rememberVectorPainter(image = Icons.Filled.Star)
                } else {
                    rememberVectorPainter(image = ImageVector.vectorResource(id = R.drawable.baseline_star_outline_24))
                }

                Icon(
                    painter = starIcon,
                    contentDescription = if (index <= starCount) "Filled star" else "Empty star",
                    tint = Pink,
                    modifier = Modifier
                        .size(32.dp)
                        .clip(CircleShape)
                        .clickable {
                            setStarCount(index)
                            // starCount.value = index
                        }
                )
                Spacer(modifier = Modifier.width(8.dp))
            }
        }

    }

}

@Preview
@Composable
private fun FeedbackScreenPreview() {

    URNetworkTheme {
        Scaffold(
            modifier = Modifier.fillMaxSize()
        ) { paddingValues ->
            Column(
                modifier = Modifier
                    .padding(paddingValues)
            ) {
                FeedbackScreen(
                    feedbackMsg = TextFieldValue(),
                    setFeedbackMsg = {},
                    sendFeedback = {},
                    launchOverlay = {},
                    isSendEnabled = true,
                    starCount = 3,
                    setStarCount = {},
                    bundleStore = null,
                    promptSolanaReview = false,
                    setPromptSolanaReview = {},
                    includeLogs = false,
                    toggleIncludeLogs = {},
                )
            }
        }
    }
}