//
//  SubscriptionManager.swift
//  URnetwork
//
//  Created by Stuart Kuentzel on 2024/12/31.
//

import Foundation
import StoreKit
import URnetworkSdk

/**
 * For creating a subscription with the App Store
 */
@MainActor
class AppStoreSubscriptionManager: ObservableObject {
    // @Published var products: [Product] = []
    
    @Published var monthlySubscription: Product?
    @Published var yearlySubscription: Product?
    
    @Published var isPurchasing: Bool = false
    @Published private(set) var purchaseSuccess: Bool = false
    
    private var networkId: SdkId?
    var onPurchaseSuccess: () -> Void = {}
    
    private var updateListenerTask: Task<Void, Error>?
    
    init(networkId: SdkId?) {
        self.networkId = networkId
        
        updateListenerTask = listenForTransactions()
        
        Task {
            await fetchProducts()
        }
    }
    
    deinit {
        updateListenerTask?.cancel()
    }
    
    func fetchProducts() async {
        do {
            let productIdentifiers = ["supporter", "supporter_yearly"]
            
            let storeProducts = try await Product.products(for: productIdentifiers)
            
            if let monthlySub = storeProducts.first(where: { $0.id == "supporter" }) {
                self.monthlySubscription = monthlySub
                print("monthly sub price is: \(monthlySub.displayPrice)")
            }
            
            if let yearlySub = storeProducts.first(where: { $0.id == "supporter_yearly" }) {
                self.yearlySubscription = yearlySub
                print("monthly sub price is: \(yearlySub.displayPrice)")
            }
            
            print("Retrieved products: \(storeProducts.count)")
        } catch {
            print("Failed to fetch products: \(error)")
        }
    }
    
    @MainActor
    private func setIsPurchasing(_ isPurchasing: Bool) async {
        self.isPurchasing = isPurchasing
    }
    
    func purchase(product: Product, onSuccess: @escaping (() -> Void)) async throws {
        guard !isPurchasing else { return }
        
        await setIsPurchasing(true)
        
        self.onPurchaseSuccess = onSuccess
        
        do {
            var purchaseOptions: Set<Product.PurchaseOption> = []
            
            guard let networkId = self.networkId else {
                print("no network id found")
                return
            }
            
            print("networkId.idStr: \(networkId.idStr)")
            
            if let networkUUID = UUID(uuidString: networkId.idStr) {
                print("setting purchase options with app account token: \(networkUUID)")
                purchaseOptions.insert(.appAccountToken(networkUUID))
            } else {
                print("error adding network id to purchase options")
                return
            }
            
            let result = try await product.purchase(options: purchaseOptions)
            
            switch result {
            case .success(let verification):
                switch verification {
                case .verified(let transaction):
                    print("✅ Purchase verified: \(transaction.id)")
                    
                    logTransactionDetails(transaction)
                    
                    await transaction.finish()
                    
                    onPurchaseSuccess()
                    setPurchaseSuccess(true)
                    
                case .unverified( _, let error):
                    print("Purchase unverified: \(error)")
                    throw error
                }
                
            case .userCancelled:
                print("Purchase cancelled by user")
                throw SKError(.paymentCancelled)
                
            case .pending:
                print("Purchase pending approval")
                
            @unknown default:
                print("Unknown purchase result")
            }
        } catch {
            print("Purchase failed: \(error)")
            await setIsPurchasing(false)
            throw error
        }
        
        await setIsPurchasing(false)
    }
    
    @MainActor
    func setPurchaseSuccess(_ success: Bool) {
        self.purchaseSuccess = success
    }

    private func logTransactionDetails(_ transaction: Transaction) {
        print("Transaction ID: \(transaction.id)")
        print("Product ID: \(transaction.productID)")
        print("Purchase Date: \(transaction.purchaseDate)")
        
        if let appAccountToken = transaction.appAccountToken {
            print("App Account Token: \(appAccountToken)")
        }
    }
    
    
    private func listenForTransactions() -> Task<Void, Error> {
        return Task.detached {
            // Iterate through any transactions that didn't come from a direct call to `purchase()`
            for await result in Transaction.updates {
                do {
                    // Check if the transaction is verified
                    switch result {
                    case .verified(let transaction):
                        print("✅ Verified transaction update: \(transaction.id)")
                        
                        // Log transaction details for debugging
                        await self.logTransactionDetails(transaction)
                        
                        // Finish the transaction
                        await transaction.finish()
                        
                        // If this is a new purchase notification, not just an update
                        if transaction.revocationDate == nil {
                            await MainActor.run {
                                self.onPurchaseSuccess()
                            }
                        }
                        
                    case .unverified(_, let error):
                        print("Unverified transaction: \(error.localizedDescription)")
                    }
                }
            }
        }
    }
}
