package com.reactlibrary;

import android.content.Intent;
import android.os.Bundle;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;
import android.graphics.Color;
import android.view.Gravity;
import android.util.TypedValue;

import com.journeyapps.barcodescanner.BarcodeResult;
import com.journeyapps.barcodescanner.BarcodeCallback;
import com.journeyapps.barcodescanner.DecoratedBarcodeView;
import com.journeyapps.barcodescanner.CaptureActivity;
import com.journeyapps.barcodescanner.CaptureManager;

import java.util.List;
import java.lang.reflect.Field;
import com.google.zxing.ResultPoint;

/**
 * This Activity supports both single scan (default) and continuous scanning modes.
 * Continuous mode is enabled via intent extra "CONTINUOUS_SCAN".
 */
public class ContinuousCaptureActivity extends CaptureActivity {
    private static final String CONTINUOUS_SCAN_EXTRA = "CONTINUOUS_SCAN";
    private static ContinuousCaptureActivity currentInstance = null;
    private static final Object instanceLock = new Object();
    private boolean isContinuousMode = false;
    private DecoratedBarcodeView barcodeView;
    private String lastScannedText = null;
    private boolean continuousScanningSetup = false;
    private CaptureManager captureManager;
    private TextView progressTextView;
    private String customStatusMessage = null;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        Intent intent = getIntent();
        isContinuousMode = intent != null && intent.getBooleanExtra(CONTINUOUS_SCAN_EXTRA, false);
        // Read custom status message from Intent
        if (intent != null && intent.hasExtra("CUSTOM_STATUS_MESSAGE")) {
            customStatusMessage = intent.getStringExtra("CUSTOM_STATUS_MESSAGE");
        }
        
        if (isContinuousMode) {
            // Store static reference to this activity instance
            synchronized (instanceLock) {
                currentInstance = this;
            }
            // For continuous mode, we need to bypass CaptureManager's single-scan behavior
            // We'll set up the view manually and use decodeContinuous
            super.onCreate(savedInstanceState);
            // The parent onCreate() sets up CaptureManager and calls capture.decode()
            // We'll override this in onResume() by pausing and using decodeContinuous
        } else {
            // Normal single-scan mode - use parent behavior
            super.onCreate(savedInstanceState);
        }
    }

    @Override
    protected DecoratedBarcodeView initializeContent() {
        DecoratedBarcodeView view = super.initializeContent();
        // Store reference to barcode view for continuous scanning
        barcodeView = view;
        return view;
    }

    @Override
    protected void onResume() {
        if (isContinuousMode && !continuousScanningSetup) {
            // For continuous mode, we need to prevent CaptureManager from handling results
            // Try to get barcode view if not already set from initializeContent
            if (barcodeView == null) {
                // Try to find it in the view hierarchy
                View rootView = findViewById(android.R.id.content);
                if (rootView != null) {
                    barcodeView = findBarcodeView(rootView);
                }
            }
            
            // Pause the barcode view BEFORE calling super.onResume()
            // This prevents CaptureManager from starting its decode
            if (barcodeView != null) {
                barcodeView.pause();
            }
        }
        
        super.onResume();
        
        // Set custom status message if provided (for both single and continuous mode)
        if (customStatusMessage != null && !customStatusMessage.isEmpty()) {
            setCustomStatusMessage(customStatusMessage);
        }
        
        if (isContinuousMode && !continuousScanningSetup && barcodeView != null) {
            // Set up continuous scanning after parent onResume completes
            try {
                continuousScanningSetup = true;
                
                // Set up progress text overlay
                setupProgressTextView();
                
                // Try to pause the CaptureManager to prevent it from finishing the activity
                try {
                    Field captureField = CaptureActivity.class.getDeclaredField("capture");
                    captureField.setAccessible(true);
                    CaptureManager capture = (CaptureManager) captureField.get(this);
                    if (capture != null) {
                        // Pause the CaptureManager so it doesn't handle results
                        capture.onPause();
                    }
                } catch (Exception e) {
                    // Reflection failed - that's okay, we'll rely on decodeContinuous
                    e.printStackTrace();
                }
                
                // Use decodeContinuous instead of the default decodeSingle
                // This prevents the activity from auto-finishing after each scan
                // Reset lastScannedText for new scan session
                lastScannedText = null;
                barcodeView.decodeContinuous(new BarcodeCallback() {
                    @Override
                    public void barcodeResult(BarcodeResult result) {
                        String text = result.getText();
                        // Send all results - let the module handle deduplication
                        // The module's sendContinuousScanResult has its own duplicate handling
                        // and we want to allow UR decoder to receive all frames for fountain codes
                        if (text != null) {
                            // Send result via static callback in module
                            // The sendContinuousScanResult method has its own synchronization and duplicate check
                            BarcodeZxingScanModule.sendContinuousScanResult(text);
                            // IMPORTANT: Don't finish the activity - keep scanning
                            // The decodeContinuous callback doesn't trigger activity finish
                        }
                    }

                    @Override
                    public void possibleResultPoints(List<ResultPoint> resultPoints) {
                        // Optional: handle result points for UI feedback
                    }
                });
                
                // Resume scanning in continuous mode
                barcodeView.resume();
            } catch (Exception e) {
                // If we can't set up continuous scanning, fall back to default behavior
                e.printStackTrace();
            }
        }
    }
    
    @Override
    protected void onPause() {
        // In continuous mode, don't pause if we want to keep scanning
        // But we should still call super to handle camera properly
        if (!isContinuousMode) {
            super.onPause();
        } else {
            // In continuous mode, we might want to keep the camera active
            // But we need to call super to maintain proper lifecycle
            super.onPause();
        }
    }
    
    @Override
    protected void onDestroy() {
        // Clear static reference when activity is destroyed
        synchronized (instanceLock) {
            if (currentInstance == this) {
                currentInstance = null;
            }
        }
        super.onDestroy();
    }
    
    /**
     * Static method to finish the current continuous scan activity
     * Called from BarcodeZxingScanModule.stopContinuousScan()
     */
    public static void finishCurrentInstance() {
        synchronized (instanceLock) {
            if (currentInstance != null) {
                final ContinuousCaptureActivity instance = currentInstance;
                // Clear the reference immediately to prevent reuse
                currentInstance = null;
                instance.runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        if (!instance.isFinishing() && !instance.isDestroyed()) {
                            instance.finish();
                        }
                    }
                });
            }
        }
    }
    
    /**
     * Static method to update the progress text overlay
     * Called from BarcodeZxingScanModule.updateProgressText()
     */
    public static void updateProgressText(final String text) {
        synchronized (instanceLock) {
            if (currentInstance != null) {
                final ContinuousCaptureActivity instance = currentInstance;
                instance.runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        if (instance.progressTextView != null && !instance.isFinishing() && !instance.isDestroyed()) {
                            if (text != null && !text.isEmpty()) {
                                instance.progressTextView.setText(text);
                                instance.progressTextView.setVisibility(View.VISIBLE);
                            } else {
                                instance.progressTextView.setVisibility(View.GONE);
                            }
                        }
                    }
                });
            }
        }
    }
    
    /**
     * Set up the progress text overlay view
     */
    private void setupProgressTextView() {
        try {
            View rootView = findViewById(android.R.id.content);
            if (rootView instanceof ViewGroup) {
                ViewGroup rootGroup = (ViewGroup) rootView;
                
                // Create TextView for progress feedback
                progressTextView = new TextView(this);
                progressTextView.setText("PSBT signing progress...");
                progressTextView.setTextColor(Color.WHITE);
                progressTextView.setTextSize(TypedValue.COMPLEX_UNIT_SP, 16);
                progressTextView.setGravity(Gravity.CENTER);
                progressTextView.setPadding(32, 24, 32, 24);
                
                // Semi-transparent dark background
                progressTextView.setBackgroundColor(Color.argb(200, 0, 0, 0));
                
                // Position at top center
                ViewGroup.MarginLayoutParams params = new ViewGroup.MarginLayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT,
                    ViewGroup.LayoutParams.WRAP_CONTENT
                );
                params.topMargin = (int) TypedValue.applyDimension(
                    TypedValue.COMPLEX_UNIT_DIP, 80, getResources().getDisplayMetrics()
                );
                progressTextView.setLayoutParams(params);
                
                // Initially hidden, will be shown when progress updates
                progressTextView.setVisibility(View.GONE);
                
                // Add to root view
                rootGroup.addView(progressTextView);
            }
        } catch (Exception e) {
            // If we can't set up the overlay, that's okay - scanning will still work
            e.printStackTrace();
        }
    }

    @Override
    protected void onStop() {
        // Clear static reference when activity stops (before destroy)
        // This ensures the reference is cleared even if onDestroy is delayed
        if (isContinuousMode) {
            synchronized (instanceLock) {
                if (currentInstance == this) {
                    currentInstance = null;
                }
            }
        }
        super.onStop();
    }

    // Helper method to find DecoratedBarcodeView in view hierarchy
    private DecoratedBarcodeView findBarcodeView(View view) {
        if (view instanceof DecoratedBarcodeView) {
            return (DecoratedBarcodeView) view;
        }
        if (view instanceof ViewGroup) {
            ViewGroup group = (ViewGroup) view;
            for (int i = 0; i < group.getChildCount(); i++) {
                DecoratedBarcodeView found = findBarcodeView(group.getChildAt(i));
                if (found != null) {
                    return found;
                }
            }
        }
        return null;
    }
    
    /**
     * Set custom status message on the ViewfinderView
     * Uses reflection to access the status TextView in the ZXing library
     */
    private void setCustomStatusMessage(String message) {
        try {
            // Find the DecoratedBarcodeView
            View rootView = findViewById(android.R.id.content);
            if (rootView == null) {
                return;
            }
            
            DecoratedBarcodeView barcodeView = findBarcodeView(rootView);
            if (barcodeView == null) {
                return;
            }
            
            // Use reflection to access the ViewfinderView and its status TextView
            // The ViewfinderView is typically a child of DecoratedBarcodeView
            ViewGroup barcodeViewGroup = (ViewGroup) barcodeView;
            for (int i = 0; i < barcodeViewGroup.getChildCount(); i++) {
                View child = barcodeViewGroup.getChildAt(i);
                // Look for ViewfinderView (com.journeyapps.barcodescanner.ViewfinderView)
                if (child.getClass().getName().contains("ViewfinderView")) {
                    // Try to find TextView inside ViewfinderView
                    if (child instanceof ViewGroup) {
                        ViewGroup viewfinderGroup = (ViewGroup) child;
                        for (int j = 0; j < viewfinderGroup.getChildCount(); j++) {
                            View viewfinderChild = viewfinderGroup.getChildAt(j);
                            if (viewfinderChild instanceof TextView) {
                                TextView statusTextView = (TextView) viewfinderChild;
                                // Check if this is the status text view (usually has specific ID or is the only TextView)
                                // Set the custom message
                                statusTextView.setText(message);
                                return;
                            }
                        }
                    }
                    // Alternative: Try to use reflection to access statusView field
                    try {
                        Field statusViewField = child.getClass().getDeclaredField("statusView");
                        statusViewField.setAccessible(true);
                        TextView statusView = (TextView) statusViewField.get(child);
                        if (statusView != null) {
                            statusView.setText(message);
                            return;
                        }
                    } catch (NoSuchFieldException e) {
                        // Field doesn't exist, try other approaches
                    }
                }
            }

            // Fallback: Try to find any TextView that might be the status view
            // by checking if it contains the default status message
            TextView statusTextView = findStatusTextView(rootView);
            if (statusTextView != null) {
                statusTextView.setText(message);
            }
        } catch (Exception e) {
            // If we can't set the custom message, that's okay - scanner will still work
            e.printStackTrace();
        }
    }

    /**
     * Helper method to find the status TextView in the view hierarchy
     */
    private TextView findStatusTextView(View view) {
        if (view instanceof TextView) {
            TextView textView = (TextView) view;
            String text = textView.getText().toString();
            // Check if this TextView contains the default status message
            if (text.contains("barcode") || text.contains("viewfinder") || text.contains("scan")) {
                return textView;
            }
        }
        if (view instanceof ViewGroup) {
            ViewGroup group = (ViewGroup) view;
            for (int i = 0; i < group.getChildCount(); i++) {
                TextView found = findStatusTextView(group.getChildAt(i));
                if (found != null) {
                    return found;
                }
            }
        }
        return null;
    }
}
