'use strict';

import { isDefined, isNumber, ReanimatedError } from '../../../../common';
import { PERCENTAGE_REGEX } from '../../../constants';
import { normalizeTimingFunction } from '../common';
export const ERROR_MESSAGES = {
  invalidOffsetType: selector => `Invalid keyframe selector "${selector}". Only numbers, percentages, "from", and "to" are supported.`,
  invalidOffsetRange: selector => `Invalid keyframe selector "${selector}". Expected a number between 0 and 1 or a percentage between 0% and 100%.`
};
export function normalizeKeyframeSelector(keyframeSelector) {
  const selectors = typeof keyframeSelector === 'string' ? keyframeSelector.split(',').map(k => k.trim()) : [keyframeSelector];
  const offsets = selectors.map(selector => {
    if (selector === 'from') {
      return 0;
    }
    if (selector === 'to') {
      return 1;
    }
    let offset;
    if (typeof selector === 'number' || !isNaN(+selector)) {
      offset = +selector;
    } else if (PERCENTAGE_REGEX.test(selector)) {
      offset = parseFloat(selector) / 100;
    }
    if (!isNumber(offset)) {
      throw new ReanimatedError(ERROR_MESSAGES.invalidOffsetType(selector));
    }
    if (offset < 0 || offset > 1) {
      throw new ReanimatedError(ERROR_MESSAGES.invalidOffsetRange(selector));
    }
    return offset;
  });
  return offsets;
}
export function processKeyframes(keyframes, styleBuilder) {
  return Object.entries(keyframes).flatMap(([selector, {
    animationTimingFunction = undefined,
    ...style
  } = {}]) => {
    const normalizedStyle = styleBuilder.buildFrom(style);
    if (!normalizedStyle) {
      return [];
    }
    return normalizeKeyframeSelector(selector).map(offset => ({
      offset,
      style: normalizedStyle,
      ...(animationTimingFunction && {
        timingFunction: animationTimingFunction
      })
    }));
  }).sort((a, b) => a.offset - b.offset).reduce((acc, keyframe) => {
    const lastKeyframe = acc[acc.length - 1];
    if (lastKeyframe && lastKeyframe.offset === keyframe.offset) {
      lastKeyframe.style = {
        ...lastKeyframe.style,
        ...keyframe.style
      };
      lastKeyframe.timingFunction = keyframe.timingFunction;
    } else {
      acc.push(keyframe);
    }
    return acc;
  }, []);
}
function processStyleProperties(offset, style, keyframeStyle, styleBuilder) {
  Object.entries(style).forEach(([property, value]) => {
    if (!isDefined(value)) {
      return;
    }
    if (typeof value === 'object') {
      if (styleBuilder.isSeparatelyInterpolatedNestedProperty(property)) {
        if (!keyframeStyle[property]) {
          keyframeStyle[property] = Array.isArray(value) ? [] : {};
        }
        processStyleProperties(offset, value, keyframeStyle[property], styleBuilder);
        return;
      }
    }
    if (!keyframeStyle[property]) {
      keyframeStyle[property] = [];
    }
    keyframeStyle[property].push({
      offset,
      value
    });
  });
}
export function normalizeAnimationKeyframes(keyframes, styleBuilder) {
  const keyframesStyle = {};
  const timingFunctions = {};
  processKeyframes(keyframes, styleBuilder).forEach(({
    offset,
    style,
    timingFunction
  }) => {
    processStyleProperties(offset, style, keyframesStyle, styleBuilder);
    if (timingFunction && offset < 1) {
      timingFunctions[offset] = normalizeTimingFunction(timingFunction);
    }
  });
  return {
    keyframesStyle,
    keyframeTimingFunctions: timingFunctions
  };
}
//# sourceMappingURL=keyframes.js.map