'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var bcUrRegistry = require('@keystonehq/bc-ur-registry');
var uuid = require('uuid');

const ExtendedRegistryTypes = {
  BTC_SIGN_REQUEST: /*#__PURE__*/new bcUrRegistry.RegistryType("btc-sign-request", 8101),
  BTC_SIGNATURE: /*#__PURE__*/new bcUrRegistry.RegistryType("btc-signature", 8102)
};

const {
  decodeToDataItem,
  RegistryTypes
} = bcUrRegistry.extend;
(function (DataType) {
  DataType[DataType["message"] = 1] = "message";
})(exports.DataType || (exports.DataType = {}));
var Keys;
(function (Keys) {
  Keys[Keys["requestId"] = 1] = "requestId";
  Keys[Keys["signData"] = 2] = "signData";
  Keys[Keys["dataType"] = 3] = "dataType";
  Keys[Keys["derivationPaths"] = 4] = "derivationPaths";
  Keys[Keys["addresses"] = 5] = "addresses";
  Keys[Keys["origin"] = 6] = "origin";
})(Keys || (Keys = {}));
class BtcSignRequest extends bcUrRegistry.RegistryItem {
  constructor(args) {
    super();
    this.getRegistryType = () => ExtendedRegistryTypes.BTC_SIGN_REQUEST;
    this.getRequestId = () => this.requestId;
    this.getSignData = () => this.signData;
    this.getDataype = () => this.dataType;
    this.getDerivationPaths = () => this.derivationPaths.map(key => key.getPath());
    this.getAddress = () => this.addresses;
    this.getOrigin = () => this.origin;
    this.toDataItem = () => {
      const map = {};
      map[Keys.requestId] = new bcUrRegistry.DataItem(this.requestId, RegistryTypes.UUID.getTag());
      map[Keys.signData] = this.signData;
      map[Keys.dataType] = this.dataType;
      map[Keys.derivationPaths] = this.derivationPaths.map(item => {
        const dataItem = item.toDataItem();
        dataItem.setTag(item.getRegistryType().getTag());
        return dataItem;
      });
      if (this.addresses) {
        map[Keys.addresses] = this.addresses;
      }
      if (this.origin) {
        map[Keys.origin] = this.origin;
      }
      return new bcUrRegistry.DataItem(map);
    };
    this.requestId = args.requestId;
    this.signData = args.signData;
    this.dataType = args.dataType;
    this.derivationPaths = args.derivationPaths;
    this.addresses = args.addresses;
    this.origin = args.origin;
  }
  static constructBtcRequest(uuidString, xfps, signData, dataType, derivationHDPaths, addresses, origin) {
    const derivationHdPathObjects = derivationHDPaths.map((path, index) => {
      const paths = path.replace(/[m|M]\//, "").split("/");
      const pathComponent = paths.map(path => {
        const index = parseInt(path.replace("'", ""));
        let isHardened = false;
        if (path.endsWith("'")) {
          isHardened = true;
        }
        return new bcUrRegistry.PathComponent({
          index,
          hardened: isHardened
        });
      });
      return new bcUrRegistry.CryptoKeypath(pathComponent, Buffer.from(xfps[index], "hex"));
    });
    return new BtcSignRequest({
      requestId: Buffer.from(uuid.parse(uuidString)),
      signData,
      dataType: dataType || exports.DataType.message,
      derivationPaths: derivationHdPathObjects,
      addresses: addresses || undefined,
      origin: origin || undefined
    });
  }
}
BtcSignRequest.fromDataItem = dataItem => {
  const map = dataItem.getData();
  const requestId = map[Keys.requestId] ? map[Keys.requestId].getData() : undefined;
  const signData = map[Keys.signData];
  const dataType = map[Keys.dataType];
  const derivationPaths = map[Keys.derivationPaths].map(item => bcUrRegistry.CryptoKeypath.fromDataItem(item));
  const addresses = map[Keys.addresses] ? map[Keys.addresses] : undefined;
  const origin = map[Keys.origin] ? map[Keys.origin] : undefined;
  return new BtcSignRequest({
    requestId,
    signData,
    dataType,
    derivationPaths,
    addresses,
    origin
  });
};
BtcSignRequest.fromCBOR = _cborPayload => {
  const dataItem = decodeToDataItem(_cborPayload);
  return BtcSignRequest.fromDataItem(dataItem);
};

const {
  RegistryTypes: RegistryTypes$1,
  decodeToDataItem: decodeToDataItem$1
} = bcUrRegistry.extend;
var Keys$1;
(function (Keys) {
  Keys[Keys["requestId"] = 1] = "requestId";
  Keys[Keys["signature"] = 2] = "signature";
  Keys[Keys["publicKey"] = 3] = "publicKey";
})(Keys$1 || (Keys$1 = {}));
class BtcSignature extends bcUrRegistry.RegistryItem {
  constructor(signature, requestId, publicKey) {
    super();
    this.getRegistryType = () => ExtendedRegistryTypes.BTC_SIGNATURE;
    this.getRequestId = () => this.requestId;
    this.getSignature = () => this.signature;
    this.getPublicKey = () => this.publicKey;
    this.toDataItem = () => {
      const map = {};
      map[Keys$1.requestId] = new bcUrRegistry.DataItem(this.requestId, RegistryTypes$1.UUID.getTag());
      map[Keys$1.signature] = this.signature;
      map[Keys$1.publicKey] = this.publicKey;
      return new bcUrRegistry.DataItem(map);
    };
    this.signature = signature;
    this.requestId = requestId;
    this.publicKey = publicKey;
  }
}
BtcSignature.fromDataItem = dataItem => {
  const map = dataItem.getData();
  const signature = map[Keys$1.signature];
  const requestId = map[Keys$1.requestId].getData();
  const authenticationPublicKey = map[Keys$1.publicKey];
  return new BtcSignature(signature, requestId, authenticationPublicKey);
};
BtcSignature.fromCBOR = _cborPayload => {
  const dataItem = decodeToDataItem$1(_cborPayload);
  return BtcSignature.fromDataItem(dataItem);
};

bcUrRegistry.patchTags(Object.values(ExtendedRegistryTypes).filter(rt => !!rt.getTag()).map(rt => rt.getTag()));

Object.keys(bcUrRegistry).forEach(function (k) {
  if (k !== 'default') Object.defineProperty(exports, k, {
    enumerable: true,
    get: function () {
      return bcUrRegistry[k];
    }
  });
});
exports.BtcSignRequest = BtcSignRequest;
exports.BtcSignature = BtcSignature;
//# sourceMappingURL=bc-ur-registry-btc.cjs.development.js.map
