package com.reactlibrary;

import android.app.Activity;
import android.content.Intent;

import com.facebook.react.bridge.ActivityEventListener;
import com.facebook.react.bridge.Arguments;
import com.facebook.react.bridge.Promise;
import com.facebook.react.bridge.ReactApplicationContext;
import com.facebook.react.bridge.ReactContextBaseJavaModule;
import com.facebook.react.bridge.ReactMethod;
import com.facebook.react.bridge.Callback;
import com.facebook.react.bridge.WritableMap;
import com.facebook.react.modules.core.DeviceEventManagerModule;
import com.google.zxing.integration.android.IntentIntegrator;
import com.google.zxing.integration.android.IntentResult;

public class BarcodeZxingScanModule extends ReactContextBaseJavaModule implements ActivityEventListener {

    private final ReactApplicationContext reactContext;
    private Callback mCallback;
    private static final Object callbackLock = new Object();
    private static String lastProcessedText = null;
    private static ReactApplicationContext staticReactContext = null;
    private static String customStatusMessage = null;

    public BarcodeZxingScanModule(ReactApplicationContext reactContext) {
        super(reactContext);
        this.reactContext = reactContext;
        this.reactContext.addActivityEventListener(this); // Register the activity event listener
        synchronized (callbackLock) {
            staticReactContext = reactContext; // Store for continuous scanning events
        }
    }

    @Override
    public String getName() {
        return "BarcodeZxingScan";
    }

    @ReactMethod
    public void sampleMethod(String stringArgument, Promise promise) {
        try {
            promise.resolve(stringArgument);
        } catch (Exception e) {
            promise.reject(e);
        }
    }

    @ReactMethod
    public void showQrReader(Callback callback) {
        this.mCallback = callback; // Assign the callback
        Activity currentActivity = getCurrentActivity();

        if (currentActivity != null) {
            IntentIntegrator integrator = new IntentIntegrator(currentActivity);
            integrator.setOrientationLocked(true);
            integrator.setBeepEnabled(false);      // Disable beep sound
            integrator.setCaptureActivity(ContinuousCaptureActivity.class);
            // Pass custom status message via Intent
            Intent intent = integrator.createScanIntent();
            if (customStatusMessage != null && !customStatusMessage.isEmpty()) {
                intent.putExtra("CUSTOM_STATUS_MESSAGE", customStatusMessage);
            }
            currentActivity.startActivityForResult(intent, IntentIntegrator.REQUEST_CODE);
            reactContext.addActivityEventListener(this); // Ensure we handle the result
        } else {
            callback.invoke("Error: Activity is null");
        }
    }

    @ReactMethod
    public void showQrReaderContinuous(Callback callback) {
        this.mCallback = null; // Don't use single-scan callback
        // Don't store callback - we'll use EventEmitter instead for continuous scanning
        // Callbacks can only be invoked once, but we need to send multiple events
        Activity currentActivity = getCurrentActivity();

        if (currentActivity != null) {
            synchronized (callbackLock) {
                lastProcessedText = null; // Reset for new scan session
            }
            IntentIntegrator integrator = new IntentIntegrator(currentActivity);
            integrator.setOrientationLocked(true);
            integrator.setBeepEnabled(false);
            integrator.setCaptureActivity(ContinuousCaptureActivity.class);
            // Add extra to indicate continuous mode
            Intent intent = integrator.createScanIntent();
            intent.putExtra("CONTINUOUS_SCAN", true);
            // Pass custom status message via Intent if set
            if (customStatusMessage != null && !customStatusMessage.isEmpty()) {
                intent.putExtra("CUSTOM_STATUS_MESSAGE", customStatusMessage);
            }
            // Use the REQUEST_CODE constant from IntentIntegrator
            currentActivity.startActivityForResult(intent, IntentIntegrator.REQUEST_CODE);
            // Invoke callback once to acknowledge scanner is starting
            callback.invoke(null, "SCANNER_STARTED");
        } else {
            callback.invoke("Error: Activity is null", null);
        }
    }

    public static void sendContinuousScanResult(String text) {
        synchronized (callbackLock) {
            if (staticReactContext != null && text != null) {
                // For animated QR codes (UR format), the decoder uses fountain codes
                // The UR decoder can handle duplicate frames internally, so we don't need
                // to filter them here. However, to avoid excessive spam, we'll still track
                // the last processed text but allow duplicates through (the decoder will deduplicate).
                // We only skip if it's the exact same text that was just sent (within the same scan cycle).
                // Actually, let's be more lenient - allow all frames through and let the
                // UR decoder handle deduplication. The decoder is designed for this.
                
                // Update last processed text for reference, but don't block duplicates
                lastProcessedText = text;
                
                // Use EventEmitter instead of Callback for continuous scanning
                // Callbacks can only be invoked once, but EventEmitter can send multiple events
                try {
                    WritableMap params = Arguments.createMap();
                    params.putString("data", text);
                    params.putString("error", null);
                    
                    staticReactContext
                        .getJSModule(DeviceEventManagerModule.RCTDeviceEventEmitter.class)
                        .emit("BarcodeZxingScanContinuous", params);
                } catch (Exception e) {
                    // Event emission failed
                    e.printStackTrace();
                }
            }
        }
    }

    @ReactMethod
    public void stopContinuousScan() {
        synchronized (callbackLock) {
            // Clear state to stop receiving results
            lastProcessedText = null;
            // Use the static method to finish the continuous scan activity
            ContinuousCaptureActivity.finishCurrentInstance();
        }
    }

    @ReactMethod
    public void updateProgressText(String text) {
        // Update the progress text overlay in the continuous scan activity
        ContinuousCaptureActivity.updateProgressText(text);
    }

    @ReactMethod
    public void setStatusMessage(String message) {
        // Set custom status message for the scanner
        // This will be used when opening the scanner activity
        synchronized (callbackLock) {
            customStatusMessage = message;
        }
    }

    @Override
    public void onActivityResult(Activity activity, int requestCode, int resultCode, Intent data) {
        if (mCallback != null) {
            // Use IntentIntegrator to parse the activity result
            IntentResult result = IntentIntegrator.parseActivityResult(requestCode, resultCode, data);
            if (result != null) {
                if (result.getContents() != null) {
                    // Pass the scanned data back to the callback
                    mCallback.invoke(null, result.getContents());
                } else {
                    // No result; notify the callback
                    mCallback.invoke("Scan canceled", null);
                }
            } else {
                // Parsing failed or unrelated result
                mCallback.invoke("Error: Failed to parse scan result", null);
            }
        }
    }


    @Override
    public void onNewIntent(Intent intent) {
        // No specific behavior required for this module
    }
}