/*
 * Copyright (C) 2016 The Android Open Source Project
 * modified
 * SPDX-License-Identifier: Apache-2.0 AND GPL-3.0-only
 */

package com.best.deskclock.data;

import static java.util.Calendar.DAY_OF_WEEK;
import static java.util.Calendar.FRIDAY;
import static java.util.Calendar.MONDAY;
import static java.util.Calendar.SATURDAY;
import static java.util.Calendar.SUNDAY;
import static java.util.Calendar.THURSDAY;
import static java.util.Calendar.TUESDAY;
import static java.util.Calendar.WEDNESDAY;

import android.content.Context;
import android.graphics.Color;
import android.graphics.Typeface;
import android.text.SpannableStringBuilder;
import android.text.Spanned;
import android.text.TextUtils;
import android.text.style.ForegroundColorSpan;
import android.text.style.StyleSpan;
import android.util.ArrayMap;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

import com.best.deskclock.R;
import com.google.android.material.color.MaterialColors;

import java.text.DateFormatSymbols;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * This class is responsible for encoding a weekly repeat cycle in a {@link #getBits bitset}. It
 * also converts between those bits and the {@link Calendar#DAY_OF_WEEK} values for easier mutation
 * and querying.
 *
 * @param mBits An encoded form of a weekly repeat schedule.
 */
public record Weekdays(int mBits) {

    /**
     * An instance with no weekdays in the weekly repeat cycle.
     */
    public static final Weekdays NONE = Weekdays.fromBits(0);
    /**
     * All valid bits set.
     */
    private static final int ALL_DAYS = 0x7F;

    /**
     * Maps calendar weekdays to the bit masks that represent them in this class.
     */
    private static final Map<Integer, Integer> sCalendarDayToBit;

    static {
        final Map<Integer, Integer> map = new ArrayMap<>(7);
        map.put(MONDAY, 0x01);
        map.put(TUESDAY, 0x02);
        map.put(WEDNESDAY, 0x04);
        map.put(THURSDAY, 0x08);
        map.put(FRIDAY, 0x10);
        map.put(SATURDAY, 0x20);
        map.put(SUNDAY, 0x40);
        sCalendarDayToBit = Collections.unmodifiableMap(map);
    }

    public Weekdays(int mBits) {
        // Mask off the unused bits.
        this.mBits = ALL_DAYS & mBits;
    }

    /**
     * @param bits {@link #getBits bits} representing the encoded weekly repeat schedule
     * @return a Weekdays instance representing the same repeat schedule as the {@code bits}
     */
    public static Weekdays fromBits(int bits) {
        return new Weekdays(bits);
    }

    /**
     * @param calendarDays an array containing any or all of the following values
     *                     <ul>
     *                     <li>{@link Calendar#SUNDAY}</li>
     *                     <li>{@link Calendar#MONDAY}</li>
     *                     <li>{@link Calendar#TUESDAY}</li>
     *                     <li>{@link Calendar#WEDNESDAY}</li>
     *                     <li>{@link Calendar#THURSDAY}</li>
     *                     <li>{@link Calendar#FRIDAY}</li>
     *                     <li>{@link Calendar#SATURDAY}</li>
     *                     </ul>
     * @return a Weekdays instance representing the given {@code calendarDays}
     */
    public static Weekdays fromCalendarDays(int... calendarDays) {
        int bits = 0;
        for (int calendarDay : calendarDays) {
            final Integer bit = sCalendarDayToBit.get(calendarDay);
            if (bit != null) {
                bits = bits | bit;
            }
        }
        return new Weekdays(bits);
    }

    /**
     * @param calendarDay any of the following values
     *                    <ul>
     *                    <li>{@link Calendar#SUNDAY}</li>
     *                    <li>{@link Calendar#MONDAY}</li>
     *                    <li>{@link Calendar#TUESDAY}</li>
     *                    <li>{@link Calendar#WEDNESDAY}</li>
     *                    <li>{@link Calendar#THURSDAY}</li>
     *                    <li>{@link Calendar#FRIDAY}</li>
     *                    <li>{@link Calendar#SATURDAY}</li>
     *                    </ul>
     * @param on          {@code true} if the {@code calendarDay} is on; {@code false} otherwise
     * @return a WeekDays instance with the {@code calendarDay} mutated
     */
    public Weekdays setBit(int calendarDay, boolean on) {
        final Integer bit = sCalendarDayToBit.get(calendarDay);
        if (bit == null) {
            return this;
        }
        return new Weekdays(on ? (mBits | bit) : (mBits & ~bit));
    }

    /**
     * @param calendarDay any of the following values
     *                    <ul>
     *                    <li>{@link Calendar#SUNDAY}</li>
     *                    <li>{@link Calendar#MONDAY}</li>
     *                    <li>{@link Calendar#TUESDAY}</li>
     *                    <li>{@link Calendar#WEDNESDAY}</li>
     *                    <li>{@link Calendar#THURSDAY}</li>
     *                    <li>{@link Calendar#FRIDAY}</li>
     *                    <li>{@link Calendar#SATURDAY}</li>
     *                    </ul>
     * @return {@code true} if the given {@code calendarDay}
     */
    public boolean isBitOn(int calendarDay) {
        final Integer bit = sCalendarDayToBit.get(calendarDay);
        if (bit == null) {
            throw new IllegalArgumentException(calendarDay + " is not a valid weekday");
        }
        return (mBits & bit) > 0;
    }

    /**
     * @return the weekly repeat schedule encoded as an integer
     */
    public int getBits() {
        return mBits;
    }

    /**
     * @return {@code true} if at least one weekday is enabled in the repeat schedule
     */
    public boolean isRepeating() {
        return mBits != 0;
    }

    /**
     * @return {@code true} if all days of the week are selected; {@code false} otherwise.
     */
    public boolean isAllDaysSelected() {
        return mBits == ALL_DAYS;
    }

    /**
     * Note: only the day-of-week is read from the {@code time}. The time fields
     * are not considered in this computation.
     *
     * @param time a timestamp relative to which the answer is given
     * @return the number of days between the given {@code time} and the previous enabled weekday
     * which is always between 1 and 7 inclusive; {@code -1} if no weekdays are enabled
     */
    public int getDistanceToPreviousDay(Calendar time) {
        int calendarDay = time.get(DAY_OF_WEEK);
        for (int count = 1; count <= 7; count++) {
            calendarDay--;
            if (calendarDay < SUNDAY) {
                calendarDay = SATURDAY;
            }
            if (isBitOn(calendarDay)) {
                return count;
            }
        }

        return -1;
    }

    /**
     * Note: only the day-of-week is read from the {@code time}. The time fields
     * are not considered in this computation.
     *
     * @param time a timestamp relative to which the answer is given
     * @return the number of days between the given {@code time} and the next enabled weekday which
     * is always between 0 and 6 inclusive; {@code -1} if no weekdays are enabled
     */
    public int getDistanceToNextDay(Calendar time) {
        int calendarDay = time.get(DAY_OF_WEEK);
        for (int count = 0; count < 7; count++) {
            if (isBitOn(calendarDay)) {
                return count;
            }

            calendarDay++;
            if (calendarDay > SATURDAY) {
                calendarDay = SUNDAY;
            }
        }

        return -1;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        final Weekdays weekdays = (Weekdays) o;
        return mBits == weekdays.mBits;
    }

    @NonNull
    @Override
    public String toString() {
        final StringBuilder builder = new StringBuilder(19);
        builder.append("[");
        if (isBitOn(MONDAY)) {
            builder.append(builder.length() > 1 ? " M" : "M");
        }
        if (isBitOn(TUESDAY)) {
            builder.append(builder.length() > 1 ? " T" : "T");
        }
        if (isBitOn(WEDNESDAY)) {
            builder.append(builder.length() > 1 ? " W" : "W");
        }
        if (isBitOn(THURSDAY)) {
            builder.append(builder.length() > 1 ? " Th" : "Th");
        }
        if (isBitOn(FRIDAY)) {
            builder.append(builder.length() > 1 ? " F" : "F");
        }
        if (isBitOn(SATURDAY)) {
            builder.append(builder.length() > 1 ? " Sa" : "Sa");
        }
        if (isBitOn(SUNDAY)) {
            builder.append(builder.length() > 1 ? " Su" : "Su");
        }
        builder.append("]");
        return builder.toString();
    }

    /**
     * @param context for accessing resources
     * @param order   the order in which to present the weekdays
     * @return the enabled weekdays in the given {@code order}
     */
    public String toString(Context context, Order order) {
        return toString(context, order, false);
    }

    /**
     * @param context for accessing resources
     * @param order   the order in which to present the weekdays
     * @return the enabled weekdays in the given {@code order} in a manner that
     * is most appropriate for talk-back
     */
    public String toAccessibilityString(Context context, Order order) {
        return toString(context, order, true);
    }

    @VisibleForTesting
    int getCount() {
        int count = 0;
        for (int calendarDay = SUNDAY; calendarDay <= SATURDAY; calendarDay++) {
            if (isBitOn(calendarDay)) {
                count++;
            }
        }
        return count;
    }

    /**
     * @param context        for accessing resources
     * @param order          the order in which to present the weekdays
     * @param forceLongNames if {@code true} the un-abbreviated weekdays are used
     * @return the enabled weekdays in the given {@code order}
     */
    private String toString(Context context, Order order, boolean forceLongNames) {
        if (!isRepeating()) {
            return "";
        }

        if (mBits == ALL_DAYS) {
            return context.getString(R.string.every_day);
        }

        final boolean longNames = forceLongNames || getCount() <= 1;
        final DateFormatSymbols dfs = new DateFormatSymbols();
        final String[] weekdays = longNames ? dfs.getWeekdays() : dfs.getShortWeekdays();

        final String separator = context.getString(R.string.day_concat);

        final StringBuilder builder = new StringBuilder(40);
        for (int calendarDay : order.getCalendarDays()) {
            if (isBitOn(calendarDay)) {
                if (!TextUtils.isEmpty(builder)) {
                    builder.append(separator);
                }
                builder.append(weekdays[calendarDay]);
            }
        }
        return builder.toString();
    }

    /**
     * Returns a styled representation of the repeating days of the week.
     * <p>
     * This method builds a {@link SpannableStringBuilder} containing the names of the days
     * on which the alarm is set to repeat. The day corresponding to the next scheduled alarm
     * (if provided) will be colored and displayed in bold.
     * </p>
     *
     * @param context        the context used to access resources
     * @param order          the preferred order of weekdays (e.g., starting on Monday or Sunday)
     * @param forceLongNames whether to force the use of full weekday names
     * @param nextAlarmDay   the calendar day (e.g., {@link Calendar#MONDAY}) of the next alarm;
     *                       if matched, that day will be styled in bold
     * @return a {@link CharSequence} with the formatted and styled weekday names
     */
    public CharSequence toStyledString(Context context, Order order, boolean forceLongNames, int nextAlarmDay) {
        if (!isRepeating()) {
            return "";
        }

        final boolean longNames = forceLongNames || getCount() <= 1;
        final DateFormatSymbols dfs = new DateFormatSymbols();
        final String[] weekdays = longNames ? dfs.getWeekdays() : dfs.getShortWeekdays();
        final String separator = context.getString(R.string.day_concat);

        final SpannableStringBuilder builder = new SpannableStringBuilder();
        for (int calendarDay : order.getCalendarDays()) {
            if (isBitOn(calendarDay)) {
                if (!TextUtils.isEmpty(builder)) {
                    builder.append(separator);
                }

                String dayName = weekdays[calendarDay];
                int start = builder.length();
                builder.append(dayName);
                int end = builder.length();

                if (calendarDay == nextAlarmDay) {
                    int primaryColor = MaterialColors.getColor(context, androidx.appcompat.R.attr.colorPrimary, Color.BLACK);
                    builder.setSpan(new ForegroundColorSpan(primaryColor), start, end, Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
                    builder.setSpan(new StyleSpan(Typeface.BOLD), start, end, Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
                }
            }
        }
        return builder;
    }

    /**
     * The preferred starting day of the week can differ by locale. This enumerated value is used to
     * describe the preferred ordering.
     */
    public enum Order {
        SAT_TO_FRI(SATURDAY, SUNDAY, MONDAY, TUESDAY, WEDNESDAY, THURSDAY, FRIDAY),
        SUN_TO_SAT(SUNDAY, MONDAY, TUESDAY, WEDNESDAY, THURSDAY, FRIDAY, SATURDAY),
        MON_TO_SUN(MONDAY, TUESDAY, WEDNESDAY, THURSDAY, FRIDAY, SATURDAY, SUNDAY);

        private final List<Integer> mCalendarDays;

        Order(Integer... calendarDays) {
            mCalendarDays = Arrays.asList(calendarDays);
        }

        public List<Integer> getCalendarDays() {
            return mCalendarDays;
        }
    }
}
