package com.baldo.bob.ui.compose.settings

import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.selection.selectable
import androidx.compose.foundation.selection.selectableGroup
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.automirrored.filled.ArrowForward
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.RadioButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.platform.LocalUriHandler
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.semantics.Role
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import com.baldo.bob.R
import com.baldo.bob.ui.viewModel.BobViewModel
import com.baldo.bob.ui.viewModel.UserLanguageUiState
import com.baldo.bob.utils.langs

@Composable
fun LanguageScreen(
    navigateBack: () -> Unit,
    bobViewModel: BobViewModel,
    userLanguageUiState: UserLanguageUiState,
) {

    val currentLanguage = userLanguageUiState.userLanguage

    Column(
        Modifier
            .fillMaxSize()
            .padding(16.dp)

    ) {
        val uriHandler = LocalUriHandler.current

        Icon(
            Icons.AutoMirrored.Filled.ArrowBack,
            contentDescription = "back",
            modifier = Modifier
                .padding(bottom = 32.dp)
                .clickable { navigateBack() }
        )


        Text(
            stringResource(id = R.string.language),
            fontWeight = FontWeight.Bold,
            style = MaterialTheme.typography.headlineLarge,
            color = MaterialTheme.colorScheme.primary,
            modifier = Modifier.padding(bottom = 32.dp)
        )

        Box(
            modifier = Modifier
                .padding(16.dp, end = 16.dp, bottom = 32.dp)
                .clip(RoundedCornerShape(64.dp))
                .background(color = MaterialTheme.colorScheme.primary)
                .clickable { uriHandler.openUri("https://poeditor.com/join/project/c4b0FCs3UU") }
        ) {
            Row(
                Modifier
                    .fillMaxWidth()
                    .padding(vertical = 24.dp),
                horizontalArrangement = Arrangement.SpaceAround
            ) {
                Text(
                    color = MaterialTheme.colorScheme.onPrimary,
                    style = MaterialTheme.typography.titleLarge,
                    text =
                        stringResource(
                            R.string.help_translate,

                            )
                )
                Icon(
                    imageVector = Icons.AutoMirrored.Filled.ArrowForward,
                    contentDescription = null,
                    tint = MaterialTheme.colorScheme.onPrimary
                )
            }
        }

        LanguageSelector(currentLanguage) { selected:Int -> bobViewModel.updateUserLanguage(selected) }

    }
}

@Composable
fun LanguageSelector(selectedPosition: Int, onLanguageSelected: (Int) -> Unit) {

    val scrollState = rememberScrollState()

    Column(
        Modifier
            .selectableGroup()
            .verticalScroll(scrollState)
    ) {
        langs.forEachIndexed { index, it ->
            Row(
                Modifier
                    .fillMaxWidth()
                    .height(56.dp)
                    .selectable(
                        selected = (index == selectedPosition),
                        onClick = { onLanguageSelected(index) },
                        role = Role.RadioButton
                    )
                    .padding(horizontal = 16.dp),
                verticalAlignment = Alignment.CenterVertically
            ) {

                RadioButton(
                    selected = (index == selectedPosition),
                    onClick = null // null recommended for accessibility with screenreaders
                )
                Text(
                    text = it.displayName,
                    style = MaterialTheme.typography.bodyLarge,
                    modifier = Modifier.padding(start = 16.dp)
                )
            }
        }
    }
}