package com.baldo.bob.ui.compose.contractions

import android.annotation.SuppressLint
import android.os.Build
import android.util.Log
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.wrapContentSize
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material.icons.filled.HourglassBottom
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.VerticalDivider
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableLongStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.lifecycle.viewmodel.compose.viewModel
import com.baldo.bob.R
import com.baldo.bob.data.Contraction
import com.baldo.bob.ui.AppViewModelProvider
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import java.time.Duration
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.format.DateTimeFormatter
import java.time.format.FormatStyle
import java.time.temporal.ChronoUnit
import java.util.Locale

fun formatedHour(date: LocalDateTime): String {
    return DateTimeFormatter.ofLocalizedTime(FormatStyle.SHORT).withLocale(Locale.FRANCE)
        .format(date)
}

fun contractionDuration(contraction: Contraction): String {
    val seconds = contraction.contractionDuration.seconds
    return secondToTime(seconds)
}

fun secondToTime(seconds: Long): String {
    val min = (seconds / 60) % 60
    val sec = seconds % 60
    return if (sec < 10) {
        "$min:0$sec"
    } else {
        "$min:$sec"
    }
}

@SuppressLint("UnusedMaterial3ScaffoldPaddingParameter")
@Composable
@Preview(showBackground = true)
fun ContractionScreen(
    viewModel: ContractionsViewModel = viewModel(factory = AppViewModelProvider.Factory)
) {
    val contractionUiState by viewModel.displayContractionUiState.collectAsState()
    val coroutineScope = rememberCoroutineScope()
    val recordingContraction = remember { mutableStateOf(false) }
    var contractionStartTime by remember { mutableStateOf(LocalDateTime.now()) }
    var elapsedTime by remember { mutableLongStateOf(0L) } // État pour la durée écoulée
    var timeBetweenContractions by remember { mutableLongStateOf(0L) } // Durée entre les contractions
    var lastContractionEndTime by remember { mutableStateOf<LocalDateTime?>(null) }
    val onValueChange = viewModel::updateUiState

    fun onAddButtonClicked() {
        if (recordingContraction.value) {
            onValueChange(
                viewModel.contractionUiState.copy(
                    startTime = contractionStartTime, contractionDuration = Duration.between(
                        contractionStartTime, LocalDateTime.now()
                    )
                )
            )
            coroutineScope.launch {
                try {
                    viewModel.saveContraction()
                } catch (e: Exception) {
                    Log.e("ContractionScreen", "Failed to save contraction", e)
                }
            }
            recordingContraction.value = false
            lastContractionEndTime = LocalDateTime.now() // Enregistrer la fin de la contraction
            elapsedTime = 0 // Réinitialiser le temps écoulé
            coroutineScope.launch {
                try {
                    while (!recordingContraction.value) {
                        delay(1000L) // Mettre à jour toutes les secondes
                        timeBetweenContractions =
                            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
                                Duration.between(lastContractionEndTime, LocalDateTime.now())
                                    .toSeconds()
                            } else {
                                ChronoUnit.SECONDS.between(
                                    lastContractionEndTime, LocalDateTime.now()
                                )
                            }
                    }
                } finally {
                    timeBetweenContractions = 0 // Réinitialiser le temps entre les contractions
                }
            }
        } else {
            contractionStartTime = LocalDateTime.now()
            recordingContraction.value = true
            coroutineScope.launch {
                try {
                    while (recordingContraction.value) {
                        delay(1000L) // Mettre à jour toutes les secondes
                        elapsedTime = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
                            Duration.between(contractionStartTime, LocalDateTime.now()).toSeconds()
                        } else {
                            ChronoUnit.SECONDS.between(contractionStartTime, LocalDateTime.now())
                        }
                    }
                } finally {
                    elapsedTime = 0 // Réinitialiser le temps écoulé
                }
            }
        }
    }

    Scaffold(
        floatingActionButton = {
            Column(horizontalAlignment = Alignment.CenterHorizontally) {
                if (recordingContraction.value) {
                    Box(
                        modifier = Modifier
                            .padding(bottom = 16.dp)
                            .background(
                                color = MaterialTheme.colorScheme.primaryContainer,
                                shape = RoundedCornerShape(8.dp)
                            )
                    ) {
                        Text(
                            text = secondToTime(elapsedTime),
                            style = MaterialTheme.typography.bodyLarge,
                            modifier = Modifier.padding(8.dp)
                        )
                    }
                } else if (lastContractionEndTime != null) {
                    Box(
                        modifier = Modifier
                            .padding(bottom = 16.dp)
                            .background(
                                color = MaterialTheme.colorScheme.tertiaryContainer,
                                shape = RoundedCornerShape(8.dp)
                            )
                    ) {
                        Text(
                            text = secondToTime(timeBetweenContractions),
                            style = MaterialTheme.typography.bodyLarge,
                            modifier = Modifier.padding(8.dp)
                        )
                    }
                }
                FloatingActionButton(
                    onClick = { onAddButtonClicked() },
                    containerColor = if (recordingContraction.value) {
                        MaterialTheme.colorScheme.error
                    } else {
                        MaterialTheme.colorScheme.primaryContainer
                    },
                ) {
                    if (recordingContraction.value) {
                        Icon(
                            imageVector = Icons.Filled.HourglassBottom,
                            contentDescription = "End contraction"
                        )
                    } else {
                        Icon(imageVector = Icons.Filled.Add, contentDescription = "Add contraction")
                    }
                }
            }
        },
    ) {
        if (contractionUiState.contractionList.isNotEmpty()) {
            ContractionBody(
                sortedContraction = contractionUiState.contractionList.sortedByDescending { it.id },
                contractionUiState.contractionList, contractionsViewModel = viewModel
            )
        } else {
            Column(
                Modifier
                    .fillMaxSize()
                    .padding(32.dp)
            ) {
                Text(text = stringResource(R.string.press_button_contraction))
            }
        }
    }
}

@Composable
fun ContractionBody(sortedContraction: List<Contraction>, contractions: List<Contraction>, contractionsViewModel: ContractionsViewModel) {
    val openAlertDialog = remember { mutableStateOf(false) }
    val coroutineScope = rememberCoroutineScope()
    fun onConfirmation() {
        coroutineScope.launch {
            contractionsViewModel.deleteAllContractions()
        }
    }
    Column(
        modifier = Modifier.fillMaxSize()
    ) {
        Row(
            modifier = Modifier
                .fillMaxWidth()
                .padding(vertical = 24.dp),
        ) {
            Row(modifier = Modifier.fillMaxWidth(.50f), horizontalArrangement = Arrangement.End) {
                Text(text = stringResource(R.string.Start))
                Spacer(modifier = Modifier.width(16.dp))
                Text(text = stringResource(R.string.Duration))
                Spacer(modifier = Modifier.width(34.dp))
            }
            Spacer(modifier = Modifier.fillMaxWidth(.14f))
            Row {
                Text(
                    text = stringResource(R.string.Frequency),
                    color = MaterialTheme.colorScheme.secondary,
                )
            }
            Spacer(modifier = Modifier.weight(1f))
            Icon(
                Icons.Default.Delete,
                contentDescription = "delete contractions",
                modifier = Modifier
                    .padding(end = 16.dp)
                    .clickable {openAlertDialog.value = true})
        }
        when {
            openAlertDialog.value -> AlertDialog(
                icon = Icons.Default.Delete,
                dialogTitle = stringResource(R.string.delete_contractions_data),
                dialogText = stringResource(R.string.delete_contractions_data_message),
                onDismissRequest = {
                    openAlertDialog.value = false
                },
                onConfirmation = {
                    openAlertDialog.value = false
                    onConfirmation()
                })
        }

        LazyColumn(
            modifier = Modifier.fillMaxSize()
        ) {
            itemsIndexed(sortedContraction) { index, contraction ->
                var timeBetweenContractionDuration: Long?
                if (index > 0) {
                    timeBetweenContractionDuration = ChronoUnit.SECONDS.between(
                        contraction.startTime,
                        sortedContraction[index - 1].startTime - sortedContraction[index - 1].contractionDuration
                    )
                    // Vérifiez si le temps entre les contractions dépasse une heure
                    if (timeBetweenContractionDuration >= 3600) {
                        // Afficher une ligne verticale
                        Box(
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(8.dp),
                            contentAlignment = Alignment.Center
                        ) {
                            VerticalDivider(
                                color = MaterialTheme.colorScheme.primary,
                                modifier = Modifier
                                    .fillMaxWidth(0.01f)
                                    .height(48.dp)
                                    .padding(vertical = 8.dp)
                            )
                        }
                    } else {
                        TimeBetweenContractions(
                            timeBetweenContractionDuration = timeBetweenContractionDuration
                        )
                    }
                }
                ContractionRow(
                    contraction,
                    contractionNumber = contractions.indexOf(contraction) + 1,
                    isLastContraction = index == sortedContraction.lastIndex,
                    isFirstContraction = index == 0,
                    previousContractionDay = if (index > 0) sortedContraction[index - 1].startTime.toLocalDate() else null
                )
            }
        }
    }
}

@Composable
fun ContractionRow(
    contraction: Contraction,
    contractionNumber: Int,
    isLastContraction: Boolean,
    isFirstContraction: Boolean,
    previousContractionDay: LocalDate?,
) {
    val currentContractionDay = contraction.startTime.toLocalDate()
    val showDay =
        isFirstContraction || isLastContraction || (previousContractionDay != null && previousContractionDay != currentContractionDay)

    Row(
        modifier = Modifier
            .fillMaxWidth(.55f)
            .padding(vertical = 6.dp),
        verticalAlignment = Alignment.CenterVertically,
        horizontalArrangement = Arrangement.End
    ) {
        if (showDay) {
            Text(
                text = contraction.startTime.format(DateTimeFormatter.ofPattern("dd/MM")),
                color = Color.Gray,
                fontSize = 10.sp,
                modifier = Modifier.padding(end = 8.dp)
            )
        }
        Text(
            text = formatedHour(contraction.startTime), color = Color.Gray, fontSize = 10.sp
        )
        Spacer(modifier = Modifier.width(16.dp))
        Text(
            text = contractionDuration(contraction),
            fontStyle = FontStyle.Italic,
            fontWeight = FontWeight.Medium
        )
        Spacer(modifier = Modifier.width(24.dp))
        TextWithCircleBackground(contractionNumber.toString())
    }
}

@Composable
fun TimeBetweenContractions(timeBetweenContractionDuration: Long) {
    Row(
        Modifier.fillMaxWidth()
    ) {
        Spacer(modifier = Modifier.fillMaxWidth(.6f))
        Text(
            secondToTime(timeBetweenContractionDuration),
            color = MaterialTheme.colorScheme.tertiary,
            fontWeight = FontWeight.ExtraBold,
        )
    }
}

@Composable
fun TextWithCircleBackground(text: String) {
    Box(
        modifier = Modifier
            .size(42.dp) // Taille du cercle
            .clip(CircleShape) // Découpe en forme de cercle
            .background(MaterialTheme.colorScheme.primaryContainer) // Couleur de fond du cercle
            .wrapContentSize(Alignment.Center), // Centre le contenu
        contentAlignment = Alignment.Center // Centre le texte
    ) {
        Text(
            text = text, fontSize = 16.sp // Taille du texte
        )
    }
}

@Composable
fun AlertDialog(
    onDismissRequest: () -> Unit,
    onConfirmation: () -> Unit,
    dialogTitle: String,
    dialogText: String,
    icon: ImageVector,
) {
    AlertDialog(icon = { Icon(icon, contentDescription = "delete") }, title = {
        Text(text = dialogTitle)
    }, text = {
        Text(text = dialogText)
    }, onDismissRequest = {
        onDismissRequest()
    }, confirmButton = {
        TextButton(
            onClick = {
                onConfirmation()
            }) {
            Text(stringResource(R.string.validate))
        }
    }, dismissButton = {
        TextButton(
            onClick = {
                onDismissRequest()
            }) {
            Text(stringResource(R.string.cancel))
        }
    })
}