package com.anysoftkeyboard.janus.app.repository

import com.anysoftkeyboard.janus.app.di.LangWikipediaFactory
import com.anysoftkeyboard.janus.app.util.StringProvider
import com.anysoftkeyboard.janus.database.dao.TranslationDao
import com.anysoftkeyboard.janus.database.entities.Translation
import javax.inject.Inject
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.map

class FakeTranslationRepository
@Inject
constructor(
    private val translationDao: TranslationDao,
    private val wikipediaApi: LangWikipediaFactory,
    private val stringProvider: StringProvider
) : TranslationRepository(translationDao, wikipediaApi, stringProvider) {

  private val _history = MutableStateFlow(emptyList<Translation>())
  private val _bookmarks = MutableStateFlow(emptyList<Translation>())
  var nextSearchResults: List<OptionalSourceTerm> = emptyList()
  var nextTranslations: List<Translation> = emptyList()
  var searchException: Exception? = null
  var fetchException: Exception? = null

  override fun getHistory(): Flow<List<Translation>> = _history.asStateFlow()

  override fun searchHistory(query: String): Flow<List<Translation>> {
    return _history.map { history ->
      history.filter { translation ->
        translation.sourceWord.contains(query, ignoreCase = true) ||
            translation.translatedWord.contains(query, ignoreCase = true)
      }
    }
  }

  override fun getBookmarks(): Flow<List<Translation>> = _bookmarks.asStateFlow()

  override suspend fun searchArticles(sourceLang: String, term: String): List<OptionalSourceTerm> {
    searchException?.let { throw it }
    return nextSearchResults
  }

  override suspend fun fetchTranslations(
      searchPage: OptionalSourceTerm,
      sourceLang: String,
      targetLang: String
  ): List<Translation> {
    fetchException?.let { throw it }
    return nextTranslations
  }

  fun setHistory(history: List<Translation>) {
    _history.value = history
  }

  fun setBookmarks(bookmarks: List<Translation>) {
    _bookmarks.value = bookmarks
  }

  override suspend fun deleteTranslation(id: Int) {
    _history.value = _history.value.filter { it.id != id }
  }

  override suspend fun restoreTranslation(translation: Translation) {
    _history.value = _history.value + translation
  }
}
