import 'dart:async';

import 'package:flutter/material.dart';
import 'package:flutter_gemma/core/chat.dart';
import 'package:flutter_gemma/flutter_gemma.dart';
import 'package:flutter_gemma_example/chat_message.dart';
import 'package:flutter_gemma_example/services/gemma_service.dart';
import 'package:flutter_gemma_example/thinking_widget.dart';

class GemmaInputField extends StatefulWidget {
  const GemmaInputField({
    super.key,
    required this.messages,
    required this.streamHandler,
    required this.errorHandler,
    this.chat,
    this.onThinkingCompleted,
    this.isProcessing = false,
  });

  final InferenceChat? chat;
  final List<Message> messages;
  final ValueChanged<ModelResponse> streamHandler; // Returns ModelResponse (tokens or functions)
  final ValueChanged<String> errorHandler;
  final ValueChanged<String>? onThinkingCompleted; // Callback for completed thinking content
  final bool isProcessing; // Global processing state from ChatScreen

  @override
  GemmaInputFieldState createState() => GemmaInputFieldState();
}

class GemmaInputFieldState extends State<GemmaInputField> {
  GemmaLocalService? _gemma;
  var _message = const Message(text: '', isUser: false);
  bool _processing = false;
  StreamSubscription<ModelResponse>? _streamSubscription;
  FunctionCallResponse? _pendingFunctionCall; // Store function to send in onDone
  String _thinkingContent = ''; // Accumulate thinking content
  bool _isThinkingExpanded = false; // Thinking block expansion state
  bool _thinkingCompleted = false; // Thinking completed
  ThinkingResponse? _completedThinking; // Store completed thinking

  @override
  void initState() {
    super.initState();
    _gemma = GemmaLocalService(widget.chat!);
    _processMessages();
  }

  @override
  void dispose() {
    debugPrint('GemmaInputField: dispose() called');
    _streamSubscription?.cancel();
    debugPrint('GemmaInputField: StreamSubscription cancelled');
    super.dispose();
  }

  void _processMessages() async {
    if (_processing) return;
    
    // DEBUG: Track what we're processing
    final isAfterFunction = widget.messages.isNotEmpty && !widget.messages.last.isUser;
    debugPrint('🔵 GemmaInputField: Starting processing - isAfterFunction: $isAfterFunction');
    
    setState(() {
      _processing = true;
      _message = const Message(text: '', isUser: false);
      _thinkingContent = '';
      _thinkingCompleted = false;
      _completedThinking = null;
      _pendingFunctionCall = null;
    });

    try {
      final responseStream = await _gemma?.processMessage(widget.messages.last);
      
      if (responseStream != null) {
        _streamSubscription = responseStream.listen(
          (response) {
            if (mounted) {
              setState(() {
                if (response is String) {
                  _message = Message(text: '${_message.text}$response', isUser: false);
                } else if (response is TextResponse) {
                  _message = Message(text: '${_message.text}${response.token}', isUser: false);
                  // DEBUG: Track text accumulation
                  debugPrint('📝 GemmaInputField: Text accumulated: "${response.token}" -> total: "${_message.text}"');
                } else if (response is ThinkingResponse) {
                  _thinkingContent += response.content;
                } else if (response is FunctionCallResponse) {
                  debugPrint('🔧 GemmaInputField: Function call received: ${response.name}');
                  _pendingFunctionCall = response;
                }
              });
            }
          },
          onError: (error) {
            if (mounted) {
              if (_pendingFunctionCall != null) {
                widget.streamHandler(_pendingFunctionCall!);
              } else {
                final text = _message.text.isNotEmpty ? _message.text : '...';
                widget.streamHandler(TextResponse(text));
              }
              widget.errorHandler(error.toString());
              setState(() {
                _processing = false;
              });
            }
          },
          onDone: () {
            debugPrint('🏁 GemmaInputField: Stream completed');
            if (mounted) {
              // Handle thinking completion
              if (_thinkingContent.isNotEmpty) {
                setState(() {
                  _thinkingCompleted = true;
                  _completedThinking = ThinkingResponse(_thinkingContent);
                });
                widget.onThinkingCompleted?.call(_thinkingContent);
              }
              
              if (_pendingFunctionCall != null) {
                debugPrint('🔧 GemmaInputField: Sending function call: ${_pendingFunctionCall!.name}');
                widget.streamHandler(_pendingFunctionCall!);
              } else {
                final text = _message.text.isNotEmpty ? _message.text : '...';
                // DEBUG: Track what we're sending to ChatScreen
                debugPrint('📤 GemmaInputField: Sending final text: "$text" (length: ${text.length})');
                widget.streamHandler(TextResponse(text));
              }
              setState(() {
                _processing = false;
              });
            }
          },
        );
        debugPrint('GemmaInputField: StreamSubscription created and listening');
      } else {
        debugPrint('GemmaInputField: responseStream is null!');
        if (mounted) {
          setState(() {
            _processing = false;
          });
        }
      }
    } catch (e) {
      debugPrint('GemmaInputField: Exception caught: $e');
      if (mounted) {
        widget.errorHandler(e.toString());
        setState(() {
          _processing = false;
        });
      }
    }
  }

  @override
  Widget build(BuildContext context) {
    // Determine whether to show thinking (only if last message is from user)
    final shouldShowThinking = widget.messages.isNotEmpty && 
                             widget.messages.last.isUser &&
                             (_thinkingContent.isNotEmpty || _completedThinking != null);
    
    // Determine which message to display
    // If processing after function (not from user) - show empty for loading
    final displayMessage = widget.isProcessing && 
                          (widget.messages.isEmpty || !widget.messages.last.isUser)
        ? const Message(text: '', isUser: false) // Force loading indicator
        : _message; // Regular accumulated message
    
    
    return SingleChildScrollView(
      child: Column(
        children: [
          // Show thinking block only if shouldShowThinking
          if (shouldShowThinking) ...[
            if (_thinkingCompleted && _completedThinking != null)
              // Completed thinking block
              ThinkingWidget(
                thinking: _completedThinking!,
                isExpanded: _isThinkingExpanded,
                onToggle: () {
                  setState(() {
                    _isThinkingExpanded = !_isThinkingExpanded;
                  });
                },
              )
            else
              // Thinking in progress
              StreamingThinkingWidget(
                content: _thinkingContent,
                isExpanded: _isThinkingExpanded,
                onToggle: () {
                  setState(() {
                    _isThinkingExpanded = !_isThinkingExpanded;
                  });
                },
              ),
          ],
          // Main message with correct content
          ChatMessageWidget(message: displayMessage),
        ],
      ),
    );
  }
}
