// ignore_for_file: exhaustive_cases

import 'dart:io';

import 'package:globbing/glob_filter.dart';
import 'package:globbing/glob_parser.dart';
import 'package:path/path.dart' as pathos;

import 'file_list.dart';
import 'file_path.dart';

// ignore: avoid_classes_with_only_static_members
class FileUtils {
  static final bool _isWindows = Platform.isWindows;

  // /// Removes any leading directory components from [name].
  // ///
  // /// If [suffix] is specified and it is identical to the end of [name], it is
  // /// removed from [name] as well.
  // ///
  // /// If [name] is null returns null.
  // static String basename(String name, {String suffix}) {
  //   if (name == null) {
  //     return null;
  //   }

  //   if (name.isEmpty) {
  //     return '';
  //   }

  //   final segments = pathos.split(name);
  //   if (pathos.isAbsolute(name)) {
  //     if (segments.length == 1) {
  //       return '';
  //     }
  //   }

  //   var result = segments.last;
  //   if (suffix.isNotEmpty) {
  //     final index = result.lastIndexOf(suffix);
  //     if (index != -1) {
  //       result = result.substring(0, index);
  //     }
  //   }

  //   return result;
  // }

  /// Changes the current directory to [name]. Returns true if the operation was
  /// successful; otherwise false.
  static bool chdir(String name) {
    if (name.isEmpty) {
      return false;
    }

    name = FilePath.expand(name);
    final directory = Directory(name);
    if (!directory.existsSync()) {
      return false;
    }

    try {
      Directory.current = directory;
      // ignore: avoid_catches_without_on_clauses
    } catch (e) {
      return false;
    }

    return true;
  }

  /// Returns true if directory is empty; otherwise false;
  static bool dirempty(String name) {
    name = FilePath.expand(name);
    final directory = Directory(name);
    if (!directory.existsSync()) {
      return false;
    }

    return directory.listSync().isEmpty;
  }

  /// Returns a list of files from which will be removed elements
  /// that match glob
  /// [pattern].
  ///
  /// Parameters:
  ///  [files]
  ///   List of file paths.
  ///  [pattern]
  ///   Pattern of glob filter.
  ///  [added]
  ///   Function that is called whenever an item is added.
  ///  [caseSensitive]
  ///   True, if the pattern is case sensitive; otherwise false.
  ///  [removed]
  ///   Function that is called whenever an item is removed.
  static List<String> exclude(List<String> files, String pattern,
      {void Function(String path)? added,
      bool? caseSensitive,
      void Function(String path)? removed}) {
    pattern = FilePath.expand(pattern);
    if (!pathos.isAbsolute(pattern)) {
      pattern = '${getcwd()}/$pattern';
    }

    bool isDirectory(String path) => Directory(path).existsSync();

    final filter = GlobFilter(pattern,
        caseSensitive: caseSensitive,
        isDirectory: isDirectory,
        isWindows: _isWindows);

    return filter.exclude(files, added: added, removed: removed);
  }

  /// Returns the full name of the path if possible.
  ///
  /// Resolves the following segments:
  /// - Segments '.' indicating the current directory
  /// - Segments '..' indicating the parent directory
  /// - Leading '~' character indicating the home directory
  /// - Environment variables in IEEE Std 1003.1-2001 format, eg. $HOME/dart-sdk
  ///
  /// Useful when you get path name in a format incompatible with POSIX, and
  /// intend to use it as part of the wildcard patterns.
  ///
  /// Do not use this method directly on wildcard patterns because it can deform
  /// the patterns.
  static String fullpath(String name) {
    if (name.startsWith('..')) {
      final path = Directory.current.parent.path;
      if (name == '..') {
        name = path;
      } else if (name.startsWith('../')) {
        name = pathos.join(path, name.substring(3));
        name = pathos.normalize(name);
      } else {
        name = pathos.normalize(name);
      }
    } else if (name.startsWith('.')) {
      final path = Directory.current.path;
      if (name == '.') {
        name = path;
      } else if (name.startsWith('./')) {
        name = pathos.join(path, name.substring(2));
        name = pathos.normalize(name);
      } else {
        name = pathos.normalize(name);
      }
    } else {
      name = pathos.normalize(name);
    }

    name = FilePath.expand(name);
    if (_isWindows) {
      name = name.replaceAll(r'\', '/');
    }

    return name;
  }

  /// Returns the path of the current directory.
  static String getcwd() {
    var path = Directory.current.path;
    if (_isWindows) {
      path = path.replaceAll(r'\', '/');
    }

    return path;
  }

  /// Returns a list of files which match the specified glob [pattern].
  ///
  /// Parameters:
  ///  [pattern]
  ///   Glob pattern of file list.
  ///  [caseSensitive]
  ///   True, if the pattern is case sensitive; otherwise false.
  ///  [notify]
  ///   Function that is called whenever an item is added.
  static List<String> glob(String pattern,
      {bool? caseSensitive, void Function(String path)? notify}) {
    pattern = FilePath.expand(pattern);
    Directory directory;
    if (pathos.isAbsolute(pattern)) {
      final parser = GlobParser();
      final node = parser.parse(pattern);
      final parts = <GlobNodeSegment>[];
      final nodes = node.nodes;
      final length = nodes.length;
      for (var i = 1; i < length; i++) {
        final element = node.nodes[i];
        final strict = element.strict ?? false;
        if (strict) {
          parts.add(element);
        } else {
          break;
        }
      }

      final path = (nodes.first.source ?? '') + parts.join('/');
      directory = Directory(path);
    } else {
      directory = Directory.current;
    }

    return FileList(directory, pattern,
        caseSensitive: caseSensitive, notify: notify);
  }

  /// Returns a list of paths from which will be removed elements that do not
  /// match glob pattern.
  ///
  /// Parameters:
  ///  [files]
  ///   List of file paths.
  ///  [pattern]
  ///   Pattern of glob filter.
  ///  [added]
  ///   Function that is called whenever an item is added.
  ///  [caseSensitive]
  ///   True, if the pattern is case sensitive; otherwise false.
  ///  [removed]
  ///   Function that is called whenever an item is removed.
  static List<String> include(List<String> files, String pattern,
      {void Function(String path)? added,
      bool? caseSensitive,
      void Function(String path)? removed}) {
    pattern = FilePath.expand(pattern);
    if (!pathos.isAbsolute(pattern)) {
      pattern = '${getcwd()}/$pattern';
    }

    bool isDirectory(String path) => Directory(path).existsSync();

    final filter = GlobFilter(pattern,
        caseSensitive: caseSensitive,
        isDirectory: isDirectory,
        isWindows: _isWindows);

    return filter.include(files, added: added, removed: removed);
  }

  /// Creates listed directories and returns true if the operation was
  /// successful; otherwise false.
  ///
  /// If listed directories exists returns false.
  ///
  /// If [recursive] is set to true creates all required subdirectories and
  /// returns true if not errors occured.
  static bool mkdir(List<String> names, {bool recursive = false}) {
    if (names.isEmpty) {
      return false;
    }

    var result = true;
    for (var name in names) {
      name = FilePath.expand(name);
      final directory = Directory(name);
      final exists = directory.existsSync();
      if (exists) {
        if (!recursive) {
          result = false;
        }
      } else {
        try {
          directory.createSync(recursive: recursive);
          // ignore: avoid_catches_without_on_clauses
        } catch (e) {
          result = false;
        }
      }
    }

    return result;
  }

  /// Moves files [files] to the directory [dir]. Returns true if the operation
  /// was successful; otherwise false.
  static bool move(List<String> files, String dir) {
    if (!testfile(dir, 'directory')) {
      return false;
    }

    var result = true;
    for (final file in files) {
      if (file.isEmpty) {
        result = false;
        continue;
      }

      final list = glob(file);
      if (list.isEmpty) {
        result = false;
        continue;
      }

      for (final name in list) {
        final basename = pathos.basename(name);
        if (basename.isEmpty) {
          result = false;
          continue;
        }

        final dest = pathos.join(dir, basename);
        if (!rename(name, dest)) {
          result = false;
        }
      }
    }

    return result;
  }

  /// Renames or moves [src] to [dest]. Returns true if the operation was
  /// successful; otherwise false.
  static bool rename(String src, String dest) {
    src = FilePath.expand(src);
    dest = FilePath.expand(dest);
    FileSystemEntity? entity;
    switch (FileStat.statSync(src).type) {
      case FileSystemEntityType.directory:
        entity = Directory(src);
        break;
      case FileSystemEntityType.file:
        entity = File(src);
        break;
      case FileSystemEntityType.link:
        entity = Link(src);
        break;
    }

    if (entity == null) {
      return false;
    }

    try {
      entity.renameSync(dest);
      // ignore: avoid_catches_without_on_clauses
    } catch (e) {
      return false;
    }

    return true;
  }

  /// Removes the [files] and returns true if the operation was successful;
  /// otherwise false.
  ///
  /// By default, it does not remove directories.
  ///
  /// If [directory] is set to true removes the directories if they are empty.
  ///
  /// If [force] is set to true ignores nonexistent files.
  ///
  /// If [recursive] is set to true remove the directories and their contents
  /// recursively.
  static bool rm(List<String> files,
      {bool directory = false, bool force = false, bool recursive = false}) {
    if (files.isEmpty) {
      return false;
    }

    var result = true;
    for (final file in files) {
      if (file.isEmpty) {
        if (!force) {
          result = false;
        }

        continue;
      }

      final list = glob(file);
      if (list.isEmpty) {
        if (!force) {
          result = false;
        }

        continue;
      }

      for (final name in list) {
        FileSystemEntity? entity;
        var isDirectory = false;
        if (testfile(name, 'link')) {
          entity = Link(name);
        } else if (testfile(name, 'file')) {
          entity = File(name);
        } else if (testfile(name, 'directory')) {
          entity = Directory(name);
          isDirectory = true;
        }

        if (entity == null) {
          if (!force) {
            result = false;
          }
        } else {
          if (isDirectory) {
            if (recursive) {
              try {
                entity.deleteSync(recursive: recursive);
                // ignore: avoid_catches_without_on_clauses
              } catch (e) {
                result = false;
              }
            } else if (directory) {
              result = rmdir([entity.path], parents: true);
            } else {
              result = false;
            }
          } else {
            try {
              entity.deleteSync();
              // ignore: avoid_catches_without_on_clauses
            } catch (e) {
              result = false;
            }
          }
        }
      }
    }

    return result;
  }

  /// Removes empty directories. Returns true if the operation was successful;
  /// otherwise false.
  static bool rmdir(List<String> names, {bool parents = false}) {
    if (names.isEmpty) {
      return false;
    }

    var result = true;
    for (final name in names) {
      if (name.isEmpty) {
        result = false;
        continue;
      }

      final list = glob(name);
      if (list.isEmpty) {
        result = false;
        continue;
      }

      for (final name in list) {
        if (testfile(name, 'file')) {
          result = false;
          continue;
        } else if (testfile(name, 'link')) {
          result = false;
          continue;
        } else if (!testfile(name, 'directory')) {
          result = false;
          continue;
        }

        if (dirempty(name)) {
          try {
            Directory(name).deleteSync();
            // ignore: avoid_catches_without_on_clauses
          } catch (e) {
            result = false;
          }
        } else {
          if (parents) {
            if (!canDelete(name)) {
              result = false;
            } else {
              try {
                Directory(name).deleteSync(recursive: true);
                // ignore: avoid_catches_without_on_clauses
              } catch (e) {
                result = false;
              }
            }
          } else {
            result = false;
          }
        }
      }
    }

    return result;
  }

  static bool canDelete(String name) {
    final directory = Directory(name);
    for (final entry in directory.listSync()) {
      if (entry is File) {
        return false;
      } else if (entry is Link) {
        return false;
      } else if (entry is Directory) {
        if (!canDelete(entry.path)) {
          return false;
        }
      } else {
        return false;
      }
    }

    return true;
  }

  /// Creates the symbolic [link] to the [target] and returns true if the
  /// operation was successful; otherwise false.
  ///
  /// If [target] does not exists returns false.
  ///
  /// IMPORTANT:
  /// On the Windows platform, this will only work with directories.
  static bool symlink(String target, String link) {
    target = FilePath.expand(target);
    link = FilePath.expand(link);
    if (_isWindows) {
      if (!testfile(target, 'directory')) {
        return false;
      }
    } else {
      if (!testfile(target, 'exists')) {
        return false;
      }
    }

    final symlink = Link(link);
    try {
      symlink.createSync(target);
      // ignore: avoid_catches_without_on_clauses
    } catch (e) {
      return false;
    }

    return true;
  }

  /// Performs specified test on [file] and returns true if success; otherwise
  /// returns false;
  ///
  /// Available test:
  /// directory:
  ///   [file] exists and is a directory.
  /// exists:
  ///   [file] exists.
  /// file:
  ///   [file] exists and is a regular file.
  /// link:
  ///   [file] exists and is a symbolic link.
  static bool testfile(String file, String test) {
    file = FilePath.expand(file);
    switch (test) {
      case 'directory':
        return Directory(file).existsSync();
      case 'exists':
        return FileStat.statSync(file).type != FileSystemEntityType.notFound;
      case 'file':
        return File(file).existsSync();
      case 'link':
        return Link(file).existsSync();
      default:
        return false;
    }
  }

  /// Changes the modification time of the specified [files]. Returns
  /// true if the
  /// operation was successful; otherwise false.
  ///
  /// If [create] is set to true creates files that do not exist, reports
  /// failure  if the files can not be created.
  ///
  /// If [create] is set to false do not creates files that do not exist and do
  /// not reports failure about files that do not exist.
  static bool touch(List<String> files, {bool create = true}) {
    if (files.isEmpty) {
      return false;
    }

    var result = true;
    for (var file in files) {
      if (file.isEmpty) {
        result = false;
        continue;
      }

      file = FilePath.expand(file);
      if (_isWindows) {
        if (!_touchOnWindows(file, create)) {
          result = false;
        }
      } else {
        if (!_touchOnPosix(file, create)) {
          result = false;
        }
      }
    }

    return result;
  }

  /// Returns true if [file] is newer than all [depends]; otherwise false.
  static bool uptodate(String file, [List<String>? depends]) {
    if (file.isEmpty) {
      return false;
    }

    file = FilePath.expand(file);
    final stat = FileStat.statSync(file);
    if (stat.type == FileSystemEntityType.notFound) {
      return false;
    }

    if (depends == null) {
      return true;
    }

    final date = stat.modified;
    for (final name in depends) {
      final stat = FileStat.statSync(name);
      if (stat.type == FileSystemEntityType.notFound) {
        return false;
      }

      if (date.compareTo(stat.modified) < 0) {
        return false;
      }
    }

    return true;
  }

  static int _shell(String command, List<String> arguments,
          {String? workingDirectory}) =>
      Process.runSync(command, arguments,
              runInShell: true, workingDirectory: workingDirectory)
          .exitCode;

  static bool _touchOnPosix(String name, bool create) {
    final arguments = <String>[name];
    if (!create) {
      arguments.add('-c');
    }

    return _shell('touch', arguments) == 0;
  }

  static bool _touchOnWindows(String name, bool create) {
    if (!testfile(name, 'file')) {
      if (!create) {
        return true;
      } else {
        final file = File(name);
        try {
          file.createSync();
          return true;
          // ignore: avoid_catches_without_on_clauses
        } catch (e) {
          if (create) {
            return false;
          } else {
            return true;
          }
        }
      }
    }

    final dirName = pathos.dirname(name);
    String workingDirectory;
    if (dirName.isNotEmpty) {
      name = pathos.basename(name);
      if (pathos.isAbsolute(dirName)) {
        workingDirectory = dirName;
      } else {
        workingDirectory = '${Directory.current.path}\\$dirName';
      }
    } else {
      workingDirectory = '.';
    }

    return _shell('copy', ['/b', name, '+', ',', ','],
            workingDirectory: workingDirectory) ==
        0;
  }
}
