package com.amirsteinbeck.taskmate

import android.content.Context
import android.graphics.Canvas
import android.graphics.Color
import android.os.Bundle
import android.view.View
import android.view.inputmethod.InputMethodManager
import androidx.activity.enableEdgeToEdge
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.app.AppCompatDelegate
import androidx.core.content.ContextCompat
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.widget.addTextChangedListener
import androidx.recyclerview.widget.ItemTouchHelper
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.amirsteinbeck.taskmate.com.amirsteinbeck.focusmate.FadeItemAnimator
import com.amirsteinbeck.taskmate.com.amirsteinbeck.focusmate.LocaleHelper
import com.amirsteinbeck.taskmate.com.amirsteinbeck.focusmate.SettingsHelper
import com.amirsteinbeck.taskmate.databinding.ActivityMainBinding
import com.google.android.material.bottomsheet.BottomSheetDialog
import com.google.android.material.color.MaterialColors
import com.google.android.material.snackbar.Snackbar
import com.google.android.material.textfield.TextInputEditText
import it.xabaras.android.recyclerview.swipedecorator.RecyclerViewSwipeDecorator
import java.util.Locale


class MainActivity : AppCompatActivity() {

    private lateinit var binding: ActivityMainBinding
    private lateinit var fullList: MutableList<Task>
    private lateinit var displayList: MutableList<Task>
    private lateinit var adapter: TaskAdapter

    fun updateLists() {
        val fullList = StorageHelper.loadTasks(this)
        val displayList = fullList.filter { !it.isArchived }.toMutableList()

        adapter.sortTasks(this)
        adapter.updateData(displayList)
    }

    override fun attachBaseContext(newBase: Context?) {
        val context = newBase?.let { LocaleHelper.applyLanguage(it) }
        super.attachBaseContext(context)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityMainBinding.inflate(layoutInflater)
        enableEdgeToEdge()
        setContentView(binding.root)

        ViewCompat.setOnApplyWindowInsetsListener(findViewById(R.id.main)) { v, insets ->
            val imeInsets = insets.getInsets(WindowInsetsCompat.Type.ime())
            v.setPadding(0, 0, 0, imeInsets.bottom)
            insets
        }

        AppCompatDelegate.setDefaultNightMode(
            if (SettingsHelper.isDarkMode(this))
                AppCompatDelegate.MODE_NIGHT_YES
            else
                AppCompatDelegate.MODE_NIGHT_NO
        )

        fullList = StorageHelper.loadTasks(this)
        displayList = fullList.filter { !it.isArchived }.toMutableList()

        fun updateEmptyView() {
            if (adapter.itemCount == 0) {
                binding.emptyTasksView.visibility = View.VISIBLE
                binding.recyclerView.visibility = View.GONE
            } else {
                binding.emptyTasksView.visibility = View.GONE
                binding.recyclerView.visibility = View.VISIBLE
            }
        }

        fun openTaskBottomSheet(task: Task? = null, position: Int = -1, isEdit: Boolean) {
            val dialog = BottomSheetDialog(this)
            val view = layoutInflater.inflate(R.layout.bottomsheet_edit_task, null)

            val titleInput = view.findViewById<TextInputEditText>(R.id.editTitle)
            val descInput = view.findViewById<TextInputEditText>(R.id.editDescription)
            val saveButton = view.findViewById<View>(R.id.saveButton)

            if (!isEdit) saveButton.isEnabled = false

            if (task == null) titleInput.setText("") else titleInput.setText(task.title)
            if (task == null) descInput.setText("") else descInput.setText(task.description)

            dialog.setOnShowListener {
                titleInput.requestFocus()
                titleInput.post {
                    val imm = getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager
                    imm.showSoftInput(titleInput, InputMethodManager.SHOW_IMPLICIT)
                }
            }

            titleInput.addTextChangedListener {
                saveButton.isEnabled = !it.isNullOrBlank()
            }

            saveButton.setOnClickListener {
                val newTitle = titleInput.text.toString().trim()
                    .replaceFirstChar { if (it.isLowerCase()) it.titlecase(Locale.getDefault()) else it.toString() }
                val newDesc = descInput.text.toString().trim()
                    .replaceFirstChar { if (it.isLowerCase()) it.titlecase(Locale.getDefault()) else it.toString() }

                val theTask = Task(
                    newTitle,
                    newDesc.ifEmpty { getString(R.string.newTask) },
                    false)

                if (isEdit) adapter.updateItem(position, theTask) else adapter.addItem(theTask)
                StorageHelper.saveTasks(this, displayList)
                binding.recyclerView.scrollToPosition(displayList.size - 1)


                dialog.dismiss()
                updateLists()
                fullList.add(theTask)
                updateEmptyView()
                adapter.sortTasks(this)
            }
            updateEmptyView()
            dialog.setContentView(view)
            dialog.show()
        }

        adapter = TaskAdapter(
            displayList,
            if (LocaleHelper.getLanguage(this) == "en") "ltr" else "rtl",
            { clickedTask, position ->
//                NavigationHelper.goToTaskDetails(this, clickedTask.title, clickedTask.description)

            },
            { clickedTask, position -> openTaskBottomSheet(clickedTask, position, true) }
            )

        adapter.sortTasks(this)

        val leftSwipeHandler = object : ItemTouchHelper.SimpleCallback(0, ItemTouchHelper.LEFT) {
            override fun onMove(
                recyclerView: RecyclerView,
                viewHolder: RecyclerView.ViewHolder,
                target: RecyclerView.ViewHolder
            ) = false

            override fun onChildDraw(
                c: Canvas,
                recyclerView: RecyclerView,
                viewHolder: RecyclerView.ViewHolder,
                dX: Float,
                dY: Float,
                actionState: Int,
                isCurrentlyActive: Boolean
            ) {
                RecyclerViewSwipeDecorator.Builder(
                    c, recyclerView, viewHolder, dX, dY, actionState, isCurrentlyActive
                ).addSwipeLeftBackgroundColor(MaterialColors.getColor(this@MainActivity,
                            com.google.android.material.R.attr.colorOnError,
                            Color.RED)
                )
                    .addSwipeLeftActionIcon(R.drawable.trash_bin_minimalistic_2_svgrepo_com_24)
                    .setSwipeLeftActionIconTint(MaterialColors.getColor(this@MainActivity,
                            com.google.android.material.R.attr.colorSurfaceInverse,
                            Color.WHITE)
                    )
                    .addSwipeLeftLabel(getString(R.string.delete))
                    .setSwipeLeftLabelColor(MaterialColors.getColor(this@MainActivity,
                        com.google.android.material.R.attr.colorSurfaceInverse,
                        Color.WHITE)
                    )
                    .create()
                    .decorate()
                super.onChildDraw(c, recyclerView, viewHolder, dX, dY, actionState, isCurrentlyActive)
                updateEmptyView()
            }

            override fun onSwiped(viewHolder: RecyclerView.ViewHolder, direction: Int) {
                val position = viewHolder.bindingAdapterPosition
                val removedTask = displayList[position]
                displayList.removeAt(position)
                adapter.notifyItemRemoved(position)
                val indexOfRemovedTaskInFullList = fullList.indexOfFirst { it.id == removedTask.id }
                if (indexOfRemovedTaskInFullList != -1) fullList.removeAt(indexOfRemovedTaskInFullList)
                StorageHelper.saveTasks(this@MainActivity, fullList)
                if (fullList.size > 1) adapter.sortTasks(this@MainActivity)
                updateEmptyView()

                Snackbar.make(binding.root, getString(R.string.removeSnackbarMessage, removedTask.title), Snackbar.LENGTH_LONG)
                    .setAction(getString(R.string.undo)) {
                        displayList.add(position, removedTask)
                        adapter.notifyItemInserted(position)
                        fullList.add(indexOfRemovedTaskInFullList, removedTask)
                        StorageHelper.saveTasks(this@MainActivity, fullList)
                        adapter.sortTasks(this@MainActivity)
                        updateEmptyView()

                    }.show()
            }
        }
        ItemTouchHelper(leftSwipeHandler).attachToRecyclerView(binding.recyclerView)

        val rightSwipeHandler = object: ItemTouchHelper.SimpleCallback(0, ItemTouchHelper.RIGHT) {
            override fun onMove(
                recyclerView: RecyclerView,
                viewHolder: RecyclerView.ViewHolder,
                target: RecyclerView.ViewHolder
            ) = false

            override fun onChildDraw(
                c: Canvas,
                recyclerView: RecyclerView,
                viewHolder: RecyclerView.ViewHolder,
                dX: Float,
                dY: Float,
                actionState: Int,
                isCurrentlyActive: Boolean
            ) {
                RecyclerViewSwipeDecorator.Builder(
                    c, recyclerView, viewHolder, dX, dY, actionState, isCurrentlyActive
                ).addSwipeRightBackgroundColor(
                    ContextCompat.getColor(this@MainActivity, R.color.teal)
                )
                    .addSwipeRightActionIcon(R.drawable.archive_check_svgrepo_com_24)
                    .setSwipeRightActionIconTint(MaterialColors.getColor(this@MainActivity,
                        com.google.android.material.R.attr.colorSurfaceInverse,
                        Color.WHITE)
                    )
                    .addSwipeRightLabel(getString(R.string.archive))
                    .setSwipeRightLabelColor(MaterialColors.getColor(this@MainActivity,
                        com.google.android.material.R.attr.colorSurfaceInverse,
                        Color.WHITE)
                    )
                    .create()
                    .decorate()
                super.onChildDraw(c, recyclerView, viewHolder, dX, dY, actionState, isCurrentlyActive)
                updateEmptyView()
            }

            override fun onSwiped(viewHolder: RecyclerView.ViewHolder, direction: Int) {
                val position = viewHolder.bindingAdapterPosition
                val archivedTask = displayList[position]
                archivedTask.isArchived = true

                val indexInFullList = fullList.indexOfFirst { it.id == archivedTask.id }
                if (indexInFullList != -1) {
                    fullList[indexInFullList] = archivedTask
                    StorageHelper.saveTasks(this@MainActivity, fullList)
                }

                displayList.removeAt(position)
                adapter.notifyItemRemoved(position)
                updateEmptyView()
                if (fullList.size > 1) adapter.sortTasks(this@MainActivity)

                Snackbar.make(binding.root, getString(R.string.archiveSnackbarMessage, archivedTask.title), Snackbar.LENGTH_LONG)
                    .setAction(getString(R.string.undo)) {
                        archivedTask.isArchived = false

                        if (indexInFullList != -1) {
                            fullList[indexInFullList] = archivedTask
                            StorageHelper.saveTasks(this@MainActivity, fullList)
                        }
                        displayList.add(position, archivedTask)
                        adapter.notifyItemInserted(position)
                        updateEmptyView()
                        adapter.sortTasks(this@MainActivity)
                    }.show()
            }
        }
        ItemTouchHelper(rightSwipeHandler).attachToRecyclerView(binding.recyclerView)

        binding.recyclerView.itemAnimator = FadeItemAnimator()
        binding.recyclerView.layoutManager = LinearLayoutManager(this)
        binding.recyclerView.adapter = adapter
        updateEmptyView()

        binding.submitButton.setOnClickListener {
            openTaskBottomSheet(isEdit = false)
        }

        binding.userInput.setOnClickListener {
            openTaskBottomSheet(isEdit = false)
        }

        binding.settingsButton.setOnClickListener {
            NavigationHelper.goToSettings(this)
        }

        binding.archivesButton.setOnClickListener {
            NavigationHelper.goToArchivedTasks(this)
        }

        binding.appInfoButton.setOnClickListener {
            NavigationHelper.goToCredentials(this)
        }

        AnimationHelper.applyPressAnimation(this, binding.submitButton)
        AnimationHelper.applyPressAnimation(this, binding.settingsButton)
        AnimationHelper.applyPressAnimation(this, binding.archivesButton)
        AnimationHelper.applyPressAnimation(this, binding.appInfoButton)
    }

    override fun onResume() {
        super.onResume()

        updateLists()
        adapter.sortTasks(this@MainActivity)
    }
}