"use strict";

import { View, StyleSheet } from 'react-native';
import { IconButton, Text, useTheme } from 'react-native-paper';
import { useHeaderTextColor } from '../shared/utils';
import Color from 'color';
import { getTranslation } from '../translations/utils';
import React, { useMemo } from 'react';
import { sharedStyles } from '../shared/styles';
import { jsx as _jsx, jsxs as _jsxs, Fragment as _Fragment } from "react/jsx-runtime";
function getLabel(locale, mode, configuredLabel) {
  if (configuredLabel) {
    return configuredLabel;
  }
  if (mode === 'range') {
    return getTranslation(locale, 'selectRange');
  }
  if (mode === 'multiple') {
    return getTranslation(locale, 'selectMultiple');
  }
  if (mode === 'single') {
    return getTranslation(locale, 'selectSingle');
  }
  return '...?';
}
export default function DatePickerModalContentHeader(props) {
  const {
    onToggle,
    collapsed,
    mode,
    moreLabel,
    uppercase,
    editIcon,
    calendarIcon,
    allowEditing
  } = props;
  const theme = useTheme();
  const label = getLabel(props.locale, props.mode, props.label);
  const color = useHeaderTextColor();
  const isEditingEnabled = allowEditing && mode !== 'multiple';
  const supportingTextColor = theme.isV3 ? theme.colors.onSurfaceVariant : color;
  const textFont = theme?.isV3 ? theme.fonts.labelMedium : theme.fonts.medium;
  const collapsedIcon = theme.isV3 ? 'pencil-outline' : 'pencil';
  const expandedIcon = theme.isV3 ? 'calendar-blank' : 'calendar';
  const finalCollapsedIcon = editIcon ?? collapsedIcon;
  const finalExpandedIcon = calendarIcon ?? expandedIcon;
  return /*#__PURE__*/_jsxs(View, {
    style: styles.header,
    children: [/*#__PURE__*/_jsxs(View, {
      children: [/*#__PURE__*/_jsx(Text, {
        maxFontSizeMultiplier: 1.5,
        style: [styles.label, {
          color: supportingTextColor,
          ...textFont
        }],
        children: uppercase ? label.toUpperCase() : label
      }), /*#__PURE__*/_jsxs(View, {
        style: styles.headerContentContainer,
        children: [mode === 'range' ? /*#__PURE__*/_jsx(HeaderContentRange, {
          ...props,
          color: color
        }) : null, mode === 'single' ? /*#__PURE__*/_jsx(HeaderContentSingle, {
          ...props,
          color: color
        }) : null, mode === 'multiple' ? /*#__PURE__*/_jsx(HeaderContentMulti, {
          ...props,
          color: color,
          moreLabel: moreLabel
        }) : null]
      })]
    }), /*#__PURE__*/_jsx(View, {
      style: sharedStyles.root
    }), isEditingEnabled ? /*#__PURE__*/_jsx(IconButton, {
      icon: collapsed ? finalCollapsedIcon : finalExpandedIcon,
      accessibilityLabel: collapsed ? getTranslation(props.locale, 'typeInDate') : getTranslation(props.locale, 'pickDateFromCalendar'),
      iconColor: theme.isV3 ? theme.colors.onSurface : color,
      onPress: onToggle
    }) : null]
  });
}
export function HeaderContentSingle({
  state,
  emptyLabel = ' ',
  color,
  locale
}) {
  const theme = useTheme();
  const lighterColor = Color(color).fade(0.5).rgb().toString();
  const dateColor = state.date ? theme.isV3 ? theme.colors.onSurface : color : lighterColor;
  const formatter = useMemo(() => {
    return new Intl.DateTimeFormat(locale, {
      month: 'short',
      day: 'numeric'
    });
  }, [locale]);
  return /*#__PURE__*/_jsx(Text, {
    maxFontSizeMultiplier: 1.5,
    style: [styles.text, {
      color: dateColor
    }],
    children: state.date ? formatter.format(state.date) : emptyLabel
  });
}
export function HeaderContentMulti({
  state,
  emptyLabel = ' ',
  moreLabel = 'more',
  color,
  locale
}) {
  const theme = useTheme();
  const dateCount = state.dates?.length || 0;
  const lighterColor = Color(color).fade(0.5).rgb().toString();
  const dateColor = dateCount ? theme.isV3 ? theme.colors.onSurface : color : lighterColor;
  const formatter = useMemo(() => {
    return new Intl.DateTimeFormat(locale, {
      month: 'short',
      day: 'numeric'
    });
  }, [locale]);
  let label = emptyLabel;
  if (dateCount) {
    if (dateCount <= 2) {
      label = state.dates.map(date => formatter.format(date)).join(', ');
    } else {
      label = formatter.format(state.dates[0]) + ` (+ ${dateCount - 1} ${moreLabel})`;
    }
  }
  return /*#__PURE__*/_jsx(Text, {
    maxFontSizeMultiplier: 1.5,
    style: [styles.text, {
      color: dateColor
    }],
    children: label
  });
}
export function HeaderContentRange({
  locale,
  state,
  headerSeparator = '-',
  startLabel = 'Start',
  endLabel = 'End',
  color
}) {
  const theme = useTheme();
  const lighterColor = Color(color).fade(0.5).rgb().toString();
  const startColorFilled = theme.isV3 ? theme.colors.onSurface : color;
  const endColorFilled = theme.isV3 ? theme.colors.onSurface : color;
  const startColor = state.startDate ? startColorFilled : lighterColor;
  const endColor = state.endDate ? endColorFilled : lighterColor;
  const formatter = useMemo(() => {
    return new Intl.DateTimeFormat(locale, {
      month: 'short',
      day: 'numeric'
    });
  }, [locale]);
  return /*#__PURE__*/_jsxs(_Fragment, {
    children: [/*#__PURE__*/_jsx(Text, {
      maxFontSizeMultiplier: 1.5,
      style: [styles.text, {
        color: startColor
      }],
      children: state.startDate ? formatter.format(state.startDate) : startLabel
    }), /*#__PURE__*/_jsx(Text, {
      maxFontSizeMultiplier: 1.5,
      style: [styles.headerSeparator, {
        color
      }],
      children: headerSeparator
    }), /*#__PURE__*/_jsx(Text, {
      maxFontSizeMultiplier: 1.5,
      style: [styles.text, {
        color: endColor
      }],
      children: state.endDate ? formatter.format(state.endDate) : endLabel
    })]
  });
}
const styles = StyleSheet.create({
  header: {
    height: 75,
    alignItems: 'center',
    flexDirection: 'row',
    paddingLeft: 24,
    paddingRight: 12
  },
  headerContentContainer: {
    flexDirection: 'row',
    marginTop: 5
  },
  headerSeparator: {
    color: 'rgba(255,255,255,1)',
    fontSize: 25,
    paddingLeft: 6,
    paddingRight: 6
  },
  label: {
    color: '#fff',
    fontSize: 13,
    letterSpacing: 1
  },
  text: {
    color: '#fff',
    fontSize: 25
  }
});
//# sourceMappingURL=DatePickerModalContentHeader.js.map