package expo.modules.location

import android.Manifest
import android.app.Activity
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.hardware.GeomagneticField
import android.hardware.Sensor
import android.hardware.SensorEvent
import android.hardware.SensorEventListener
import android.hardware.SensorManager
import android.location.Geocoder
import android.location.Location
import android.location.LocationListener
import android.location.LocationManager
import android.os.Build
import android.os.Bundle
import android.os.Looper
import android.util.Log
import androidx.annotation.ChecksSdkIntAtLeast
import androidx.core.os.bundleOf
import expo.modules.core.interfaces.ActivityEventListener
import expo.modules.core.interfaces.LifecycleEventListener
import expo.modules.core.interfaces.services.UIManager
import expo.modules.interfaces.taskManager.TaskManagerInterface
import expo.modules.kotlin.Promise
import expo.modules.kotlin.exception.Exceptions
import expo.modules.kotlin.functions.Coroutine
import expo.modules.kotlin.modules.Module
import expo.modules.kotlin.modules.ModuleDefinition
import expo.modules.location.records.GeocodeResponse
import expo.modules.location.records.GeofencingOptions
import expo.modules.location.records.Heading
import expo.modules.location.records.HeadingEventResponse
import expo.modules.location.records.LocationLastKnownOptions
import expo.modules.location.records.LocationOptions
import expo.modules.location.records.LocationProviderStatus
import expo.modules.location.records.LocationResponse
import expo.modules.location.records.LocationTaskOptions
import expo.modules.location.records.PermissionDetailsLocationAndroid
import expo.modules.location.records.PermissionRequestResponse
import expo.modules.location.records.ReverseGeocodeLocation
import expo.modules.location.records.ReverseGeocodeResponse
import expo.modules.location.taskConsumers.GeofencingTaskConsumer
import expo.modules.location.taskConsumers.LocationTaskConsumer
import java.util.Locale
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException
import kotlin.coroutines.suspendCoroutine
import kotlin.math.abs

class LocationModule : Module(), LifecycleEventListener, SensorEventListener, ActivityEventListener {
  private var mGeofield: GeomagneticField? = null
  private var mPendingLocationRequests = ArrayList<LocationActivityResultListener>()
  private lateinit var mContext: Context
  private lateinit var mSensorManager: SensorManager
  private lateinit var mUIManager: UIManager
  private lateinit var mLocationProvider: LocationManager

  private var mGravity: FloatArray = FloatArray(9)
  private var mGeomagnetic: FloatArray = FloatArray(9)
  private var mHeadingId = 0
  private var mLastAzimuth = 0f
  private var mAccuracy = 0
  private var mLastUpdate: Long = 0
  private var mGeocoderPaused = false

  private val mTaskManager: TaskManagerInterface by lazy {
    return@lazy appContext.legacyModule<TaskManagerInterface>()
      ?: throw TaskManagerNotFoundException()
  }

  override fun definition() = ModuleDefinition {
    Name("ExpoLocation")

    OnCreate {
      mContext = appContext.reactContext ?: throw Exceptions.ReactContextLost()
      mUIManager = appContext.legacyModule<UIManager>() ?: throw MissingUIManagerException()
      mLocationManager = mContext.getSystemService(Context.LOCATION_SERVICE) as LocationManager
      mSensorManager = mContext.getSystemService(Context.SENSOR_SERVICE) as? SensorManager
        ?: throw SensorManagerUnavailable()
    }

    Events(HEADING_EVENT_NAME, LOCATION_EVENT_NAME)

    // Deprecated
    AsyncFunction("requestPermissionsAsync") Coroutine { ->
      val permissionsManager = appContext.permissions ?: throw NoPermissionsModuleException()

      return@Coroutine if (Build.VERSION.SDK_INT == Build.VERSION_CODES.Q) {
        LocationHelpers.askForPermissionsWithPermissionsManager(
          permissionsManager,
          Manifest.permission.ACCESS_FINE_LOCATION,
          Manifest.permission.ACCESS_COARSE_LOCATION,
          Manifest.permission.ACCESS_BACKGROUND_LOCATION
        )
      } else {
        LocationHelpers.askForPermissionsWithPermissionsManager(permissionsManager, Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION)
      }
    }

    // Deprecated
    AsyncFunction("getPermissionsAsync") Coroutine { ->
      val permissionsManager = appContext.permissions ?: throw NoPermissionsModuleException()

      return@Coroutine if (Build.VERSION.SDK_INT == Build.VERSION_CODES.Q) {
        LocationHelpers.getPermissionsWithPermissionsManager(
          permissionsManager,
          Manifest.permission.ACCESS_FINE_LOCATION,
          Manifest.permission.ACCESS_COARSE_LOCATION,
          Manifest.permission.ACCESS_BACKGROUND_LOCATION
        )
      } else {
        getForegroundPermissionsAsync()
      }
    }

    AsyncFunction("requestForegroundPermissionsAsync") Coroutine { ->
      val permissionsManager = appContext.permissions ?: throw NoPermissionsModuleException()

      LocationHelpers.askForPermissionsWithPermissionsManager(permissionsManager, Manifest.permission.ACCESS_FINE_LOCATION, Manifest.permission.ACCESS_COARSE_LOCATION)
      // We aren't using the values returned above, because we need to check if the user has provided fine location permissions
      return@Coroutine getForegroundPermissionsAsync()
    }

    AsyncFunction("requestBackgroundPermissionsAsync") Coroutine { ->
      return@Coroutine requestBackgroundPermissionsAsync()
    }

    AsyncFunction("getForegroundPermissionsAsync") Coroutine { ->
      return@Coroutine getForegroundPermissionsAsync()
    }

    AsyncFunction("getBackgroundPermissionsAsync") Coroutine { ->
      return@Coroutine getBackgroundPermissionsAsync()
    }

    /*
    AsyncFunction("getLastKnownPositionAsync") Coroutine { options: LocationLastKnownOptions ->
    }
    */

    AsyncFunction("getCurrentPositionAsync") { options: LocationOptions, promise: Promise ->
      return@AsyncFunction getCurrentPositionAsync(options, promise)
    }

    /*
    AsyncFunction<LocationProviderStatus>("getProviderStatusAsync") {
    }
    */

    /*
    AsyncFunction("watchDeviceHeading") { watchId: Int ->
    }
    */

    /*
    AsyncFunction("watchPositionImplAsync") { watchId: Int, options: LocationOptions, promise: Promise ->
    }
    */

    /*
    AsyncFunction("removeWatchAsync") { watchId: Int ->
    }
    */

    /*
    AsyncFunction("geocodeAsync") Coroutine { address: String ->
    }
    */

    /*
    AsyncFunction("reverseGeocodeAsync") Coroutine { location: ReverseGeocodeLocation ->
    }
    */

    /*
    AsyncFunction("enableNetworkProviderAsync") Coroutine { ->
    }
    */

    /*
    AsyncFunction<Boolean>("hasServicesEnabledAsync") {
    }
    */

    /*
    AsyncFunction("startLocationUpdatesAsync") { taskName: String, options: LocationTaskOptions ->
    }
    */

    /*
    AsyncFunction("stopLocationUpdatesAsync") { taskName: String ->
    }
    */

    /*
    AsyncFunction("hasStartedLocationUpdatesAsync") { taskName: String ->
    }
    */

    /*
    AsyncFunction("startGeofencingAsync") { taskName: String, options: GeofencingOptions ->
    }
    */

    /*
    AsyncFunction("hasStartedGeofencingAsync") { taskName: String ->
    }
    */

    /*
    AsyncFunction("stopGeofencingAsync") { taskName: String ->
    }
    */

    OnActivityEntersForeground {
      AppForegroundedSingleton.isForegrounded = true
    }

    OnActivityEntersBackground {
      AppForegroundedSingleton.isForegrounded = false
    }
  }

  private suspend fun getForegroundPermissionsAsync(): PermissionRequestResponse {
    appContext.permissions?.let {
      val locationPermission = LocationHelpers.getPermissionsWithPermissionsManager(it, Manifest.permission.ACCESS_COARSE_LOCATION)
      val fineLocationPermission = LocationHelpers.getPermissionsWithPermissionsManager(it, Manifest.permission.ACCESS_FINE_LOCATION)

      var accuracy = "none"
      if (locationPermission.granted) {
        accuracy = "coarse"
      }
      if (fineLocationPermission.granted) {
        accuracy = "fine"
      }

      locationPermission.android = PermissionDetailsLocationAndroid(
        accuracy = accuracy
      )

      return locationPermission
    } ?: throw NoPermissionsModuleException()
  }

  private suspend fun requestBackgroundPermissionsAsync(): PermissionRequestResponse {
    if (!isBackgroundPermissionInManifest()) {
      throw NoPermissionInManifestException("ACCESS_BACKGROUND_LOCATION")
    }
    if (!shouldAskBackgroundPermissions()) {
      return getForegroundPermissionsAsync()
    }
    return appContext.permissions?.let {
      val permissionResponseBundle = LocationHelpers.askForPermissionsWithPermissionsManager(it, Manifest.permission.ACCESS_BACKGROUND_LOCATION)
      PermissionRequestResponse(permissionResponseBundle)
    } ?: throw NoPermissionsModuleException()
  }

  private suspend fun getBackgroundPermissionsAsync(): PermissionRequestResponse {
    if (!isBackgroundPermissionInManifest()) {
      throw NoPermissionInManifestException("ACCESS_BACKGROUND_LOCATION")
    }
    if (!shouldAskBackgroundPermissions()) {
      return getForegroundPermissionsAsync()
    }
    appContext.permissions?.let {
      return LocationHelpers.getPermissionsWithPermissionsManager(it, Manifest.permission.ACCESS_BACKGROUND_LOCATION)
    } ?: throw NoPermissionsModuleException()
  }

  /**
   * Resolves to the last known position if it is available and matches given requirements or null otherwise.
   */
  /*
  private suspend fun getLastKnownPositionAsync(options: LocationLastKnownOptions): LocationResponse? {
  }
  */

  /**
   * Requests for the current position. Depending on given accuracy, it may take some time to resolve.
   * If you don't need an up-to-date location see `getLastKnownPosition`.
   */
  @Suppress("UNUSED_PARAMETER") // Parameter 'options' is never used
  private fun getCurrentPositionAsync(options: LocationOptions, promise: Promise) {
    // Check for permissions
    if (isMissingForegroundPermissions()) {
      promise.reject(LocationUnauthorizedException())
      return
    }

    val locationListener = object : LocationListener {
      var promiseResolved = false

      override fun onLocationChanged(location: Location) {
        if (!promiseResolved) {
          promiseResolved = true
          promise.resolve(LocationResponse(location))
        }
      }

      override fun onProviderDisabled(provider: String) {}
      override fun onProviderEnabled(provider: String) {}
    }

    // Request location updates from both GPS and Network providers
    if (mLocationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)) {
      mLocationManager.requestLocationUpdates(LocationManager.GPS_PROVIDER, 0L, 0f, locationListener)
    }
    if (mLocationManager.isProviderEnabled(LocationManager.NETWORK_PROVIDER)) {
      mLocationManager.requestLocationUpdates(LocationManager.NETWORK_PROVIDER, 0L, 0f, locationListener)
    }
  }

  /*
  fun requestLocationUpdates(locationRequest: LocationRequest, requestId: Int?, callbacks: LocationRequestCallbacks) {
  }
  */

  /*
  private fun addPendingLocationRequest(locationRequest: LocationRequest, listener: LocationActivityResultListener) {
  }
  */

  /**
   * Triggers system's dialog to ask the user to enable settings required for given location request.
   */
  /*
  private fun resolveUserSettingsForRequest(locationRequest: LocationRequest) {
  }
  */

  /*
  private fun executePendingRequests(resultCode: Int) {
  }
  */

  /*
  private fun startHeadingUpdate() {
  }
  */

  /*
  private fun sendUpdate() {
  }
  */

  /*
  internal fun sendLocationResponse(watchId: Int, response: LocationResponse) {
  }
  */

  /*
  private fun calcMagNorth(azimuth: Float): Float {
  }
  */

  /*
  private fun calcTrueNorth(magNorth: Float): Float {
  }
  */

  /*
  private fun stopHeadingWatch() {
  }
  */

  /*
  private fun destroyHeadingWatch() {
  }
  */

  /*
  private fun startWatching() {
  }
  */

  /*
  private fun stopWatching() {
  }
  */

  /*
  private fun pauseLocationUpdatesForRequest(requestId: Int) {
  }
  */

  /*
  private fun removeLocationUpdatesForRequest(requestId: Int) {
  }
  */

  /*
  private fun resumeLocationUpdates() {
  }
  */

  /**
   * Gets the best most recent location found by the provider.
   */
  /*
  private suspend fun getLastKnownLocation(): Location? {
  }
  */

  /*
  private suspend fun geocode(address: String): List<GeocodeResponse> {
  }
  */

  /*
  private suspend fun reverseGeocode(location: ReverseGeocodeLocation): List<ReverseGeocodeResponse> {
  }
  */

  //region private methods
  /**
   * Checks whether all required permissions have been granted by the user.
   */
  private fun isMissingForegroundPermissions(): Boolean {
    appContext.permissions?.let {
      val canAccessFineLocation = it.hasGrantedPermissions(Manifest.permission.ACCESS_FINE_LOCATION)
      val canAccessCoarseLocation = it.hasGrantedPermissions(Manifest.permission.ACCESS_COARSE_LOCATION)
      return !canAccessFineLocation && !canAccessCoarseLocation
    } ?: throw Exceptions.AppContextLost()
  }

  private fun hasForegroundServicePermissions(): Boolean {
    appContext.permissions?.let {
      return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
        val canAccessForegroundServiceLocation = it.hasGrantedPermissions(Manifest.permission.FOREGROUND_SERVICE_LOCATION)
        val canAccessForegroundService = it.hasGrantedPermissions(Manifest.permission.FOREGROUND_SERVICE)
        canAccessForegroundService && canAccessForegroundServiceLocation
      } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
        val canAccessForegroundService = it.hasGrantedPermissions(Manifest.permission.FOREGROUND_SERVICE)
        canAccessForegroundService
      } else {
        true
      }
    } ?: throw Exceptions.AppContextLost()
  }

  /**
   * Checks if the background location permission is granted by the user.
   */
  private fun isMissingBackgroundPermissions(): Boolean {
    appContext.permissions?.let {
      return Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q && !it.hasGrantedPermissions(Manifest.permission.ACCESS_BACKGROUND_LOCATION)
    }
    return true
  }

  /**
   * Check if we need to request background location permission separately.
   *
   * @see `https://medium.com/swlh/request-location-permission-correctly-in-android-11-61afe95a11ad`
   */
  @ChecksSdkIntAtLeast(api = Build.VERSION_CODES.Q)
  private fun shouldAskBackgroundPermissions(): Boolean {
    return Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q
  }

  private fun isBackgroundPermissionInManifest(): Boolean {
    return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
      appContext.permissions?.let {
        return it.isPermissionPresentInManifest(Manifest.permission.ACCESS_BACKGROUND_LOCATION)
      }
      throw NoPermissionsModuleException()
    } else {
      true
    }
  }

  /**
   * Helper method that lazy-loads the location provider for the context that the module was created.
   */

  companion object {
    internal val TAG = LocationModule::class.java.simpleName
    private const val LOCATION_EVENT_NAME = "Expo.locationChanged"
    private const val HEADING_EVENT_NAME = "Expo.headingChanged"
    private const val CHECK_SETTINGS_REQUEST_CODE = 42

    const val ACCURACY_BALANCED = 3

    private lateinit var mLocationManager: LocationManager
  }

  override fun onHostResume() {
  }

  override fun onHostPause() {
  }

  override fun onHostDestroy() {
  }

  override fun onSensorChanged(event: SensorEvent?) {
  }

  override fun onAccuracyChanged(sensor: Sensor?, accuracy: Int) {
  }

  override fun onActivityResult(activity: Activity?, requestCode: Int, resultCode: Int, data: Intent?) {
  }

  override fun onNewIntent(intent: Intent?) {}
}
