/*!
 * React Native Autolink
 *
 * Copyright 2016-2023 Josh Swan
 * Released under the MIT license
 * https://github.com/joshswan/react-native-autolink/blob/master/LICENSE
 */
import React, { createElement, useCallback, useRef } from 'react';
import { Autolinker, AnchorTagBuilder, } from 'autolinker/dist/es2015';
import { Alert, Linking, StyleSheet, Text } from 'react-native';
import { truncate } from './truncate';
import { CustomMatch } from './CustomMatch';
import * as urls from './urls';
const makeTokenGenerator = (uid) => {
    let counter = 0;
    return [
        // eslint-disable-next-line no-plusplus
        () => `@__ELEMENT-${uid}-${counter++}__@`,
        new RegExp(`(@__ELEMENT-${uid}-\\d+__@)`, 'g'),
    ];
};
const styles = StyleSheet.create({
    link: {
        color: '#0E7AFE',
    },
});
const tagBuilder = new AnchorTagBuilder();
export const Autolink = React.memo(({ as, component, email = true, hashtag = false, linkProps = {}, linkStyle, matchers = [], mention = false, onPress: onPressProp, onLongPress: onLongPressProp, phone = false, renderLink: renderLinkProp, renderText, showAlert = false, stripPrefix = true, stripTrailingSlash = true, text, textProps = {}, truncate: truncateProp = 0, truncateChars = '..', truncateLocation = 'smart', url = true, useNativeSchemes = false, ...props }) => {
    const getUrl = useCallback((match) => {
        switch (match.getType()) {
            case 'email':
                return urls.getEmailUrl(match);
            case 'hashtag':
                return urls.getHashtagUrl(match, hashtag, useNativeSchemes);
            case 'mention':
                return urls.getMentionUrl(match, mention, useNativeSchemes);
            case 'phone':
                return urls.getPhoneUrl(match, phone);
            default:
                return match.getAnchorHref();
        }
    }, [hashtag, mention, phone, useNativeSchemes]);
    const onPress = useCallback((match, alertShown) => {
        var _a, _b;
        // Bypass default press handling if matcher has custom onPress
        if (match instanceof CustomMatch && match.getMatcher().onPress) {
            (_b = (_a = match.getMatcher()).onPress) === null || _b === void 0 ? void 0 : _b.call(_a, match);
            return;
        }
        // Check if alert needs to be shown
        if (showAlert && !alertShown) {
            Alert.alert('Leaving App', 'Do you want to continue?', [
                { text: 'Cancel', style: 'cancel' },
                { text: 'OK', onPress: () => onPress(match, true) },
            ]);
            return;
        }
        const linkUrl = getUrl(match);
        if (onPressProp) {
            onPressProp(linkUrl, match);
        }
        else {
            Linking.openURL(linkUrl);
        }
    }, [getUrl, onPressProp, showAlert]);
    const onLongPress = useCallback((match) => {
        var _a, _b;
        // Bypass default press handling if matcher has custom onLongPress
        if (match instanceof CustomMatch && match.getMatcher().onLongPress) {
            (_b = (_a = match.getMatcher()).onLongPress) === null || _b === void 0 ? void 0 : _b.call(_a, match);
            return;
        }
        if (onLongPressProp) {
            const linkUrl = getUrl(match);
            onLongPressProp(linkUrl, match);
        }
    }, [getUrl, onLongPressProp]);
    const renderLink = useCallback((linkText, match, index) => {
        const truncated = truncateProp
            ? truncate(linkText, truncateProp, truncateChars, truncateLocation)
            : linkText;
        return (React.createElement(Text, { style: (match instanceof CustomMatch && match.getMatcher().style) || linkStyle || styles.link, onPress: () => onPress(match), onLongPress: () => onLongPress(match), ...linkProps, key: index }, truncated));
    }, [linkProps, linkStyle, truncateProp, truncateChars, truncateLocation, onPress, onLongPress]);
    // Creates a token with a random UID that should not be guessable or
    // conflict with other parts of the string.
    const uid = useRef(Math.floor(Math.random() * 0x10000000000).toString(16));
    const [generateToken, tokenRegexp] = makeTokenGenerator(uid.current);
    const matches = {};
    let linkedText;
    try {
        linkedText = Autolinker.link(text || '', {
            email,
            hashtag,
            mention,
            phone: !!phone,
            urls: url,
            stripPrefix,
            stripTrailingSlash,
            replaceFn: (match) => {
                const token = generateToken();
                matches[token] = match;
                return token;
            },
        });
        // User-specified custom matchers
        matchers.forEach((matcher) => {
            linkedText = linkedText.replace(matcher.pattern, (...replacerArgs) => {
                const token = generateToken();
                const matchedText = replacerArgs[0];
                matches[token] = new CustomMatch({
                    matcher,
                    matchedText,
                    offset: replacerArgs[replacerArgs.length - 2],
                    replacerArgs,
                    tagBuilder,
                });
                return token;
            });
        });
    }
    catch (e) {
        // eslint-disable-next-line no-console
        console.warn('RN Autolink error:', e);
        return null;
    }
    const nodes = linkedText
        .split(tokenRegexp)
        .filter((part) => !!part)
        .map((part, index) => {
        var _a, _b;
        const match = matches[part];
        // Check if rendering link or text node
        if (match === null || match === void 0 ? void 0 : match.getType()) {
            return (((_b = (_a = match).getRenderFn) === null || _b === void 0 ? void 0 : _b.call(_a)) || renderLinkProp || renderLink)(match.getAnchorText(), match, index);
        }
        return renderText ? (renderText(part, index)) : (
        // eslint-disable-next-line react/jsx-props-no-spreading, react/no-array-index-key
        React.createElement(Text, { ...textProps, key: index }, part));
    });
    return createElement(as || component || Text, props, ...nodes);
});
