package com.akslabs.cloudgallery.ui.main.nav

import androidx.compose.animation.EnterTransition
import androidx.compose.animation.core.tween
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.animation.scaleOut
import androidx.compose.animation.slideOutHorizontally
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.ui.Modifier
import androidx.navigation.NavHostController
import androidx.compose.runtime.rememberUpdatedState
import androidx.navigation.compose.NavHost
import androidx.compose.ui.graphics.TransformOrigin
import androidx.navigation.compose.composable
import com.akslabs.cloudgallery.ui.main.MainViewModel
import com.akslabs.cloudgallery.ui.main.screens.local.LocalPhotoGrid
import com.akslabs.cloudgallery.ui.main.screens.remote.RemotePhotosGrid
import com.akslabs.cloudgallery.ui.main.screens.settings.SettingsScreen
import com.akslabs.cloudgallery.ui.main.screens.trash.TrashBinScreen
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.viewmodel.compose.LocalViewModelStoreOwner
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import androidx.paging.compose.collectAsLazyPagingItems

@Composable
fun AppNavHost(
    modifier: Modifier = Modifier,
    navController: NavHostController,
    expanded: Boolean,
    onExpandedChange: (Boolean) -> Unit,
    selectionMode: Boolean,
    selectedPhotos: Set<String>,
    onSelectionModeChange: (Boolean) -> Unit,
    onSelectedPhotosChange: (Set<String>) -> Unit,
    deletedPhotoIds: List<String> = emptyList()
) {
    val currentSelectionMode by rememberUpdatedState(selectionMode)
    val currentOnSelectionModeChange by rememberUpdatedState(onSelectionModeChange)
    val currentOnSelectedPhotosChange by rememberUpdatedState(onSelectedPhotosChange)

    LaunchedEffect(navController) {
        navController.addOnDestinationChangedListener { _, _, _ ->
            if (currentSelectionMode) {
                currentOnSelectionModeChange(false)
                currentOnSelectedPhotosChange(emptySet())
            }
        }
    }

    NavHost(
        modifier = modifier,
        navController = navController,
        startDestination = Screens.LocalPhotos.route,

        popEnterTransition = { EnterTransition.None },
        popExitTransition = {
            scaleOut(
                targetScale = 0.9F,
                transformOrigin = TransformOrigin(pivotFractionX = 0.5f, pivotFractionY = 0.5f)
            )
        },
            ) {
        composable(route = Screens.LocalPhotos.route) {
            val viewModel: MainViewModel = screenScopedViewModel()
            val localPhotos = viewModel.localPhotosFlow.collectAsLazyPagingItems()
            val localPhotosCount by viewModel.localPhotosCount.collectAsStateWithLifecycle()

            LocalPhotoGrid(
                localPhotos = localPhotos,
                totalCount = localPhotosCount,
                expanded = expanded,
                onExpandedChange = onExpandedChange,
                selectionMode = selectionMode,
                selectedPhotos = selectedPhotos,
                onSelectionModeChange = onSelectionModeChange,
                onSelectedPhotosChange = onSelectedPhotosChange,
                deletedPhotoIds = deletedPhotoIds
            )
        }

        composable(route = Screens.RemotePhotos.route) {
            val viewModel: MainViewModel = screenScopedViewModel()
            val allCloudPhotos = viewModel.allCloudPhotosFlow.collectAsLazyPagingItems()
            val totalCloudPhotosCount by viewModel.totalCloudPhotosCount.collectAsStateWithLifecycle()

            RemotePhotosGrid(
                cloudPhotos = allCloudPhotos,
                onPhotoClick = { _, _ -> },
                expanded = expanded,
                onExpandedChange = onExpandedChange,
                selectionMode = selectionMode,
                selectedPhotos = selectedPhotos,
                onSelectionModeChange = onSelectionModeChange,
                onSelectedPhotosChange = onSelectedPhotosChange
            )

        }

        composable(route = Screens.Settings.route) {
            SettingsScreen()
        }

        composable(route = Screens.TrashBin.route) {
            TrashBinScreen(
                expanded = expanded,
                onExpandedChange = onExpandedChange,
                selectionMode = selectionMode,
                selectedPhotos = selectedPhotos,
                onSelectionModeChange = onSelectionModeChange,
                onSelectedPhotosChange = onSelectedPhotosChange
            )
        }
    }
}

/**
 * Provides a [ViewModel] instance scoped the screen's life.
 * When the user navigates away from the screen all screen scoped
 * viewModels are destroyed.
 */
@Composable
inline fun <reified T : ViewModel> screenScopedViewModel(
    factory: ViewModelProvider.Factory? = null,
): T {
    val viewModelStoreOwner = LocalViewModelStoreOwner.current
    requireNotNull(viewModelStoreOwner) { "No ViewModelStoreOwner provided" }
    val viewModelProvider = factory?.let {
        ViewModelProvider(viewModelStoreOwner, it)
    } ?: ViewModelProvider(viewModelStoreOwner)
    return viewModelProvider[T::class.java]
}
