/*
 * Copyright (C) 2025 aisleron.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package com.aisleron.ui.shoppinglist

import android.app.SearchManager
import android.content.Context
import android.os.Bundle
import android.view.LayoutInflater
import android.view.Menu
import android.view.MenuInflater
import android.view.MenuItem
import android.view.View
import android.view.View.OnAttachStateChangeListener
import android.view.ViewGroup
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.view.ActionMode
import androidx.appcompat.widget.SearchView
import androidx.core.content.ContextCompat.getSystemService
import androidx.core.view.MenuProvider
import androidx.fragment.app.Fragment
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.navigation.fragment.findNavController
import androidx.recyclerview.widget.ItemTouchHelper
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.aisleron.R
import com.aisleron.domain.FilterType
import com.aisleron.domain.base.AisleronException
import com.aisleron.domain.location.LocationType
import com.aisleron.domain.loyaltycard.LoyaltyCard
import com.aisleron.ui.AisleronExceptionMap
import com.aisleron.ui.AisleronFragment
import com.aisleron.ui.ApplicationTitleUpdateListener
import com.aisleron.ui.FabHandler
import com.aisleron.ui.FabHandler.FabClickedCallBack
import com.aisleron.ui.aisle.AisleDialogFragment
import com.aisleron.ui.aisle.AislePickerDialogFragment
import com.aisleron.ui.bundles.AisleDialogBundle
import com.aisleron.ui.bundles.AisleListEntry
import com.aisleron.ui.bundles.AislePickerBundle
import com.aisleron.ui.bundles.Bundler
import com.aisleron.ui.copyentity.CopyEntityDialogFragment
import com.aisleron.ui.copyentity.CopyEntityType
import com.aisleron.ui.loyaltycard.LoyaltyCardProvider
import com.aisleron.ui.note.NoteDialogFragment
import com.aisleron.ui.note.NoteParentRef
import com.aisleron.ui.settings.ShoppingListPreferences
import com.aisleron.ui.widgets.ErrorSnackBar
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import com.google.android.material.snackbar.Snackbar
import kotlinx.coroutines.launch
import org.koin.androidx.viewmodel.ext.android.viewModel

class ShoppingListFragment(
    private val applicationTitleUpdateListener: ApplicationTitleUpdateListener,
    private val fabHandler: FabHandler,
    private val shoppingListPreferences: ShoppingListPreferences,
    private val loyaltyCardProvider: LoyaltyCardProvider
) : Fragment(), SearchView.OnQueryTextListener, ActionMode.Callback, FabClickedCallBack,
    MenuProvider, AisleronFragment {


    private var searchView: SearchView? = null
    private val searchViewListener = object : OnAttachStateChangeListener {
        override fun onViewAttachedToWindow(v: View) {}
        override fun onViewDetachedFromWindow(v: View) {
            if (!hasSelectedItems()) shoppingListViewModel.requestDefaultList()
        }
    }

    private var actionMode: ActionMode? = null
    private var loyaltyCardMenuItem: MenuItem? = null

    private val showEmptyAisles: Boolean
        get() = shoppingListPreferences.showEmptyAisles(requireContext())

    private val shoppingListViewModel: ShoppingListViewModel by viewModel()

    override fun onResume() {
        super.onResume()
        shoppingListViewModel.setShowEmptyAisles(showEmptyAisles)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val shoppingListBundle = Bundler().getShoppingListBundle(arguments)
        shoppingListViewModel.hydrate(
            shoppingListBundle.locationId,
            shoppingListBundle.filterType,
            shoppingListPreferences.showEmptyAisles(requireContext())
        )

        childFragmentManager.setFragmentResultListener(
            AISLE_PICKER_REQUEST_KEY, this
        ) { _, bundle ->
            val selectedAisleId = bundle.getInt(AislePickerDialogFragment.KEY_SELECTED_AISLE_ID, -1)
            val addNewAisle = bundle.getBoolean(AislePickerDialogFragment.KEY_ADD_NEW_AISLE, false)

            if (selectedAisleId != -1) {
                shoppingListViewModel.updateSelectedProductAisle(selectedAisleId)
            }

            if (addNewAisle) {
                showAddSingleAisleDialog()
            }
        }

        childFragmentManager.setFragmentResultListener(
            ADD_AISLE_REQUEST_KEY, this
        ) { _, bundle ->
            val newAisleId = bundle.getInt(AisleDialogFragment.KEY_AISLE_ID, -1)
            shoppingListViewModel.updateSelectedProductAisle(newAisleId)
        }

        childFragmentManager.setFragmentResultListener(
            EDIT_AISLE_REQUEST_KEY, this
        ) { _, _ ->
            actionMode?.finish()
        }
    }

    override fun onCreateView(
        inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?
    ): View? {
        initializeFab()

        val view = inflater.inflate(R.layout.fragment_shopping_list, container, false)

        // Set the adapter
        if (view is RecyclerView) {
            setWindowInsetListeners(this, view, true, null)
            viewLifecycleOwner.lifecycleScope.launch {
                viewLifecycleOwner.repeatOnLifecycle(Lifecycle.State.STARTED) {
                    launch {
                        shoppingListViewModel.shoppingListUiState.collect {
                            when (it) {
                                is ShoppingListViewModel.ShoppingListUiState.Error -> {
                                    displayErrorSnackBar(
                                        it.errorCode,
                                        it.errorMessage,
                                        fabHandler.getFabView(requireActivity())
                                    )

                                    shoppingListViewModel.clearState()
                                }

                                is ShoppingListViewModel.ShoppingListUiState.Updated -> {
                                    updateTitle()
                                    setMenuItemVisibility()

                                    (view.adapter as ShoppingListItemRecyclerViewAdapter).submitList(
                                        it.shoppingList
                                    )

                                    initializeActionMode()
                                }

                                else -> Unit
                            }
                        }
                    }

                    launch {
                        shoppingListViewModel.aislesForLocation.collect { data ->
                            if (data.isNotEmpty()) {
                                showAislePickerDialog(data)
                                shoppingListViewModel.clearLocationAisles()
                            }
                        }
                    }
                }
            }

            with(view) {
                var touchHelper: ItemTouchHelper? = null

                LinearLayoutManager(context)
                adapter = ShoppingListItemRecyclerViewAdapter(
                    object :
                        ShoppingListItemRecyclerViewAdapter.ShoppingListItemListener {
                        override fun onClick(item: ShoppingListItem, view: View) {
                            // If no items are selected, do nothing
                            if (!hasSelectedItems()) return
                            shoppingListViewModel.toggleItemSelection(item)
                        }

                        override fun onProductStatusChange(
                            item: ProductShoppingListItem, inStock: Boolean
                        ) {
                            shoppingListViewModel.updateProductStatus(item, inStock)
                            displayStatusChangeSnackBar(item, inStock)
                        }

                        override fun onProductQuantityChange(
                            item: ProductShoppingListItem, quantity: Double?
                        ) {
                            shoppingListViewModel.updateProductNeededQuantity(item, quantity)
                        }

                        override fun onListPositionChanged(
                            item: ShoppingListItem, precedingItem: ShoppingListItem?
                        ) {
                            shoppingListViewModel.updateItemRank(item, precedingItem)
                        }

                        override fun onLongClick(item: ShoppingListItem, view: View): Boolean {
                            if (item.itemType == ShoppingListItem.ItemType.EMPTY_LIST) return false

                            // Finish the previous action mode and start a new one
                            actionMode?.finish()

                            shoppingListViewModel.toggleItemSelection(item)
                            return true
                        }

                        override fun onMoved(item: ShoppingListItem) {
                            shoppingListViewModel.movedItem(item)
                        }

                        override fun onAisleExpandToggle(
                            item: AisleShoppingListItem, expanded: Boolean
                        ) {
                            if (hasSelectedItems()) return

                            shoppingListViewModel.updateAisleExpanded(item, expanded)
                        }

                        override fun onDragStart(viewHolder: RecyclerView.ViewHolder) {
                            touchHelper?.startDrag(viewHolder)
                        }

                        override fun onMove(item: ShoppingListItem) {}

                        override fun hasSelectedItems(): Boolean {
                            return this@ShoppingListFragment.hasSelectedItems()
                        }
                    },

                    shoppingListPreferences.trackingMode(requireContext()),
                    getString(R.string.qty),
                    shoppingListViewModel.defaultFilter
                )

                val callback: ItemTouchHelper.Callback = ShoppingListItemMoveCallbackListener(
                    view.adapter as ShoppingListItemRecyclerViewAdapter
                )
                touchHelper = ItemTouchHelper(callback)
                touchHelper.attachToRecyclerView(view)
            }
        }
        return view
    }

    override fun onDestroyView() {
        searchView?.removeOnAttachStateChangeListener(searchViewListener)
        searchView = null
        super.onDestroyView()
    }

    private fun initializeActionMode() {
        if (!hasSelectedItems()) {
            actionMode?.finish()
            return
        }

        actionMode = actionMode ?: (requireActivity() as AppCompatActivity).startSupportActionMode(
            this@ShoppingListFragment
        )

        setActionModeOptions(actionMode)
    }

    private fun showAislePickerDialog(aisleList: List<AisleListEntry>) {
        // TODO : Replace Bundler with NavigationArgs
        val currentAisle = shoppingListViewModel.getSelectedItemAisleId()
        val aislePickerBundle = AislePickerBundle(
            aisles = aisleList,
            currentAisleId = currentAisle,
            title = getSelectedItemsDescription()
        )

        AislePickerDialogFragment.newInstance(aislePickerBundle, AISLE_PICKER_REQUEST_KEY)
            .show(childFragmentManager, AislePickerDialogFragment.TAG)
    }

    private fun setMenuItemVisibility() {
        loyaltyCardMenuItem?.isVisible = shoppingListViewModel.loyaltyCard != null
    }

    private fun displayStatusChangeSnackBar(item: ProductShoppingListItem, inStock: Boolean) {
        if (shoppingListPreferences.isStatusChangeSnackBarHidden(requireContext())) return

        val newStatus = getString(if (inStock) R.string.menu_in_stock else R.string.menu_needed)

        Snackbar.make(
            requireView(),
            getString(R.string.status_change_confirmation, item.name, newStatus),
            Snackbar.LENGTH_SHORT
        ).setAction(getString(R.string.undo)) { _ ->
            shoppingListViewModel.updateProductStatus(item, !inStock)
        }.setAnchorView(fabHandler.getFabView(this.requireActivity())).show()
    }

    private fun displayErrorSnackBar(
        errorCode: AisleronException.ExceptionCode, errorMessage: String?, anchorView: View?
    ) {
        val snackBarMessage =
            getString(AisleronExceptionMap().getErrorResourceId(errorCode), errorMessage)

        ErrorSnackBar().make(
            requireView(),
            snackBarMessage,
            Snackbar.LENGTH_SHORT,
            anchorView
        ).show()
    }

    private fun showAisleDialog(aisleId: Int, action: AisleDialogFragment.AisleDialogAction) {
        // TODO : Replace Bundler with NavigationArgs
        val requestKey = when (action) {
            AisleDialogFragment.AisleDialogAction.ADD_SINGLE -> ADD_AISLE_REQUEST_KEY
            AisleDialogFragment.AisleDialogAction.ADD_MULTIPLE -> ADD_AISLE_REQUEST_KEY
            AisleDialogFragment.AisleDialogAction.EDIT -> EDIT_AISLE_REQUEST_KEY
        }

        val aisleDialogBundle = AisleDialogBundle(
            aisleId = aisleId,
            action = action,
            locationId = shoppingListViewModel.locationId
        )

        AisleDialogFragment.newInstance(aisleDialogBundle, requestKey)
            .show(childFragmentManager, AisleDialogFragment.TAG)
    }

    private fun showAddSingleAisleDialog() {
        showAisleDialog(-1, AisleDialogFragment.AisleDialogAction.ADD_SINGLE)
    }

    private fun showAddMultiAisleDialog() {
        showAisleDialog(-1, AisleDialogFragment.AisleDialogAction.ADD_MULTIPLE)
    }

    private fun showEditAisleDialog(aisleId: Int) {
        showAisleDialog(aisleId, AisleDialogFragment.AisleDialogAction.EDIT)
    }

    private fun initializeFab() {
        fabHandler.setFabOnClickedListener(this)
        fabHandler.setFabItems(
            this.requireActivity(),
            FabHandler.FabOption.ADD_SHOP,
            FabHandler.FabOption.ADD_AISLE,
            FabHandler.FabOption.ADD_PRODUCT
        )

        fabHandler.setFabOnClickListener(this.requireActivity(), FabHandler.FabOption.ADD_PRODUCT) {
            navigateToAddProduct(shoppingListViewModel.defaultFilter)
        }

        fabHandler.setFabOnClickListener(this.requireActivity(), FabHandler.FabOption.ADD_AISLE) {
            showAddMultiAisleDialog()
        }
    }

    private fun updateTitle() {
        val appTitle =
            when (shoppingListViewModel.locationType) {
                LocationType.HOME ->
                    when (shoppingListViewModel.defaultFilter) {
                        FilterType.IN_STOCK -> resources.getString(R.string.menu_in_stock)
                        FilterType.NEEDED -> resources.getString(R.string.menu_needed)
                        FilterType.ALL -> resources.getString(R.string.menu_all_items)
                    }

                LocationType.SHOP -> shoppingListViewModel.locationName
            }

        applicationTitleUpdateListener.applicationTitleUpdated(requireActivity(), appTitle)
    }

    private fun navigateToAddProduct(filterType: FilterType, aisleId: Int? = null) {
        // TODO : Replace Bundler with NavigationArgs
        val bundle =
            Bundler().makeAddProductBundle(
                name = null,
                inStock = filterType == FilterType.IN_STOCK,
                aisleId = aisleId
            )
        this.findNavController().navigate(R.id.nav_add_product, bundle)
    }

    private fun navigateToEditProduct(productId: Int) {
        // TODO : Replace Bundler with NavigationArgs
        val bundle = Bundler().makeEditProductBundle(
            productId = productId
        )

        this.findNavController().navigate(R.id.nav_add_product, bundle)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        val menuHost = requireActivity()
        menuHost.addMenuProvider(this, viewLifecycleOwner, Lifecycle.State.RESUMED)

        view.keepScreenOn = shoppingListPreferences.keepScreenOn(requireContext())
    }

    private fun navigateToEditShop(locationId: Int) {
        // TODO : Replace Bundler with NavigationArgs
        val bundle = Bundler().makeEditLocationBundle(locationId)
        this.findNavController().navigate(R.id.nav_add_shop, bundle)
    }

    private fun confirmDelete(context: Context) {
        val builder = MaterialAlertDialogBuilder(context)
        val selectedItemsDescription = getSelectedItemsDescription()

        builder
            .setTitle(getString(R.string.delete_confirmation, selectedItemsDescription))
            .setNegativeButton(android.R.string.cancel, null)
            .setPositiveButton(android.R.string.ok) { _, _ ->
                shoppingListViewModel.removeSelectedItems()
            }

        val dialog: AlertDialog = builder.create()

        dialog.show()
    }

    private fun showCopyProductDialog(item: ShoppingListItem) {
        // TODO : Replace Bundler with NavigationArgs
        val dialog = CopyEntityDialogFragment.newInstance(
            type = CopyEntityType.Product(item.id),
            title = getString(R.string.copy_entity_title, item.name),
            defaultName = "${item.name} (${getString(android.R.string.copy)})",
            nameHint = getString(R.string.new_product_name)
        )

        dialog.onCopySuccess = {
            actionMode?.finish()
            requireView().postDelayed({
                Snackbar.make(
                    requireView(),
                    getString(R.string.entity_copied, item.name),
                    Snackbar.LENGTH_SHORT
                )
                    .setAnchorView(fabHandler.getFabView(this.requireActivity()))
                    .show()
            }, 250)
        }

        dialog.show(childFragmentManager, "copyDialog")
    }

    private fun showNoteDialog(item: ShoppingListItem) {
        // TODO : Replace Bundler with NavigationArgs
        val dialog = NoteDialogFragment.newInstance(
            noteParentRef = NoteParentRef.Product(item.id)
        )

        dialog.show(childFragmentManager, "noteDialog")
        // TODO: Add a callback from the dialog to finish the action mode
        actionMode?.finish()
    }

    private fun editShoppingListItem(item: ShoppingListItem) {
        when (item) {
            is AisleShoppingListItem -> showEditAisleDialog(item.aisleId)
            is ProductShoppingListItem -> navigateToEditProduct(item.id)
        }
    }

    override fun onQueryTextSubmit(query: String?): Boolean {
        return false
    }

    override fun onQueryTextChange(newText: String?): Boolean {
        shoppingListViewModel.submitProductSearch(productNameQuery = newText ?: "")
        return false
    }

    override fun onCreateActionMode(mode: ActionMode, menu: Menu): Boolean {
        // Inflate a menu resource providing context menu items.
        val inflater: MenuInflater = mode.menuInflater
        inflater.inflate(R.menu.shopping_list_fragment_context, menu)
        return true
    }

    private fun getSelectedItemsDescription(): String {
        val actionModeItems = shoppingListViewModel.selectedListItems
        val size = actionModeItems.size
        val productsOnly = actionModeItems.all { it is ProductShoppingListItem } && size > 0
        val aislesOnly = actionModeItems.all { it is AisleShoppingListItem } && size > 0
        val singleItem = size == 1

        return when {
            singleItem -> actionModeItems.first().name
            aislesOnly -> getString(R.string.aisles_selected, actionModeItems.size)
            productsOnly -> getString(R.string.products_selected, actionModeItems.size)
            else -> getString(R.string.items_selected, actionModeItems.size)
        }
    }

    private fun setActionModeOptions(mode: ActionMode?) {
        mode?.title = getSelectedItemsDescription()

        val actionModeItems = shoppingListViewModel.selectedListItems
        val productsOnly = actionModeItems.all { it is ProductShoppingListItem }
        val aislesOnly = actionModeItems.all { it is AisleShoppingListItem }
        val singleItem = actionModeItems.size == 1

        mode?.menu?.let {
            it.findItem(R.id.mnu_add_product_to_aisle).isVisible = singleItem && aislesOnly
            it.findItem(R.id.mnu_copy_shopping_list_item).isVisible = singleItem && productsOnly
            it.findItem(R.id.mnu_product_note).isVisible = singleItem && productsOnly
            it.findItem(R.id.mnu_aisle_picker).isVisible = productsOnly
            it.findItem(R.id.mnu_edit_shopping_list_item).isVisible = singleItem
            it.findItem(R.id.mnu_delete_shopping_list_item)
                .setShowAsAction(MenuItem.SHOW_AS_ACTION_NEVER)
        }
    }

    override fun onPrepareActionMode(mode: ActionMode, menu: Menu): Boolean {
        return false // Return false if nothing is done
    }

    override fun onActionItemClicked(mode: ActionMode, item: MenuItem): Boolean {
        val actionModeItems = shoppingListViewModel.selectedListItems

        var result = true
        when (item.itemId) {
            R.id.mnu_edit_shopping_list_item -> editShoppingListItem(actionModeItems.first())
            R.id.mnu_delete_shopping_list_item -> confirmDelete(requireContext())
            R.id.mnu_add_product_to_aisle -> navigateToAddProduct(
                shoppingListViewModel.defaultFilter, actionModeItems.first().aisleId
            )

            R.id.mnu_copy_shopping_list_item -> showCopyProductDialog(actionModeItems.first())
            R.id.mnu_product_note -> showNoteDialog(actionModeItems.first())
            R.id.mnu_aisle_picker -> shoppingListViewModel.requestLocationAisles()
            else -> result = false // No action picked
        }

        return result
    }

    override fun onDestroyActionMode(mode: ActionMode) {
        actionMode = null
        shoppingListViewModel.clearSelectedListItems()
        shoppingListViewModel.requestDefaultList()
    }

    override fun onCreateMenu(menu: Menu, menuInflater: MenuInflater) {
        if (hasSelectedItems()) return
        menuInflater.inflate(R.menu.shopping_list_fragment_main, menu)

        val searchManager =
            getSystemService(requireContext(), SearchManager::class.java) as SearchManager
        val searchableInfo =
            searchManager.getSearchableInfo(requireActivity().componentName)

        searchView = menu.findItem(R.id.action_search).actionView as SearchView
        searchView?.setMaxWidth(Integer.MAX_VALUE)
        searchView?.setSearchableInfo(searchableInfo)
        searchView?.setOnQueryTextListener(this@ShoppingListFragment)
        searchView?.setOnCloseListener {
            shoppingListViewModel.requestDefaultList()
            false
        }

        //OnAttachStateChange is here as a workaround because OnCloseListener doesn't fire
        searchView?.addOnAttachStateChangeListener(searchViewListener)

        menu.findItem(R.id.mnu_show_empty_aisles).apply { isChecked = showEmptyAisles }
        menu.findItem(R.id.mnu_edit_shop).apply {
            isVisible = shoppingListViewModel.locationType == LocationType.SHOP
        }

        loyaltyCardMenuItem = menu.findItem(R.id.mnu_show_loyalty_card)
        setMenuItemVisibility()
    }

    override fun onMenuItemSelected(menuItem: MenuItem): Boolean {
        //NOTE: If you override onMenuItemSelected, OnSupportNavigateUp will only be called when returning false
        return when (menuItem.itemId) {
            R.id.mnu_edit_shop -> {
                navigateToEditShop(locationId = shoppingListViewModel.locationId)
                true
            }

            R.id.mnu_sort_list_by_name -> {
                confirmSort(requireContext())
                true
            }

            R.id.mnu_show_loyalty_card -> {
                shoppingListViewModel.loyaltyCard?.let { showLoyaltyCard(it) }
                true
            }

            R.id.mnu_show_empty_aisles -> {
                shoppingListPreferences.setShowEmptyAisles(requireContext(), !showEmptyAisles)
                menuItem.isChecked = showEmptyAisles
                shoppingListViewModel.setShowEmptyAisles(showEmptyAisles)
                true
            }

            R.id.mnu_expand_collapse_aisles -> {
                shoppingListViewModel.expandCollapseAisles()
                true
            }

            else -> false
        }
    }

    private fun showLoyaltyCard(loyaltyCard: LoyaltyCard) {
        try {
            loyaltyCardProvider.displayLoyaltyCard(requireContext(), loyaltyCard)
        } catch (_: AisleronException.LoyaltyCardProviderException) {
            loyaltyCardProvider.showNotInstalledDialog(requireContext())
        } catch (e: Exception) {
            displayErrorSnackBar(
                AisleronException.ExceptionCode.GENERIC_EXCEPTION,
                e.message,
                fabHandler.getFabView(this.requireActivity())
            )
        }
    }

    private fun confirmSort(context: Context) {
        val builder = MaterialAlertDialogBuilder(context)
        builder
            .setTitle(getString(R.string.sort_confirm_title))
            .setMessage(R.string.sort_confirm_message)
            .setNegativeButton(android.R.string.cancel, null)
            .setPositiveButton(android.R.string.ok) { _, _ ->
                shoppingListViewModel.sortListByName()
            }

        val dialog: AlertDialog = builder.create()

        dialog.show()
    }

    override fun fabClicked(fabOption: FabHandler.FabOption) {
        actionMode?.finish()
    }

    fun hasSelectedItems(): Boolean = shoppingListViewModel.hasSelectedItems()

    companion object {

        private const val ARG_LOCATION_ID = "locationId"
        private const val ARG_FILTER_TYPE = "filterType"

        const val ADD_AISLE_REQUEST_KEY = "shoppingListAddAisleRequest"
        const val EDIT_AISLE_REQUEST_KEY = "shoppingListEditAisleRequest"

        const val AISLE_PICKER_REQUEST_KEY = "shoppingListAislePickerRequest"

        @JvmStatic
        fun newInstance(
            locationId: Long,
            filterType: FilterType,
            applicationTitleUpdateListener: ApplicationTitleUpdateListener,
            fabHandler: FabHandler,
            shoppingListPreferences: ShoppingListPreferences,
            loyaltyCardProvider: LoyaltyCardProvider
        ) =
            ShoppingListFragment(
                applicationTitleUpdateListener,
                fabHandler,
                shoppingListPreferences,
                loyaltyCardProvider
            ).apply {
                arguments = Bundle().apply {
                    putInt(ARG_LOCATION_ID, locationId.toInt())
                    putSerializable(ARG_FILTER_TYPE, filterType)
                }
            }
    }
}