/*
 * Copyright (C) 2025 aisleron.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package com.aisleron.ui.shoppinglist

import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.view.View
import android.widget.EditText
import androidx.appcompat.view.menu.ActionMenuItem
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.updatePadding
import androidx.fragment.app.testing.FragmentScenario
import androidx.fragment.app.testing.launchFragmentInContainer
import androidx.navigation.Navigation
import androidx.navigation.testing.TestNavHostController
import androidx.test.core.app.ActivityScenario
import androidx.test.core.app.ApplicationProvider
import androidx.test.espresso.Espresso.onView
import androidx.test.espresso.Espresso.openActionBarOverflowOrOptionsMenu
import androidx.test.espresso.Espresso.pressBack
import androidx.test.espresso.UiController
import androidx.test.espresso.ViewAction
import androidx.test.espresso.ViewAssertion
import androidx.test.espresso.ViewInteraction
import androidx.test.espresso.action.ViewActions
import androidx.test.espresso.action.ViewActions.clearText
import androidx.test.espresso.action.ViewActions.click
import androidx.test.espresso.action.ViewActions.longClick
import androidx.test.espresso.action.ViewActions.typeText
import androidx.test.espresso.assertion.ViewAssertions.doesNotExist
import androidx.test.espresso.assertion.ViewAssertions.matches
import androidx.test.espresso.matcher.RootMatchers.isDialog
import androidx.test.espresso.matcher.ViewMatchers
import androidx.test.espresso.matcher.ViewMatchers.hasDescendant
import androidx.test.espresso.matcher.ViewMatchers.hasSibling
import androidx.test.espresso.matcher.ViewMatchers.isAssignableFrom
import androidx.test.espresso.matcher.ViewMatchers.isDescendantOfA
import androidx.test.espresso.matcher.ViewMatchers.isDisplayed
import androidx.test.espresso.matcher.ViewMatchers.isSelected
import androidx.test.espresso.matcher.ViewMatchers.withClassName
import androidx.test.espresso.matcher.ViewMatchers.withHint
import androidx.test.espresso.matcher.ViewMatchers.withId
import androidx.test.espresso.matcher.ViewMatchers.withText
import androidx.test.platform.app.InstrumentationRegistry.getInstrumentation
import com.aisleron.AppCompatActivityTestImpl
import com.aisleron.R
import com.aisleron.di.KoinTestRule
import com.aisleron.di.daoTestModule
import com.aisleron.di.generalTestModule
import com.aisleron.di.repositoryModule
import com.aisleron.di.useCaseModule
import com.aisleron.di.viewModelTestModule
import com.aisleron.domain.FilterType
import com.aisleron.domain.aisle.Aisle
import com.aisleron.domain.aisle.AisleRepository
import com.aisleron.domain.aisleproduct.AisleProductRepository
import com.aisleron.domain.location.Location
import com.aisleron.domain.location.LocationRepository
import com.aisleron.domain.location.LocationType
import com.aisleron.domain.loyaltycard.LoyaltyCard
import com.aisleron.domain.loyaltycard.LoyaltyCardProviderType
import com.aisleron.domain.loyaltycard.LoyaltyCardRepository
import com.aisleron.domain.product.Product
import com.aisleron.domain.product.ProductRepository
import com.aisleron.domain.product.TrackingMode
import com.aisleron.domain.sampledata.usecase.CreateSampleDataUseCase
import com.aisleron.ui.ApplicationTitleUpdateListenerTestImpl
import com.aisleron.ui.FabHandler
import com.aisleron.ui.FabHandlerTestImpl
import com.aisleron.ui.aisle.AisleDialogFragment
import com.aisleron.ui.aisle.AislePickerDialogFragment
import com.aisleron.ui.bundles.AddEditLocationBundle
import com.aisleron.ui.bundles.AddEditProductBundle
import com.aisleron.ui.bundles.Bundler
import com.aisleron.ui.loyaltycard.LoyaltyCardProvider
import com.aisleron.ui.loyaltycard.LoyaltyCardProviderTestImpl
import com.aisleron.ui.settings.ShoppingListPreferencesTestImpl
import com.aisleron.utils.SystemIds
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.test.runTest
import org.hamcrest.CoreMatchers.allOf
import org.hamcrest.CoreMatchers.endsWith
import org.hamcrest.CoreMatchers.instanceOf
import org.hamcrest.CoreMatchers.not
import org.hamcrest.Matcher
import org.junit.Assert
import org.junit.Assert.assertEquals
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.koin.test.KoinTest
import org.koin.test.get
import java.text.DecimalFormat
import kotlin.test.assertFalse
import kotlin.test.assertNotEquals
import kotlin.test.assertNull
import kotlin.test.assertTrue

class ShoppingListFragmentTest : KoinTest {
    private lateinit var bundler: Bundler
    private lateinit var applicationTitleUpdateListener: ApplicationTitleUpdateListenerTestImpl
    private lateinit var fabHandler: FabHandlerTestImpl
    private lateinit var activityFragment: ShoppingListFragment

    @get:Rule
    val koinTestRule = KoinTestRule(
        modules = listOf(
            daoTestModule,
            viewModelTestModule,
            repositoryModule,
            useCaseModule,
            generalTestModule
        )
    )

    private fun setPadding(view: View) {
        ViewCompat.getRootWindowInsets(view)?.let { windowInsets ->
            val actionBarHeight = view.resources.getDimensionPixelSize(R.dimen.toolbar_height)
            val insets = windowInsets.getInsets(WindowInsetsCompat.Type.systemBars())
            view.updatePadding(top = actionBarHeight + insets.top)
        }
    }

    private fun getActivityScenario(
        bundle: Bundle,
        shoppingListPreferencesTestImpl: ShoppingListPreferencesTestImpl? = null,
        loyaltyCardProvider: LoyaltyCardProvider? = null
    ): ActivityScenario<AppCompatActivityTestImpl> {
        val intent = Intent().apply {
            setClassName("com.aisleron.debug", "com.aisleron.AppCompatActivityTestImpl")
            putExtra("fragment_args", bundle)
        }

        val scenario = ActivityScenario.launch<AppCompatActivityTestImpl>(intent)
        scenario.onActivity { activity ->

            val fragmentArgs = activity.intent.getBundleExtra("fragment_args") ?: Bundle()

            activityFragment = ShoppingListFragment(
                applicationTitleUpdateListener,
                fabHandler,
                shoppingListPreferencesTestImpl ?: ShoppingListPreferencesTestImpl(),
                loyaltyCardProvider ?: get<LoyaltyCardProvider>()
            ).apply {
                arguments = fragmentArgs
            }

            activity.supportFragmentManager.beginTransaction()
                .replace(android.R.id.content, activityFragment, "SHOPPING_LIST")
                .commitNow()

            setPadding(activityFragment.requireView())
        }

        return scenario
    }

    private fun getFragmentScenario(
        bundle: Bundle,
        shoppingListPreferencesTestImpl: ShoppingListPreferencesTestImpl? = null,
        loyaltyCardProvider: LoyaltyCardProvider? = null
    ): FragmentScenario<ShoppingListFragment> {
        val fragmentScenario = launchFragmentInContainer<ShoppingListFragment>(
            fragmentArgs = bundle,
            themeResId = R.style.Theme_Aisleron,
            instantiate = {
                ShoppingListFragment(
                    applicationTitleUpdateListener,
                    fabHandler,
                    shoppingListPreferencesTestImpl ?: ShoppingListPreferencesTestImpl(),
                    loyaltyCardProvider ?: get<LoyaltyCardProvider>()
                )
            }
        )

        fragmentScenario.onFragment {
            setPadding(it.requireView())
        }

        return fragmentScenario
    }

    private suspend fun getLocation(locationType: LocationType): Location =
        get<LocationRepository>().getAll().first { it.type == locationType }

    @Before
    fun setUp() {
        bundler = Bundler()
        applicationTitleUpdateListener = ApplicationTitleUpdateListenerTestImpl()
        fabHandler = FabHandlerTestImpl()
        runBlocking { get<CreateSampleDataUseCase>().invoke() }
    }

    @Test
    fun newInstance_CallNewInstance_ReturnsFragment() {
        val fragment =
            ShoppingListFragment.newInstance(
                1,
                FilterType.ALL,
                applicationTitleUpdateListener,
                fabHandler,
                ShoppingListPreferencesTestImpl(),
                LoyaltyCardProviderTestImpl()
            )
        Assert.assertNotNull(fragment)
    }

    @Test
    fun onCreateShoppingListFragment_HomeFilterIsInStock_AppTitleIsInStock() = runTest {
        val location = getLocation(LocationType.HOME)
        val bundle = bundler.makeShoppingListBundle(location.id, FilterType.IN_STOCK)
        val scenario = getFragmentScenario(bundle)
        scenario.onFragment {
            assertEquals(
                it.getString(R.string.menu_in_stock),
                applicationTitleUpdateListener.appTitle
            )
        }
    }

    @Test
    fun onCreateShoppingListFragment_HomeFilterIsNeeded_AppTitleIsNeeded() = runTest {
        val location = getLocation(LocationType.HOME)
        val bundle = bundler.makeShoppingListBundle(location.id, FilterType.NEEDED)
        val scenario = getFragmentScenario(bundle)
        scenario.onFragment {
            assertEquals(
                it.getString(R.string.menu_needed),
                applicationTitleUpdateListener.appTitle
            )
        }
    }

    @Test
    fun onCreateShoppingListFragment_HomeFilterIsAll_AppTitleIsShoppingList() = runTest {
        val location = getLocation(LocationType.HOME)
        val bundle = bundler.makeShoppingListBundle(location.id, FilterType.ALL)
        val scenario = getFragmentScenario(bundle)
        scenario.onFragment {
            assertEquals(
                it.getString(R.string.menu_all_items),
                applicationTitleUpdateListener.appTitle
            )
        }
    }

    @Test
    fun onCreateShoppingListFragment_LocationTypeIsShop_AppTitleIsShopName() = runTest {
        val location = getLocation(LocationType.SHOP)
        val bundle = bundler.makeShoppingListBundle(location.id, location.defaultFilter)
        val scenario = getFragmentScenario(bundle)
        scenario.onFragment {
            assertEquals(
                location.name,
                applicationTitleUpdateListener.appTitle
            )
        }
    }

    @Test
    fun onCreateShoppingListFragment_BundleIsAttributes_FragmentCreated() = runTest {
        val location = getLocation(LocationType.HOME)
        val bundle = bundler.makeShoppingListBundle(location.id, location.defaultFilter)
        val scenario = getFragmentScenario(bundle)
        scenario.onFragment {
            assertEquals(
                it.getString(R.string.menu_needed),
                applicationTitleUpdateListener.appTitle
            )
        }
    }

    @Test
    fun onCreateShoppingListFragment_ListIsEmpty_ShowEmptyListItem() = runTest {
        val location = Location(
            id = 0,
            type = LocationType.SHOP,
            defaultFilter = FilterType.NEEDED,
            name = "No Aisle Shop",
            pinned = false,
            aisles = emptyList(),
            showDefaultAisle = false
        )

        val locationId = get<LocationRepository>().add(location)
        getFragmentScenario(bundler.makeShoppingListBundle(locationId, location.defaultFilter))

        onView(withText(R.string.empty_list_title)).check(matches(isDisplayed()))
    }

    private suspend fun getShoppingList(locationId: Int? = null): Location {
        val locationRepository = get<LocationRepository>()
        val shopId =
            locationId ?: locationRepository.getAll().first { it.type != LocationType.HOME }.id
        return locationRepository.getLocationWithAislesWithProducts(shopId).first()!!
    }

    private fun validateToolbarItem(menuId: Int, isVisible: Boolean): ViewAssertion =
        if (isVisible)
            matches(hasDescendant(withId(menuId)))
        else
            matches(not(hasDescendant(withId(menuId))))

    private fun validateOverflowItem(isVisible: Boolean): ViewAssertion =
        if (isVisible)
            matches(isDisplayed())
        else
            doesNotExist()

    private fun validateActionModeMenuItems(
        actionModeTitle: String,
        showAddProductToAisle: Boolean = false,
        showEditShoppingListItem: Boolean = false,
        showProductNote: Boolean = false,
        showAislePicker: Boolean = false,
        showDelete: Boolean = false,
        showCopy: Boolean = false
    ) {
        val actionBar = onContextualActionBar()
        actionBar.check(matches(isDisplayed()))
        actionBar.check(matches(hasDescendant(withText(actionModeTitle))))

        actionBar.check(validateToolbarItem(R.id.mnu_add_product_to_aisle, showAddProductToAisle))
        actionBar.check(
            validateToolbarItem(
                R.id.mnu_edit_shopping_list_item, showEditShoppingListItem
            )
        )

        actionBar.check(validateToolbarItem(R.id.mnu_product_note, showProductNote))
        actionBar.check(validateToolbarItem(R.id.mnu_aisle_picker, showAislePicker))

        openActionBarOverflowOrOptionsMenu(getInstrumentation().targetContext)
        onView(withText(R.string.delete)).check(validateOverflowItem(showDelete))
        onView(withText(android.R.string.copy)).check(validateOverflowItem(showCopy))
    }

    @Test
    fun onLongClick_IsAisle_ShowAisleActionModeContextMenu() = runTest {
        val shoppingList = getShoppingList()
        getActivityScenario(
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        )

        val aisle = getAisle(shoppingList, isDefault = false, productsInStock = false)

        onView(withText(aisle.name)).perform(longClick())

        validateActionModeMenuItems(
            actionModeTitle = aisle.name,
            showAddProductToAisle = true,
            showEditShoppingListItem = true,
            showDelete = true
        )
    }

    @Test
    fun onLongClick_IsProduct_ShowProductActionModeContextMenu() = runTest {
        val shoppingList = getShoppingList()
        getActivityScenario(
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        )

        val product = getProduct(shoppingList, false)

        onView(withText(product.name)).perform(longClick())

        validateActionModeMenuItems(
            actionModeTitle = product.name,
            showEditShoppingListItem = true,
            showProductNote = true,
            showAislePicker = true,
            showCopy = true,
            showDelete = true
        )
    }

    @Test
    fun onSelectMode_MultipleProductsSelected_ShowCorrectActionModeOptions() = runTest {
        val shoppingList = getShoppingList()
        var actionModeTitle = ""
        getActivityScenario(
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        ).onActivity {
            actionModeTitle = it.getString(R.string.products_selected, 2)
        }

        val product1 = getProduct(shoppingList, false)
        val product2 = getProduct(shoppingList, false, product1.id)

        onView(withText(product1.name)).perform(longClick())
        onView(withText(product2.name)).perform(click())

        validateActionModeMenuItems(
            actionModeTitle = actionModeTitle,
            showAislePicker = true,
            showDelete = true
        )
    }

    @Test
    fun onSelectMode_MultipleAisleSelected_ShowCorrectActionModeOptions() = runTest {
        val shoppingList = getShoppingList()
        var actionModeTitle = ""
        getActivityScenario(
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        ).onActivity {
            actionModeTitle = it.getString(R.string.aisles_selected, 2)
        }

        val aisle1 = getAisle(shoppingList, isDefault = false, productsInStock = false)
        val aisle2 = getAisle(
            shoppingList, isDefault = false, productsInStock = false, excludeId = aisle1.id
        )

        val aisleRepository = get<AisleRepository>()
        val aisle2ExpandedBefore = aisleRepository.get(aisle2.id)!!

        onView(withText(aisle1.name)).perform(longClick())
        onView(withText(aisle2.name)).perform(click())

        validateActionModeMenuItems(
            actionModeTitle = actionModeTitle,
            showDelete = true
        )

        // Validate that clicking on an aisle during selection mode doesn't toggle aisle expansion
        assertEquals(aisle2ExpandedBefore.expanded, aisleRepository.get(aisle2.id)!!.expanded)
    }

    @Test
    fun onSelectMode_MixedItemsSelected_ShowCorrectActionModeOptions() = runTest {
        val shoppingList = getShoppingList()
        var actionModeTitle = ""
        getActivityScenario(
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        ).onActivity {
            actionModeTitle = it.getString(R.string.items_selected, 2)
        }

        val aisle = getAisle(shoppingList, isDefault = false, productsInStock = false)
        val product = getProduct(shoppingList, false)

        onView(withText(aisle.name)).perform(longClick())
        onView(withText(product.name)).perform(click())

        validateActionModeMenuItems(
            actionModeTitle = actionModeTitle,
            showDelete = true
        )
    }

    @Test
    fun onClick_ActionModeIsActive_DismissActionModeContextMenu() = runTest {
        val shoppingList = getShoppingList()
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val scenario = getActivityScenario(bundle)
        val product = getProduct(shoppingList, false)

        val productItem = onView(allOf(withText(product.name), withId(R.id.txt_product_name)))
        productItem.perform(longClick())
        productItem.perform(click())

        val actionBar = onContextualActionBar()
        actionBar.check(matches(not(isDisplayed())))
        scenario.onActivity {
            assertEquals(shoppingList.name, applicationTitleUpdateListener.appTitle)
            assertFalse(activityFragment.hasSelectedItems())
        }
    }

    @Test
    fun onBackPress_ActionModeIsActive_DismissActionModeContextMenu() = runTest {
        val shoppingList = getShoppingList()
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val scenario = getActivityScenario(bundle)
        val product = getProduct(shoppingList, false)

        val productItem = onView(allOf(withText(product.name), withId(R.id.txt_product_name)))
        productItem.perform(longClick())
        pressBack()

        val actionBar = onContextualActionBar()
        actionBar.check(matches(not(isDisplayed())))
        productItem.check(matches(not(isSelected())))

        scenario.onActivity {
            assertEquals(shoppingList.name, applicationTitleUpdateListener.appTitle)
            assertFalse(activityFragment.hasSelectedItems())
        }
    }

    @Test
    fun onActionItemClicked_DeleteConfirmedOnProduct_ProductDeleted() = runTest {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false)
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)

        val scenario = getActivityScenario(bundle)

        val productItem = onView(allOf(withText(product.name), withId(R.id.txt_product_name)))
        productItem.perform(longClick())
        openActionBarOverflowOrOptionsMenu(getInstrumentation().targetContext)
        onView(withText(R.string.delete)).perform(click())
        onView(withText(android.R.string.ok))
            .inRoot(isDialog())
            .perform(click())

        val deletedProduct = get<ProductRepository>().getByName(product.name)
        Assert.assertNull(deletedProduct)

        scenario.onActivity {
            assertFalse(activityFragment.hasSelectedItems())
        }
    }

    @Test
    fun onActionItemClicked_DeleteConfirmedOnAisle_AisleDeleted() = runTest {
        val shoppingList = getShoppingList()
        val aisle = getAisle(shoppingList, isDefault = false, productsInStock = false)
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)

        val scenario = getActivityScenario(bundle)
        val aisleItem = onView(allOf(withText(aisle.name), withId(R.id.txt_aisle_name)))
        aisleItem.perform(longClick())
        openActionBarOverflowOrOptionsMenu(getInstrumentation().targetContext)
        onView(withText(R.string.delete)).perform(click())
        onView(withText(android.R.string.ok))
            .inRoot(isDialog())
            .perform(click())

        val deletedAisle = get<AisleRepository>().get(aisle.id)
        Assert.assertNull(deletedAisle)

        scenario.onActivity {
            assertFalse(activityFragment.hasSelectedItems())
        }
    }

    @Test
    fun onActionItemClicked_DeleteConfirmedOnDefaultAisle_ErrorSnackBarShown() = runTest {
        val shoppingList = getShoppingList()
        val aisle = getAisle(shoppingList, isDefault = true, productsInStock = null)
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)

        val preferences = ShoppingListPreferencesTestImpl()
        preferences.setShowEmptyAisles(true)
        getActivityScenario(bundle, preferences)

        val aisleItem = onView(allOf(withText(aisle.name), withId(R.id.txt_aisle_name)))
        aisleItem.perform(longClick())
        openActionBarOverflowOrOptionsMenu(getInstrumentation().targetContext)
        onView(withText(R.string.delete)).perform(click())
        onView(withText(android.R.string.ok))
            .inRoot(isDialog())
            .perform(click())

        onView(withId(SystemIds.SNACKBAR_TEXT)).check(
            matches(
                ViewMatchers.withEffectiveVisibility(
                    ViewMatchers.Visibility.VISIBLE
                )
            )
        )
    }

    private fun getAisle(
        shoppingList: Location, isDefault: Boolean, productsInStock: Boolean?, excludeId: Int = -1
    ): Aisle {
        return shoppingList.aisles.first {
            it.isDefault == isDefault && it.id != excludeId && (productsInStock == null ||
                    it.products.count { ap -> ap.product.inStock == productsInStock } > 0)
        }
    }

    @Test
    fun onActionItemClicked_DeleteCancelled_AisleNotDeleted() = runTest {
        val shoppingList = getShoppingList()
        val aisle = getAisle(shoppingList, isDefault = false, productsInStock = false)
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)

        val scenario = getActivityScenario(bundle)
        val aisleItem = onView(allOf(withText(aisle.name), withId(R.id.txt_aisle_name)))
        aisleItem.perform(longClick())
        openActionBarOverflowOrOptionsMenu(getInstrumentation().targetContext)
        onView(withText(R.string.delete)).perform(click())

        // Verify the delete dialog is shown
        var deleteConfirmMessage = ""
        scenario.onActivity {
            deleteConfirmMessage =
                activityFragment.getString(R.string.delete_confirmation, aisle.name)
        }
        onView(withText(deleteConfirmMessage))
            .inRoot(isDialog())
            .check(matches(isDisplayed()))

        // Cancel the delete dialog
        onView(withText(android.R.string.cancel))
            .inRoot(isDialog())
            .perform(click())

        val deletedAisle = get<AisleRepository>().get(aisle.id)
        Assert.assertNotNull(deletedAisle)

        scenario.onActivity {
            // Action mode is not cancelled if delete is cancelled
            assertTrue(activityFragment.hasSelectedItems())
        }
    }

    @Test
    fun onActionItemClicked_ActionItemIsEditOnProduct_NavigateToEditProduct() = runTest {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false)
        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)

        val navController = TestNavHostController(ApplicationProvider.getApplicationContext())
        val scenario = getActivityScenario(shoppingListBundle)
        scenario.onActivity {
            navController.setGraph(R.navigation.mobile_navigation)
            navController.setCurrentDestination(R.id.nav_shopping_list)
            Navigation.setViewNavController(activityFragment.requireView(), navController)
        }

        val productItem = onView(allOf(withText(product.name), withId(R.id.txt_product_name)))
        productItem.perform(longClick())
        onView(withId(R.id.mnu_edit_shopping_list_item)).perform(click())

        val bundle = navController.backStack.last().arguments
        val addEditProductBundle = bundler.getAddEditProductBundle(bundle)

        assertEquals(product.id, addEditProductBundle.productId)
        assertEquals(AddEditProductBundle.ProductAction.EDIT, addEditProductBundle.actionType)
        assertEquals(R.id.nav_add_product, navController.currentDestination?.id)
    }

    @Test
    fun onProductStatusChange_SetProductInStock_ProductStatusToggled() = runTest {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false)
        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, FilterType.NEEDED)

        getFragmentScenario(shoppingListBundle)

        getCheckboxForProduct(product).perform(click())

        val updatedProduct = get<ProductRepository>().get(product.id)
        assertEquals(!product.inStock, updatedProduct?.inStock)
    }

    @Test
    fun onProductStatusChange_SetProductNeeded_ProductStatusToggled() = runTest {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, true)
        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, FilterType.IN_STOCK)

        getFragmentScenario(shoppingListBundle)

        getCheckboxForProduct(product).perform(click())

        val updatedProduct = get<ProductRepository>().get(product.id)
        assertEquals(!product.inStock, updatedProduct?.inStock)
    }

    @Test
    fun onSwipe_IsProduct_ProductStatusToggled() = runTest {
        val shoppingList = getShoppingList()
        val product =
            getProduct(shoppingList, false)
        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, FilterType.NEEDED)

        getFragmentScenario(shoppingListBundle)

        onView(
            allOf(
                withText(product.name),
                withId(R.id.txt_product_name)
            )
        ).perform(ViewActions.swipeLeft())

        val updatedProduct = get<ProductRepository>().get(product.id)
        assertEquals(!product.inStock, updatedProduct?.inStock)
    }

    private fun getProduct(shoppingList: Location, inStock: Boolean, excludeId: Int = -1): Product {
        return shoppingList.aisles
            .asSequence()
            .filterNot { it.isDefault }
            .flatMap { aisle ->
                aisle.products
                    .filter { it.product.inStock == inStock && it.product.id != excludeId }
                    .map { it.product }
            }
            .first()
    }

    @Test
    fun onActionItemClicked_ActionItemIsEditOnAisle_ShowAisleEditDialog() = runTest {
        val shoppingList = getShoppingList()
        val aisle = getAisle(shoppingList, isDefault = false, productsInStock = false)
        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)

        getActivityScenario(shoppingListBundle)

        val aisleItem = onView(allOf(withText(aisle.name), withId(R.id.txt_aisle_name)))
        aisleItem.perform(longClick())
        onView(withId(R.id.mnu_edit_shopping_list_item)).perform(click())

        onView(withText(R.string.edit_aisle))
            .inRoot(isDialog())
            .check(matches(isDisplayed()))

        onView(allOf(withText(aisle.name), instanceOf(EditText::class.java)))
            .inRoot(isDialog())
            .check(matches(isDisplayed()))
    }

    @Test
    fun onClickFab_IsAddProductFab_NavigateToAddProduct() = runTest {
        val shoppingList = getShoppingList()
        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)

        val navController = TestNavHostController(ApplicationProvider.getApplicationContext())
        getFragmentScenario(shoppingListBundle).onFragment { fragment ->
            navController.setGraph(R.navigation.mobile_navigation)
            navController.setCurrentDestination(R.id.nav_shopping_list)
            Navigation.setViewNavController(fragment.requireView(), navController)
            fabHandler.clickFab(FabHandler.FabOption.ADD_PRODUCT, fragment.requireView())
        }

        val bundle = navController.backStack.last().arguments
        val addEditProductBundle = bundler.getAddEditProductBundle(bundle)

        assertNull(addEditProductBundle.name)
        assertEquals(AddEditProductBundle.ProductAction.ADD, addEditProductBundle.actionType)
        assertEquals(
            shoppingList.defaultFilter == FilterType.IN_STOCK,
            addEditProductBundle.inStock
        )
        assertEquals(R.id.nav_add_product, navController.currentDestination?.id)
    }

    @Test
    fun onClickFab_IsAddAisleFab_ShowAddAisleDialog() = runTest {
        val shoppingList = getShoppingList()
        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)

        getFragmentScenario(shoppingListBundle).onFragment {
            fabHandler.clickFab(FabHandler.FabOption.ADD_AISLE, it.requireView())
        }

        onView(withText(R.string.add_aisle))
            .inRoot(isDialog())
            .check(matches(isDisplayed()))

        onView(withText(R.string.add_another))
            .inRoot(isDialog())
            .check(matches(isDisplayed()))

        onView(allOf(instanceOf(EditText::class.java)))
            .inRoot(isDialog())
            .check(matches(withText("")))
    }

    @Test
    fun onProductStatusChange_StatusUpdateSnackBarEnabled_ShowSnackBar() = runTest {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false)
        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, FilterType.NEEDED)

        val shoppingListPreferencesTestImpl = ShoppingListPreferencesTestImpl()
        shoppingListPreferencesTestImpl.setHideStatusChangeSnackBar(false)

        getFragmentScenario(shoppingListBundle, shoppingListPreferencesTestImpl)

        getCheckboxForProduct(product).perform(click())

        onView(withId(SystemIds.SNACKBAR_TEXT)).check(
            matches(
                ViewMatchers.withEffectiveVisibility(
                    ViewMatchers.Visibility.VISIBLE
                )
            )
        )
    }

    @Test
    fun onProductStatusChange_StatusUpdateSnackBarDisabled_HideSnackBar() = runTest {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false)
        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, FilterType.NEEDED)

        val shoppingListPreferencesTestImpl = ShoppingListPreferencesTestImpl()
        shoppingListPreferencesTestImpl.setHideStatusChangeSnackBar(true)

        getFragmentScenario(shoppingListBundle, shoppingListPreferencesTestImpl)

        getCheckboxForProduct(product).perform(click())

        onView(withId(SystemIds.SNACKBAR_TEXT)).check(doesNotExist())
    }

    @Test
    fun onProductStatusChange_StatusUpdateSnackBarUndoClicked_ProductStatusChanged() = runTest {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false)
        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, FilterType.NEEDED)

        val productStatusBefore = product.inStock

        val shoppingListPreferencesTestImpl = ShoppingListPreferencesTestImpl()
        shoppingListPreferencesTestImpl.setHideStatusChangeSnackBar(false)

        getFragmentScenario(shoppingListBundle, shoppingListPreferencesTestImpl)

        getCheckboxForProduct(product).perform(click())

        val productRepository = get<ProductRepository>()
        val productStatusAfterChange = productRepository.get(product.id)?.inStock

        onView(withId(SystemIds.SNACKBAR_ACTION)).perform(click())

        val productStatusAfterUndo = productRepository.get(product.id)?.inStock

        assertNotEquals(productStatusBefore, productStatusAfterChange)
        assertEquals(productStatusBefore, productStatusAfterUndo)
    }

    @Test
    fun onActionItemClicked_ActionItemIsAddProductToAisle_NavigateToAddProduct() = runTest {
        val shoppingList = getShoppingList()
        val aisle = getAisle(shoppingList, isDefault = false, productsInStock = false)
        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)

        val navController = TestNavHostController(ApplicationProvider.getApplicationContext())
        val scenario = getActivityScenario(shoppingListBundle)
        scenario.onActivity {
            navController.setGraph(R.navigation.mobile_navigation)
            navController.setCurrentDestination(R.id.nav_shopping_list)
            Navigation.setViewNavController(activityFragment.requireView(), navController)
        }

        onView(withText(aisle.name)).perform(longClick())
        onView(withId(R.id.mnu_add_product_to_aisle)).perform(click())

        assertEquals(R.id.nav_add_product, navController.currentDestination?.id)

        val bundle = navController.backStack.last().arguments
        val addEditProductBundle = bundler.getAddEditProductBundle(bundle)
        assertEquals(aisle.id, addEditProductBundle.aisleId)
        assertEquals(AddEditProductBundle.ProductAction.ADD, addEditProductBundle.actionType)
    }

    @Test
    fun onClick_OtherItemSelected_ClickedItemAddedToSelection() = runTest {
        val shoppingList = getShoppingList()
        val aisle = getAisle(shoppingList, isDefault = false, productsInStock = false)
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        getActivityScenario(bundle)
        val product = getProduct(shoppingList, false)

        val productItem = onView(allOf(withText(product.name), withId(R.id.txt_product_name)))
        productItem.perform(longClick())
        val aisleItem = onView(allOf(withText(aisle.name), withId(R.id.txt_aisle_name)))
        aisleItem.perform(click())

        productItem.check(matches((isSelected())))
        aisleItem.check(matches((isSelected())))
    }

    @Test
    fun onClickAddAisleFab_ActionModeIsActive_DismissActionModeContextMenu() = runTest {
        val shoppingList = getShoppingList()
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val scenario = getActivityScenario(bundle)
        val product = getProduct(shoppingList, false)
        val productItem = onView(allOf(withText(product.name), withId(R.id.txt_product_name)))

        productItem.perform(longClick())
        scenario.onActivity {
            fabHandler.clickFab(FabHandler.FabOption.ADD_AISLE, activityFragment.requireView())
        }

        onView(withText(android.R.string.cancel))
            .inRoot(isDialog())
            .perform(click())

        val actionBar = onContextualActionBar()
        actionBar.check(matches(not(isDisplayed())))
        scenario.onActivity {
            assertFalse(activityFragment.hasSelectedItems())
        }
    }

    @Test
    fun onClickAddProductFab_ActionModeIsActive_DismissActionModeContextMenu() = runTest {
        val shoppingList = getShoppingList()
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val scenario = getActivityScenario(bundle)
        val product = getProduct(shoppingList, false)
        val productItem = onView(allOf(withText(product.name), withId(R.id.txt_product_name)))

        productItem.perform(longClick())

        val navController = TestNavHostController(ApplicationProvider.getApplicationContext())
        scenario.onActivity {
            navController.setGraph(R.navigation.mobile_navigation)
            navController.setCurrentDestination(R.id.nav_shopping_list)
            Navigation.setViewNavController(activityFragment.requireView(), navController)
            fabHandler.clickFab(FabHandler.FabOption.ADD_PRODUCT, activityFragment.requireView())
        }

        onContextualActionBar().checkVisibility(View.GONE)
        scenario.onActivity {
            assertFalse(activityFragment.hasSelectedItems())
        }
    }

    @Test
    fun onClickAddShopFab_ActionModeIsActive_DismissActionModeContextMenu() = runTest {
        val shoppingList = getShoppingList()
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val scenario = getActivityScenario(bundle)
        val product = getProduct(shoppingList, false)
        val productItem = onView(allOf(withText(product.name), withId(R.id.txt_product_name)))

        productItem.perform(longClick())
        scenario.onActivity {
            fabHandler.setFabOnClickListener(
                activityFragment.requireActivity(),
                FabHandler.FabOption.ADD_SHOP
            ) {}
            fabHandler.clickFab(FabHandler.FabOption.ADD_SHOP, activityFragment.requireView())
        }

        onContextualActionBar().checkVisibility(View.GONE)
        scenario.onActivity {
            assertFalse(activityFragment.hasSelectedItems())
        }
    }

    private fun onContextualActionBar(): ViewInteraction =
        onView(withClassName(endsWith("ActionBarContextView")))

    @Test
    fun onLongClick_EarlierItemSelected_ActionBarHasNewItemTitle() = runTest {
        val shoppingList = getShoppingList()
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        getActivityScenario(bundle)
        val product = getProduct(shoppingList, false)
        val aisle = getAisle(shoppingList, isDefault = false, productsInStock = false)

        val productItem = onView(allOf(withText(product.name), withId(R.id.txt_product_name)))
        productItem.perform(longClick())
        val aisleItem = onView(allOf(withText(aisle.name), withId(R.id.txt_aisle_name)))
        aisleItem.perform(longClick())

        val actionBar = onContextualActionBar()
        actionBar.check(matches(hasDescendant(withText(aisle.name))))
    }

    private fun getMenuItem(resourceId: Int): ActionMenuItem {
        val context: Context = getInstrumentation().targetContext
        val menuItem = ActionMenuItem(context, 0, resourceId, 0, 0, null)
        return menuItem
    }

    @Test
    fun onMenuItemSelected_ItemIsEditShop_NavigateToEditShop() = runTest {
        val shoppingList = getShoppingList()
        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)

        val menuItem = getMenuItem(R.id.mnu_edit_shop)
        val navController = TestNavHostController(ApplicationProvider.getApplicationContext())
        getFragmentScenario(shoppingListBundle).onFragment { fragment ->
            navController.setGraph(R.navigation.mobile_navigation)
            navController.setCurrentDestination(R.id.nav_shopping_list)
            Navigation.setViewNavController(fragment.requireView(), navController)
            fragment.onMenuItemSelected(menuItem)
        }

        val bundle = navController.backStack.last().arguments
        val addEditShopBundle = bundler.getAddEditLocationBundle(bundle)

        assertEquals(shoppingList.id, addEditShopBundle.locationId)
        assertEquals(AddEditLocationBundle.LocationAction.EDIT, addEditShopBundle.actionType)
        assertEquals(R.id.nav_add_shop, navController.currentDestination?.id)
    }

    @Test
    fun onMenuItemSelected_ItemIsSortByName_SortConfirmDialogShown() = runTest {
        val shoppingList = getShoppingList()
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val menuItem = getMenuItem(R.id.mnu_sort_list_by_name)
        var confirmMessage = ""

        getFragmentScenario(bundle).onFragment { fragment ->
            confirmMessage = fragment.getString(R.string.sort_confirm_title)
            fragment.onMenuItemSelected(menuItem)
        }

        onView(withText(confirmMessage))
            .inRoot(isDialog())
            .check(matches(isDisplayed()))
    }

    @Test
    fun onMenuItemSelected_SortCancelled_ListNotReordered() = runTest {
        val locationId = getShoppingList().id
        val aisleRepository = get<AisleRepository>()
        val rankBefore = 2002
        val aisleId = aisleRepository.add(
            Aisle(
                name = "AAA",
                products = emptyList(),
                locationId = locationId,
                rank = rankBefore,
                id = 0,
                isDefault = false,
                expanded = true
            )
        )

        val shoppingList = getShoppingList(locationId)
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val menuItem = getMenuItem(R.id.mnu_sort_list_by_name)

        getFragmentScenario(bundle).onFragment { fragment ->
            fragment.onMenuItemSelected(menuItem)
        }

        onView(withText(android.R.string.cancel))
            .inRoot(isDialog())
            .perform(click())

        val reorderedAisle = aisleRepository.get(aisleId)
        assertEquals(rankBefore, reorderedAisle?.rank)
    }

    @Test
    fun onMenuItemSelected_SortConfirmed_ListIsReordered() = runTest {
        val locationId = getShoppingList().id
        val aisleRepository = get<AisleRepository>()
        val rankBefore = 2002
        val aisleId = aisleRepository.add(
            Aisle(
                name = "AAA",
                products = emptyList(),
                locationId = locationId,
                rank = rankBefore,
                id = 0,
                isDefault = false,
                expanded = true
            )
        )

        val shoppingList = getShoppingList(locationId)
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val menuItem = getMenuItem(R.id.mnu_sort_list_by_name)

        getFragmentScenario(bundle).onFragment { fragment ->
            fragment.onMenuItemSelected(menuItem)
        }

        onView(withText(android.R.string.ok))
            .inRoot(isDialog())
            .perform(click())

        val reorderedAisle = aisleRepository.get(aisleId)
        assertEquals(1, reorderedAisle?.rank)
    }

    private suspend fun getShoppingListWithLoyaltyCard(): Location {
        val shoppingList = getShoppingList()
        val loyaltyCardRepository = get<LoyaltyCardRepository>()
        val loyaltyCardId = loyaltyCardRepository.add(
            LoyaltyCard(
                id = 0,
                name = "Test Card",
                provider = LoyaltyCardProviderType.CATIMA,
                intent = "Dummy Intent"
            )
        )

        loyaltyCardRepository.addToLocation(shoppingList.id, loyaltyCardId)
        return shoppingList
    }

    @Test
    fun onMenuItemSelected_ItemIsShowLoyaltyCardAndHasLoyaltyCard_ShowLoyaltyCard() = runTest {
        val shoppingList = getShoppingListWithLoyaltyCard()
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val menuItem = getMenuItem(R.id.mnu_show_loyalty_card)
        val loyaltyCardProvider = LoyaltyCardProviderTestImpl()

        getFragmentScenario(
            bundle, loyaltyCardProvider = loyaltyCardProvider
        ).onFragment { fragment ->
            fragment.onMenuItemSelected(menuItem)
        }

        assertTrue { loyaltyCardProvider.loyaltyCardDisplayed }
    }

    @Test
    fun onMenuItemSelected_ItemIsShowLoyaltyCardAndNoLoyaltyCard_LoyaltyCardNotShown() = runTest {
        val shoppingList = getShoppingList()
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val menuItem = getMenuItem(R.id.mnu_show_loyalty_card)
        val loyaltyCardProvider = LoyaltyCardProviderTestImpl()

        getFragmentScenario(
            bundle, loyaltyCardProvider = loyaltyCardProvider
        ).onFragment { fragment ->
            fragment.onMenuItemSelected(menuItem)
        }

        assertFalse { loyaltyCardProvider.loyaltyCardDisplayed }
    }


    @Test
    fun onMenuItemSelected_ItemIsShowLoyaltyCardAndNoProvider_ShowNotInstalledDialog() = runTest {
        val shoppingList = getShoppingListWithLoyaltyCard()
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val menuItem = getMenuItem(R.id.mnu_show_loyalty_card)
        val loyaltyCardProvider = LoyaltyCardProviderTestImpl(throwNotInstalledException = true)

        getFragmentScenario(
            bundle, loyaltyCardProvider = loyaltyCardProvider
        ).onFragment { fragment ->
            fragment.onMenuItemSelected(menuItem)
        }

        onView(withText(R.string.loyalty_card_provider_missing_title))
            .inRoot(isDialog())
            .check(matches(isDisplayed()))
    }

    @Test
    fun onMenuItemSelected_ItemIsShowLoyaltyCardAndGenericError_ShowErrorSnackBar() = runTest {
        val shoppingList = getShoppingListWithLoyaltyCard()
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val menuItem = getMenuItem(R.id.mnu_show_loyalty_card)
        val loyaltyCardProvider = LoyaltyCardProviderTestImpl(throwGenericException = true)

        getFragmentScenario(
            bundle, loyaltyCardProvider = loyaltyCardProvider
        ).onFragment { fragment ->
            fragment.onMenuItemSelected(menuItem)
        }

        onView(withId(SystemIds.SNACKBAR_TEXT)).check(
            matches(
                ViewMatchers.withEffectiveVisibility(
                    ViewMatchers.Visibility.VISIBLE
                )
            )
        )
    }

    private fun ViewInteraction.checkVisibility(
        expectedVisibility: Int,
        timeoutMs: Long = 2000,
        pollIntervalMs: Long = 50
    ) {
        this.perform(object : ViewAction {
            override fun getConstraints(): Matcher<View> = isAssignableFrom(View::class.java)

            override fun getDescription(): String =
                "wait up to $timeoutMs ms for view to have visibility: $expectedVisibility"

            override fun perform(uiController: UiController, view: View) {
                val endTime = System.currentTimeMillis() + timeoutMs
                do {
                    if (view.visibility == expectedVisibility) return
                    uiController.loopMainThreadForAtLeast(pollIntervalMs)
                } while (System.currentTimeMillis() < endTime)

                throw AssertionError("View did not become visibility=$expectedVisibility within $timeoutMs ms")
            }
        })
    }

    @Test
    fun onMenuItemSelected_ItemIsShowEmptyAisles_EmptyAislePreferenceUpdated() = runTest {
        val shoppingList = getShoppingList()
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val menuItem = getMenuItem(R.id.mnu_show_empty_aisles)
        val shoppingListPreferencesTestImpl = ShoppingListPreferencesTestImpl()
        val valueBefore = false
        shoppingListPreferencesTestImpl.setShowEmptyAisles(valueBefore)

        getFragmentScenario(bundle, shoppingListPreferencesTestImpl).onFragment { fragment ->
            fragment.onMenuItemSelected(menuItem)
        }

        val valueAfter =
            shoppingListPreferencesTestImpl.showEmptyAisles(getInstrumentation().targetContext)

        assertNotEquals(valueBefore, valueAfter)
    }

    @Test
    fun onMenuItemSelected_ItemExpandCollapseAisles_aisleExpansionToggled() = runTest {
        val location = get<LocationRepository>().getHome()
        val aisleRepository = get<AisleRepository>()
        val expandedBefore =
            aisleRepository.getForLocation(location.id).count { it.expanded }

        val bundle = bundler.makeShoppingListBundle(location.id, location.defaultFilter)
        getFragmentScenario(bundle).onFragment { fragment ->
            val menuItem = getMenuItem(R.id.mnu_expand_collapse_aisles)
            fragment.onMenuItemSelected(menuItem)
        }

        val expandedAfter =
            aisleRepository.getForLocation(location.id).count { it.expanded }

        assertTrue(expandedBefore > expandedAfter)
    }

    @Test
    fun onActionItemClicked_CopyConfirmed_ConfirmSnackbarShown() = runTest {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false)
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val scenario = getActivityScenario(bundle)
        var copyDialogTitle = ""
        var confirmCopy = ""

        scenario.onActivity {
            confirmCopy = activityFragment.getString(R.string.entity_copied, product.name)
            copyDialogTitle =
                activityFragment.getString(R.string.copy_entity_title, product.name)
        }

        onView(withText(product.name)).perform(longClick())
        openActionBarOverflowOrOptionsMenu(getInstrumentation().targetContext)
        onView(withText(android.R.string.copy)).perform(click())

        onView(withText(android.R.string.ok))
            .inRoot(isDialog())
            .perform(click())

        onView(withText(copyDialogTitle))
            .check(doesNotExist())

        val snackbar = onView(withId(SystemIds.SNACKBAR_TEXT))
        snackbar.checkVisibility(View.VISIBLE)
        snackbar.check(matches(withText(confirmCopy)))

        scenario.onActivity {
            // Action mode is dismissed after copy is confirmed
            assertFalse(activityFragment.hasSelectedItems())
        }
    }

    @Test
    fun onActionItemClicked_CopyCancelled_DialogClosed() = runTest {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false)
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val scenario = getActivityScenario(bundle)
        var copyDialogTitle = ""

        scenario.onActivity {
            copyDialogTitle =
                activityFragment.getString(R.string.copy_entity_title, product.name)
        }

        onView(withText(product.name)).perform(longClick())
        openActionBarOverflowOrOptionsMenu(getInstrumentation().targetContext)
        onView(withText(android.R.string.copy)).perform(click())

        // Verify that the Copy dialog is shown
        onView(withText(copyDialogTitle))
            .inRoot(isDialog())
            .check(matches(isDisplayed()))

        // Cancel the Copy dialog
        onView(withText(android.R.string.cancel))
            .inRoot(isDialog())
            .perform(click())

        onView(withText(copyDialogTitle))
            .check(doesNotExist())

        scenario.onActivity {
            // Action mode is not dismissed if copy is cancelled
            assertTrue(activityFragment.hasSelectedItems())
        }
    }

    private suspend fun onCreateView_TrackingMode_ArrangeAct(trackingMode: TrackingMode): Product {
        val shoppingListPrefs = ShoppingListPreferencesTestImpl()
        shoppingListPrefs.setTrackingMode(trackingMode)
        val shoppingList = getShoppingList()
        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)

        getFragmentScenario(shoppingListBundle, shoppingListPrefs)

        return getProduct(shoppingList, false)
    }

    @Test
    fun onCreateView_TrackingModeIsCheckbox_OnlyCheckBoxShown() = runTest {
        val product =
            onCreateView_TrackingMode_ArrangeAct(TrackingMode.CHECKBOX)

        getCheckboxForProduct(product).check(matches(isDisplayed()))
        getQtyStepperForProduct(product).check(matches(not(isDisplayed())))
    }

    @Test
    fun onCreateView_TrackingModeIsNone_NothingShown() = runTest {
        val product =
            onCreateView_TrackingMode_ArrangeAct(TrackingMode.NONE)

        getCheckboxForProduct(product).check(matches(not(isDisplayed())))
        getQtyStepperForProduct(product).check(matches(not(isDisplayed())))
    }

    @Test
    fun onCreateView_TrackingModeIsQuantity_QuantityShown() = runTest {
        val product =
            onCreateView_TrackingMode_ArrangeAct(TrackingMode.QUANTITY)

        getCheckboxForProduct(product).check(matches(not(isDisplayed())))
        getQtyStepperForProduct(product).check(matches(isDisplayed()))
    }

    @Test
    fun onCreateView_TrackingModeIsCheckboxQuantity_CheckboxAndQuantityShown() = runTest {
        val product =
            onCreateView_TrackingMode_ArrangeAct(TrackingMode.CHECKBOX_QUANTITY)

        getCheckboxForProduct(product).check(matches(isDisplayed()))
        getQtyStepperForProduct(product).check(matches(isDisplayed()))
    }

    private fun getCheckboxForProduct(product: Product): ViewInteraction =
        onView(
            allOf(
                withId(R.id.chk_in_stock),
                hasSibling(allOf(withText(product.name), withId(R.id.txt_product_name)))
            )
        )

    private fun getQtyStepperForProduct(product: Product): ViewInteraction =
        onView(
            allOf(
                withId(R.id.stp_qty_selector),
                hasSibling(allOf(withText(product.name), withId(R.id.txt_product_name)))
            )
        )

    private fun getQtyStepperComponentForProduct(
        product: Product, componentId: Int
    ): ViewInteraction =
        onView(
            allOf(
                withId(componentId),
                isDescendantOfA(
                    allOf(
                        withId(R.id.stp_qty_selector),
                        hasSibling(allOf(withId(R.id.txt_product_name), withText(product.name)))
                    )
                )
            )
        )

    private suspend fun onProductQuantityChange_Arrange(
        initialQty: Double, increment: Double = 1.0
    ): Product {
        val shoppingListPrefs = ShoppingListPreferencesTestImpl()
        shoppingListPrefs.setTrackingMode(TrackingMode.QUANTITY)
        val shoppingList = getShoppingList()

        val product = getProduct(shoppingList, false).copy(qtyIncrement = increment)
        get<ProductRepository>().update(product)

        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)

        getFragmentScenario(shoppingListBundle, shoppingListPrefs)


        val edtQty = getQtyStepperComponentForProduct(product, R.id.edt_qty)
        edtQty.perform(clearText())
        edtQty.perform(typeText(initialQty.toString()))

        return product
    }

    private fun formatQty(qty: Double): String =
        DecimalFormat("0.###").format(qty)

    @Test
    fun onProductQuantityChange_decQtyButtonClicked_qtyDecreased() = runTest {
        val initialQty = 5.0
        val product = onProductQuantityChange_Arrange(initialQty)

        val btnDecQty = getQtyStepperComponentForProduct(product, R.id.btn_qty_dec)
        btnDecQty.perform(click())

        val edtQty = getQtyStepperComponentForProduct(product, R.id.edt_qty)
        val expected = formatQty(initialQty.dec())
        edtQty.check(matches(withText(expected)))
    }

    @Test
    fun onProductQuantityChange_incQtyButtonClicked_qtyIncreased() = runTest {
        val initialQty = 5.0
        val product = onProductQuantityChange_Arrange(initialQty)

        val btnIncQty = getQtyStepperComponentForProduct(product, R.id.btn_qty_inc)
        btnIncQty.perform(click())

        val edtQty = getQtyStepperComponentForProduct(product, R.id.edt_qty)
        val expected = formatQty(initialQty.inc())
        edtQty.check(matches(withText(expected)))
    }

    @Test
    fun onProductQuantityChange_decQtyButtonClickedWithZeroQty_qtyRemainsZero() = runTest {
        val initialQty = 1.0
        val product = onProductQuantityChange_Arrange(initialQty)

        val btnDecQty = getQtyStepperComponentForProduct(product, R.id.btn_qty_dec)
        btnDecQty.perform(click())
        btnDecQty.perform(click())
        btnDecQty.perform(click())

        val edtQty = getQtyStepperComponentForProduct(product, R.id.edt_qty)
        edtQty.check(matches(withText("")))

        val updatedProduct = get<ProductRepository>().get(product.id)
        assertEquals(0.0, updatedProduct?.qtyNeeded)
    }

    @Test
    fun onProductQuantityChange_decQtyButtonClickedAndProductHasQtyInc_qtyDecreased() = runTest {
        val initialQty = 5.0
        val qtyIncrement = 0.75
        val product = onProductQuantityChange_Arrange(initialQty, qtyIncrement)

        val btnDecQty = getQtyStepperComponentForProduct(product, R.id.btn_qty_dec)
        btnDecQty.perform(click())

        val edtQty = getQtyStepperComponentForProduct(product, R.id.edt_qty)
        val expected = formatQty(initialQty - qtyIncrement)
        edtQty.check(matches(withText(expected)))
    }

    @Test
    fun onProductQuantityChange_incQtyButtonClickedAndProductHasQtyInc_qtyIncreased() = runTest {
        val initialQty = 5.0
        val qtyIncrement = 0.75
        val product = onProductQuantityChange_Arrange(initialQty, qtyIncrement)

        val btnIncQty = getQtyStepperComponentForProduct(product, R.id.btn_qty_inc)
        btnIncQty.perform(click())

        val edtQty = getQtyStepperComponentForProduct(product, R.id.edt_qty)
        val expected = formatQty(initialQty + qtyIncrement)
        edtQty.check(matches(withText(expected)))
    }

    @Test
    fun onProductQuantityChange_incQtyButtonClickedWithMaxLength_qtyRemainsMaxLength() = runTest {
        val initialQty = 99999.0
        val product = onProductQuantityChange_Arrange(initialQty)

        val btnIncQty = getQtyStepperComponentForProduct(product, R.id.btn_qty_inc)
        btnIncQty.perform(click())

        val edtQty = getQtyStepperComponentForProduct(product, R.id.edt_qty)
        val expected = formatQty(initialQty)
        edtQty.check(matches(withText(expected)))
    }

    @Test
    fun onProductQuantityChange_qtyEditChanged_ProductQuantityUpdated() = runTest {
        val initialQty = 5.0
        val product = onProductQuantityChange_Arrange(initialQty)

        val edtQty = getQtyStepperComponentForProduct(product, R.id.edt_qty)
        edtQty.perform(clearText())
        edtQty.perform(typeText(initialQty.inc().toString()))

        val updatedProduct = get<ProductRepository>().get(product.id)
        assertEquals(initialQty.inc(), updatedProduct?.qtyNeeded)
    }

    @Test
    fun onProductQuantityChange_qtyEditCleared_ProductQuantitySetToZero() = runTest {
        val initialQty = 5.0
        val product = onProductQuantityChange_Arrange(initialQty)

        val edtQty = getQtyStepperComponentForProduct(product, R.id.edt_qty)
        edtQty.perform(clearText())

        val updatedProduct = get<ProductRepository>().get(product.id)
        assertEquals(0.0, updatedProduct?.qtyNeeded)
    }

    @Test
    fun onProductQuantityChange_qtyEditIsDecimal_ProductQuantityUpdated() = runTest {
        val initialQty = 5.0
        val product = onProductQuantityChange_Arrange(initialQty)
        val newQty = 6.78

        val edtQty = getQtyStepperComponentForProduct(product, R.id.edt_qty)
        edtQty.perform(clearText())
        edtQty.perform(typeText(newQty.toString()))

        val updatedProduct = get<ProductRepository>().get(product.id)
        assertEquals(newQty, updatedProduct?.qtyNeeded)
    }

    @Test
    fun onActionItemClicked_ActionItemIsShowNote_NoteDialogShown() = runTest {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false)
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val scenario = getActivityScenario(bundle)
        var showNoteDialogTitle = ""

        scenario.onActivity {
            showNoteDialogTitle =
                activityFragment.getString(R.string.note_dialog_title, product.name)
        }

        val productItem = onView(allOf(withText(product.name), withId(R.id.txt_product_name)))
        productItem.perform(longClick())

        onView(withId(R.id.mnu_product_note)).perform(click())

        onView(withText(showNoteDialogTitle))
            .inRoot(isDialog())
            .check(matches(isDisplayed()))
    }

    private suspend fun testShowNoteClosed(buttonResId: Int) {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false)
        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        val scenario = getActivityScenario(bundle)
        var showNoteDialogTitle = ""

        scenario.onActivity {
            showNoteDialogTitle =
                activityFragment.getString(R.string.note_dialog_title, product.name)
        }

        onView(withText(product.name)).perform(longClick())
        onView(withId(R.id.mnu_product_note)).perform(click())

        onView(withText(buttonResId))
            .inRoot(isDialog())
            .perform(click())

        onView(withText(showNoteDialogTitle))
            .check(doesNotExist())

        scenario.onActivity {
            assertFalse(activityFragment.hasSelectedItems())
        }
    }

    @Test
    fun onActionItemClicked_ShowNoteCancelled_DialogClosed() = runTest {
        testShowNoteClosed(android.R.string.cancel)
    }

    @Test
    fun onActionItemClicked_ShowNoteCompleted_DialogClosed() = runTest {
        testShowNoteClosed(android.R.string.ok)
    }

    @Test
    fun onActionItemClicked_ActionItemIsSelectAisle_AislePickerDialogShown() = runTest {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false)

        val bundle = bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        getActivityScenario(bundle)

        onView(withText(product.name)).perform(longClick())
        onView(withId(R.id.mnu_aisle_picker)).perform(click())

        onView(withText(product.name))
            .inRoot(isDialog())
            .check(matches(isDisplayed()))
    }

    @Test
    fun onAislePickerResult_AisleSelected_ProductAisleUpdated() = runTest {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false)
        val aisleProduct = shoppingList.aisles
            .first { it.products.any { p -> p.product.id == product.id } }
            .products.first { it.product.id == product.id }

        val newAisle = shoppingList.aisles.first { aisle ->
            aisle.products.none { it.product.id == product.id }
        }

        val scenario = getActivityScenario(
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        )

        onView(withText(product.name)).perform(longClick())

        // Simulate the fragment result with the new aisle ID
        val result = Bundle().apply {
            putInt(AislePickerDialogFragment.KEY_SELECTED_AISLE_ID, newAisle.id)
        }

        scenario.onActivity { a ->
            val fragment = a.supportFragmentManager.fragments
                .first { it is ShoppingListFragment } as ShoppingListFragment

            fragment.childFragmentManager.setFragmentResult(
                ShoppingListFragment.AISLE_PICKER_REQUEST_KEY, result
            )
        }

        val updatedAisleProduct = get<AisleProductRepository>().get(aisleProduct.id)
        assertEquals(newAisle.id, updatedAisleProduct?.aisleId)

        scenario.onActivity {
            assertFalse(activityFragment.hasSelectedItems())
        }
    }

    @Test
    fun onAislePickerResult_AddNewAisle_AddSingleAisleDialogDisplayed() = runTest {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false)
        val scenario = getActivityScenario(
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        )

        onView(withText(product.name)).perform(longClick())

        // Simulate the fragment result with the new aisle ID
        val result = Bundle().apply {
            putBoolean(AislePickerDialogFragment.KEY_ADD_NEW_AISLE, true)
        }

        scenario.onActivity { a ->
            val fragment = a.supportFragmentManager.fragments
                .first { it is ShoppingListFragment } as ShoppingListFragment

            fragment.childFragmentManager.setFragmentResult(
                ShoppingListFragment.AISLE_PICKER_REQUEST_KEY, result
            )
        }

        onView(withText(R.string.add_aisle))
            .inRoot(isDialog())
            .check(matches(isDisplayed()))

        onView(withText(R.string.add_another))
            .inRoot(isDialog())
            .check(doesNotExist())

        onView(allOf(instanceOf(EditText::class.java)))
            .inRoot(isDialog())
            .check(matches(withText("")))
    }

    @Test
    fun onAisleDialogResult_AisleAdded_SelectedProductAisleUpdated() = runTest {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false)
        val aisleProduct = shoppingList.aisles
            .first { it.products.any { p -> p.product.id == product.id } }
            .products.first { it.product.id == product.id }

        val newAisleId = get<AisleRepository>().add(
            Aisle(
                name = "New Aisle",
                products = emptyList(),
                locationId = shoppingList.id,
                rank = 1000,
                id = 0,
                isDefault = false,
                expanded = true
            )
        )

        val scenario = getActivityScenario(
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        )

        onView(withText(product.name)).perform(longClick())

        // Simulate the fragment result with the new aisle ID
        val result = Bundle().apply {
            putInt(AisleDialogFragment.KEY_AISLE_ID, newAisleId)
        }

        scenario.onActivity { a ->
            val fragment = a.supportFragmentManager.fragments
                .first { it is ShoppingListFragment } as ShoppingListFragment

            fragment.childFragmentManager.setFragmentResult(
                ShoppingListFragment.ADD_AISLE_REQUEST_KEY, result
            )
        }

        val updatedAisleProduct = get<AisleProductRepository>().get(aisleProduct.id)
        assertEquals(newAisleId, updatedAisleProduct?.aisleId)

        scenario.onActivity {
            assertFalse(activityFragment.hasSelectedItems())
        }
    }


    private suspend fun onCreateView_TrackingModeProduct_ArrangeAct(trackingMode: TrackingMode): Product {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false).copy(trackingMode = trackingMode)
        get<ProductRepository>().update(product)

        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)

        getFragmentScenario(shoppingListBundle)

        return product
    }

    @Test
    fun onCreateView_TrackingModeIsCheckboxProduct_OnlyCheckBoxShown() = runTest {
        val product =
            onCreateView_TrackingModeProduct_ArrangeAct(TrackingMode.CHECKBOX)

        getCheckboxForProduct(product).check(matches(isDisplayed()))
        getQtyStepperForProduct(product).check(matches(not(isDisplayed())))
    }

    @Test
    fun onCreateView_TrackingModeIsNoneProduct_NothingShown() = runTest {
        val product =
            onCreateView_TrackingModeProduct_ArrangeAct(TrackingMode.NONE)

        getCheckboxForProduct(product).check(matches(not(isDisplayed())))
        getQtyStepperForProduct(product).check(matches(not(isDisplayed())))
    }

    @Test
    fun onCreateView_TrackingModeIsQuantityProduct_QuantityShown() = runTest {
        val product =
            onCreateView_TrackingModeProduct_ArrangeAct(TrackingMode.QUANTITY)

        getCheckboxForProduct(product).check(matches(not(isDisplayed())))
        getQtyStepperForProduct(product).check(matches(isDisplayed()))
    }

    @Test
    fun onCreateView_TrackingModeIsCheckboxQuantityProduct_CheckboxAndQuantityShown() = runTest {
        val product =
            onCreateView_TrackingModeProduct_ArrangeAct(TrackingMode.CHECKBOX_QUANTITY)

        getCheckboxForProduct(product).check(matches(isDisplayed()))
        getQtyStepperForProduct(product).check(matches(isDisplayed()))
    }

    private suspend fun qtySetToZero_ArrangeActAssert(unitOfMeasure: String, expectedHint: String) {
        val shoppingList = getShoppingList()
        val product = getProduct(shoppingList, false).copy(
            unitOfMeasure = unitOfMeasure,
            trackingMode = TrackingMode.QUANTITY
        )

        get<ProductRepository>().update(product)

        val shoppingListBundle =
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)

        getFragmentScenario(shoppingListBundle)

        val edtQty = getQtyStepperComponentForProduct(product, R.id.edt_qty)
        edtQty.perform(clearText())

        edtQty.check(matches(withHint(expectedHint)))
    }

    @Test
    fun onProductQuantityChange_qtySetToZeroAndNoUom_showQtyHint() = runTest {
        val expectedHint = getInstrumentation().targetContext.getString(R.string.qty)
        qtySetToZero_ArrangeActAssert("", expectedHint)
    }

    @Test
    fun onProductQuantityChange_qtySetToZeroAndUomSet_showUomHint() = runTest {
        val unitOfMeasure = "kg"
        qtySetToZero_ArrangeActAssert(unitOfMeasure, unitOfMeasure)
    }

    @Test
    fun onAisleExpandToggle_noSelection_ToggleAisleExpanded() = runTest {
        val aisleRepository = get<AisleRepository>()
        val shoppingList = getShoppingList()
        getActivityScenario(
            bundler.makeShoppingListBundle(shoppingList.id, shoppingList.defaultFilter)
        )

        val aisle = getAisle(shoppingList, isDefault = false, productsInStock = false)
        val aisleBefore = aisleRepository.get(aisle.id)!!

        onView(withText(aisle.name)).perform(click())

        assertEquals(!aisleBefore.expanded, aisleRepository.get(aisle.id)!!.expanded)
    }

    /**
     * Aisle Expanded Toggle:
     * - On no selection, aisle expands
     * - on selected items, select aisle
     */
}