/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
/** Used to generate unique IDs for events. */
let uniqueIds = 0;
/**
 * Creates a browser MouseEvent with the specified options.
 * @docs-private
 */
export function createMouseEvent(type, clientX = 0, clientY = 0, button = 0, modifiers = {}) {
    // Note: We cannot determine the position of the mouse event based on the screen
    // because the dimensions and position of the browser window are not available
    // To provide reasonable `screenX` and `screenY` coordinates, we simply use the
    // client coordinates as if the browser is opened in fullscreen.
    const screenX = clientX;
    const screenY = clientY;
    const event = new MouseEvent(type, {
        bubbles: true,
        cancelable: true,
        composed: true,
        view: window,
        detail: 0,
        relatedTarget: null,
        screenX,
        screenY,
        clientX,
        clientY,
        ctrlKey: modifiers.control,
        altKey: modifiers.alt,
        shiftKey: modifiers.shift,
        metaKey: modifiers.meta,
        button: button,
        buttons: 1,
    });
    // The `MouseEvent` constructor doesn't allow us to pass these properties into the constructor.
    // Override them to `1`, because they're used for fake screen reader event detection.
    defineReadonlyEventProperty(event, 'offsetX', 1);
    defineReadonlyEventProperty(event, 'offsetY', 1);
    return event;
}
/**
 * Creates a browser `PointerEvent` with the specified options. Pointer events
 * by default will appear as if they are the primary pointer of their type.
 * https://www.w3.org/TR/pointerevents2/#dom-pointerevent-isprimary.
 *
 * For example, if pointer events for a multi-touch interaction are created, the non-primary
 * pointer touches would need to be represented by non-primary pointer events.
 *
 * @docs-private
 */
export function createPointerEvent(type, clientX = 0, clientY = 0, options = { isPrimary: true }) {
    return new PointerEvent(type, {
        bubbles: true,
        cancelable: true,
        composed: true,
        view: window,
        clientX,
        clientY,
        ...options,
    });
}
/**
 * Creates a browser TouchEvent with the specified pointer coordinates.
 * @docs-private
 */
export function createTouchEvent(type, pageX = 0, pageY = 0, clientX = 0, clientY = 0) {
    // We cannot use the `TouchEvent` or `Touch` because Firefox and Safari lack support.
    // TODO: Switch to the constructor API when it is available for Firefox and Safari.
    const event = document.createEvent('UIEvent');
    const touchDetails = { pageX, pageY, clientX, clientY, identifier: uniqueIds++ };
    // TS3.6 removes the initUIEvent method and suggests porting to "new UIEvent()".
    event.initUIEvent(type, true, true, window, 0);
    // Most of the browsers don't have a "initTouchEvent" method that can be used to define
    // the touch details.
    defineReadonlyEventProperty(event, 'touches', [touchDetails]);
    defineReadonlyEventProperty(event, 'targetTouches', [touchDetails]);
    defineReadonlyEventProperty(event, 'changedTouches', [touchDetails]);
    return event;
}
/**
 * Creates a keyboard event with the specified key and modifiers.
 * @docs-private
 */
export function createKeyboardEvent(type, keyCode = 0, key = '', modifiers = {}) {
    return new KeyboardEvent(type, {
        bubbles: true,
        cancelable: true,
        composed: true,
        view: window,
        keyCode: keyCode,
        key: key,
        shiftKey: modifiers.shift,
        metaKey: modifiers.meta,
        altKey: modifiers.alt,
        ctrlKey: modifiers.control,
    });
}
/**
 * Creates a fake event object with any desired event type.
 * @docs-private
 */
export function createFakeEvent(type, bubbles = false, cancelable = true, composed = true) {
    return new Event(type, { bubbles, cancelable, composed });
}
/**
 * Defines a readonly property on the given event object. Readonly properties on an event object
 * are always set as configurable as that matches default readonly properties for DOM event objects.
 */
function defineReadonlyEventProperty(event, propertyName, value) {
    Object.defineProperty(event, propertyName, { get: () => value, configurable: true });
}
//# sourceMappingURL=data:application/json;base64,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