"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renamePropertyWithStableKeys = exports.inlineProjectConfigurations = exports.resolveNewFormatWithInlineProjects = exports.resolveOldFormatWithInlineProjects = exports.toOldFormatOrNull = exports.toNewFormatOrNull = exports.toNewFormat = exports.reformattedWorkspaceJsonOrNull = exports.Workspaces = exports.workspaceConfigName = void 0;
const fs_1 = require("fs");
const path = require("path");
const app_root_1 = require("../utils/app-root");
const fileutils_1 = require("../utils/fileutils");
const logger_1 = require("./logger");
function workspaceConfigName(root) {
    if (fs_1.existsSync(path.join(root, 'angular.json'))) {
        return 'angular.json';
    }
    else {
        return 'workspace.json';
    }
}
exports.workspaceConfigName = workspaceConfigName;
class Workspaces {
    constructor(root) {
        this.root = root;
    }
    relativeCwd(cwd) {
        return path.relative(this.root, cwd) || null;
    }
    calculateDefaultProjectName(cwd, wc) {
        const relativeCwd = this.relativeCwd(cwd);
        if (relativeCwd) {
            const matchingProject = Object.keys(wc.projects).find((p) => {
                const projectRoot = wc.projects[p].root;
                return (relativeCwd == projectRoot ||
                    relativeCwd.startsWith(`${projectRoot}/`));
            });
            if (matchingProject)
                return matchingProject;
        }
        return wc.defaultProject;
    }
    readWorkspaceConfiguration() {
        const rawWorkspace = fileutils_1.readJsonFile(path.join(this.root, workspaceConfigName(this.root)));
        const parsedWorkspace = resolveNewFormatWithInlineProjects(rawWorkspace, this.root);
        const nxJsonPath = path.join(this.root, 'nx.json');
        const nxJson = fs_1.existsSync(nxJsonPath)
            ? fileutils_1.readJsonFile(nxJsonPath)
            : {};
        assertValidWorkspaceConfiguration(nxJson);
        return Object.assign(Object.assign({}, parsedWorkspace), nxJson);
    }
    isNxExecutor(nodeModule, executor) {
        const schema = this.readExecutor(nodeModule, executor).schema;
        return schema['cli'] === 'nx';
    }
    isNxGenerator(collectionName, generatorName) {
        const schema = this.readGenerator(collectionName, generatorName).schema;
        return schema['cli'] === 'nx';
    }
    readExecutor(nodeModule, executor) {
        try {
            const { executorsFilePath, executorConfig } = this.readExecutorsJson(nodeModule, executor);
            const executorsDir = path.dirname(executorsFilePath);
            const schemaPath = path.join(executorsDir, executorConfig.schema || '');
            const schema = fileutils_1.readJsonFile(schemaPath);
            if (!schema.properties || typeof schema.properties !== 'object') {
                schema.properties = {};
            }
            const implementationFactory = this.getImplementationFactory(executorConfig.implementation, executorsDir);
            const batchImplementationFactory = executorConfig.batchImplementation
                ? this.getImplementationFactory(executorConfig.batchImplementation, executorsDir)
                : null;
            const hasherFactory = executorConfig.hasher
                ? this.getImplementationFactory(executorConfig.hasher, executorsDir)
                : null;
            return {
                schema,
                implementationFactory,
                batchImplementationFactory,
                hasherFactory,
            };
        }
        catch (e) {
            throw new Error(`Unable to resolve ${nodeModule}:${executor}.\n${e.message}`);
        }
    }
    readGenerator(collectionName, generatorName) {
        var _a, _b;
        try {
            const { generatorsFilePath, generatorsJson, normalizedGeneratorName } = this.readGeneratorsJson(collectionName, generatorName);
            const generatorsDir = path.dirname(generatorsFilePath);
            const generatorConfig = ((_a = generatorsJson.generators) === null || _a === void 0 ? void 0 : _a[normalizedGeneratorName]) ||
                ((_b = generatorsJson.schematics) === null || _b === void 0 ? void 0 : _b[normalizedGeneratorName]);
            const schemaPath = path.join(generatorsDir, generatorConfig.schema || '');
            const schema = fileutils_1.readJsonFile(schemaPath);
            if (!schema.properties || typeof schema.properties !== 'object') {
                schema.properties = {};
            }
            generatorConfig.implementation =
                generatorConfig.implementation || generatorConfig.factory;
            const implementationFactory = this.getImplementationFactory(generatorConfig.implementation, generatorsDir);
            return { normalizedGeneratorName, schema, implementationFactory };
        }
        catch (e) {
            throw new Error(`Unable to resolve ${collectionName}:${generatorName}.\n${e.message}`);
        }
    }
    getImplementationFactory(implementation, directory) {
        const [implementationModulePath, implementationExportName] = implementation.split('#');
        return () => {
            const module = require(path.join(directory, implementationModulePath));
            return module[implementationExportName || 'default'];
        };
    }
    readExecutorsJson(nodeModule, executor) {
        var _a, _b, _c;
        const packageJsonPath = require.resolve(`${nodeModule}/package.json`, {
            paths: this.resolvePaths(),
        });
        const packageJson = fileutils_1.readJsonFile(packageJsonPath);
        const executorsFile = (_a = packageJson.executors) !== null && _a !== void 0 ? _a : packageJson.builders;
        if (!executorsFile) {
            throw new Error(`The "${nodeModule}" package does not support Nx executors.`);
        }
        const executorsFilePath = require.resolve(path.join(path.dirname(packageJsonPath), executorsFile));
        const executorsJson = fileutils_1.readJsonFile(executorsFilePath);
        const executorConfig = ((_b = executorsJson.executors) === null || _b === void 0 ? void 0 : _b[executor]) || ((_c = executorsJson.builders) === null || _c === void 0 ? void 0 : _c[executor]);
        if (!executorConfig) {
            throw new Error(`Cannot find executor '${executor}' in ${executorsFilePath}.`);
        }
        return { executorsFilePath, executorConfig };
    }
    readGeneratorsJson(collectionName, generator) {
        var _a;
        let generatorsFilePath;
        if (collectionName.endsWith('.json')) {
            generatorsFilePath = require.resolve(collectionName, {
                paths: this.resolvePaths(),
            });
        }
        else {
            const packageJsonPath = require.resolve(`${collectionName}/package.json`, {
                paths: this.resolvePaths(),
            });
            const packageJson = fileutils_1.readJsonFile(packageJsonPath);
            const generatorsFile = (_a = packageJson.generators) !== null && _a !== void 0 ? _a : packageJson.schematics;
            if (!generatorsFile) {
                throw new Error(`The "${collectionName}" package does not support Nx generators.`);
            }
            generatorsFilePath = require.resolve(path.join(path.dirname(packageJsonPath), generatorsFile));
        }
        const generatorsJson = fileutils_1.readJsonFile(generatorsFilePath);
        let normalizedGeneratorName = findFullGeneratorName(generator, generatorsJson.generators) ||
            findFullGeneratorName(generator, generatorsJson.schematics);
        if (!normalizedGeneratorName) {
            for (let parent of generatorsJson.extends || []) {
                try {
                    return this.readGeneratorsJson(parent, generator);
                }
                catch (e) { }
            }
            throw new Error(`Cannot find generator '${generator}' in ${generatorsFilePath}.`);
        }
        return { generatorsFilePath, generatorsJson, normalizedGeneratorName };
    }
    resolvePaths() {
        return this.root ? [this.root, __dirname] : [__dirname];
    }
}
exports.Workspaces = Workspaces;
function assertValidWorkspaceConfiguration(nxJson) {
    // Assert valid workspace configuration
    if (nxJson.projects) {
        logger_1.logger.error('NX As of Nx 13, project configuration should be moved from nx.json to workspace.json/project.json. Please run "nx format" to fix this.');
        process.exit(1);
    }
}
function findFullGeneratorName(name, generators) {
    if (generators) {
        for (let [key, data] of Object.entries(generators)) {
            if (key === name ||
                (data.aliases && data.aliases.includes(name))) {
                return key;
            }
        }
    }
}
function reformattedWorkspaceJsonOrNull(w) {
    return w.version === 2 ? toNewFormatOrNull(w) : toOldFormatOrNull(w);
}
exports.reformattedWorkspaceJsonOrNull = reformattedWorkspaceJsonOrNull;
function toNewFormat(w) {
    const f = toNewFormatOrNull(w);
    return f !== null && f !== void 0 ? f : w;
}
exports.toNewFormat = toNewFormat;
function toNewFormatOrNull(w) {
    let formatted = false;
    Object.values(w.projects || {}).forEach((projectConfig) => {
        if (projectConfig.architect) {
            renamePropertyWithStableKeys(projectConfig, 'architect', 'targets');
            formatted = true;
        }
        if (projectConfig.schematics) {
            renamePropertyWithStableKeys(projectConfig, 'schematics', 'generators');
            formatted = true;
        }
        Object.values(projectConfig.targets || {}).forEach((target) => {
            if (target.builder !== undefined) {
                renamePropertyWithStableKeys(target, 'builder', 'executor');
                formatted = true;
            }
        });
    });
    if (w.schematics) {
        renamePropertyWithStableKeys(w, 'schematics', 'generators');
        formatted = true;
    }
    if (w.version !== 2) {
        w.version = 2;
        formatted = true;
    }
    return formatted ? w : null;
}
exports.toNewFormatOrNull = toNewFormatOrNull;
function toOldFormatOrNull(w) {
    let formatted = false;
    Object.values(w.projects || {}).forEach((projectConfig) => {
        if (typeof projectConfig === 'string') {
            throw new Error("'project.json' files are incompatible with version 1 workspace schemas.");
        }
        if (projectConfig.targets) {
            renamePropertyWithStableKeys(projectConfig, 'targets', 'architect');
            formatted = true;
        }
        if (projectConfig.generators) {
            renamePropertyWithStableKeys(projectConfig, 'generators', 'schematics');
            formatted = true;
        }
        Object.values(projectConfig.architect || {}).forEach((target) => {
            if (target.executor !== undefined) {
                renamePropertyWithStableKeys(target, 'executor', 'builder');
                formatted = true;
            }
        });
    });
    if (w.generators) {
        renamePropertyWithStableKeys(w, 'generators', 'schematics');
        formatted = true;
    }
    if (w.version !== 1) {
        w.version = 1;
        formatted = true;
    }
    return formatted ? w : null;
}
exports.toOldFormatOrNull = toOldFormatOrNull;
function resolveOldFormatWithInlineProjects(w, root = app_root_1.appRootPath) {
    const inlined = inlineProjectConfigurations(w, root);
    const formatted = toOldFormatOrNull(inlined);
    return formatted ? formatted : inlined;
}
exports.resolveOldFormatWithInlineProjects = resolveOldFormatWithInlineProjects;
function resolveNewFormatWithInlineProjects(w, root = app_root_1.appRootPath) {
    return toNewFormat(inlineProjectConfigurations(w, root));
}
exports.resolveNewFormatWithInlineProjects = resolveNewFormatWithInlineProjects;
function inlineProjectConfigurations(w, root = app_root_1.appRootPath) {
    Object.entries(w.projects || {}).forEach(([project, config]) => {
        if (typeof config === 'string') {
            const configFilePath = path.join(root, config, 'project.json');
            const fileConfig = fileutils_1.readJsonFile(configFilePath);
            w.projects[project] = fileConfig;
        }
    });
    return w;
}
exports.inlineProjectConfigurations = inlineProjectConfigurations;
// we have to do it this way to preserve the order of properties
// not to screw up the formatting
function renamePropertyWithStableKeys(obj, from, to) {
    const copy = Object.assign({}, obj);
    Object.keys(obj).forEach((k) => {
        delete obj[k];
    });
    Object.keys(copy).forEach((k) => {
        if (k === from) {
            obj[to] = copy[k];
        }
        else {
            obj[k] = copy[k];
        }
    });
}
exports.renamePropertyWithStableKeys = renamePropertyWithStableKeys;
//# sourceMappingURL=workspace.js.map