"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.migrateCommand = void 0;
const tslib_1 = require("tslib");
const utils_fs_1 = require("@ionic/utils-fs");
const path_1 = require("path");
const rimraf_1 = tslib_1.__importDefault(require("rimraf"));
const colors_1 = tslib_1.__importDefault(require("../colors"));
const common_1 = require("../common");
const errors_1 = require("../errors");
const log_1 = require("../log");
const fs_1 = require("../util/fs");
const subprocess_1 = require("../util/subprocess");
const template_1 = require("../util/template");
const xml_1 = require("../util/xml");
// eslint-disable-next-line prefer-const
let allDependencies = {};
const libs = [
    '@capacitor/core',
    '@capacitor/cli',
    '@capacitor/ios',
    '@capacitor/android',
];
const plugins = [
    '@capacitor/action-sheet',
    '@capacitor/app',
    '@capacitor/app-launcher',
    '@capacitor/browser',
    '@capacitor/camera',
    '@capacitor/clipboard',
    '@capacitor/device',
    '@capacitor/dialog',
    '@capacitor/filesystem',
    '@capacitor/geolocation',
    '@capacitor/google-maps',
    '@capacitor/haptics',
    '@capacitor/keyboard',
    '@capacitor/local-notifications',
    '@capacitor/motion',
    '@capacitor/network',
    '@capacitor/preferences',
    '@capacitor/push-notifications',
    '@capacitor/screen-reader',
    '@capacitor/share',
    '@capacitor/splash-screen',
    '@capacitor/status-bar',
    '@capacitor/text-zoom',
    '@capacitor/toast',
];
const coreVersion = '^4.0.0';
const pluginVersion = '^4.0.0';
const gradleVersion = '7.4.2';
async function migrateCommand(config) {
    if (config === null) {
        errors_1.fatal('Config data missing');
    }
    const variablesAndClasspaths = await getAndroidVarriablesAndClasspaths(config);
    if (!variablesAndClasspaths) {
        errors_1.fatal('Variable and Classpath info could not be read.');
    }
    //*
    allDependencies = {
        ...config.app.package.dependencies,
        ...config.app.package.devDependencies,
    };
    const monorepoWarning = 'Please note this tool is not intended for use in a mono-repo enviroment, please check out the Ionic vscode extension for this functionality.';
    log_1.logger.info(monorepoWarning);
    const { migrateconfirm } = await log_1.logPrompt(`Capacitor 4 sets a deployment target of iOS 13 and Android 12 (SDK 32). \n`, {
        type: 'text',
        name: 'migrateconfirm',
        message: `Are you sure you want to migrate? (Y/n)`,
        initial: 'y',
    });
    if (typeof migrateconfirm === 'string' &&
        migrateconfirm.toLowerCase() === 'y') {
        try {
            const { npmInstallConfirm } = await log_1.logPrompt(`Would you like the migrator to run npm install to install the latest versions of capacitor packages? (Those using other package managers should answer N)`, {
                type: 'text',
                name: 'npmInstallConfirm',
                message: `Run Npm Install? (Y/n)`,
                initial: 'y',
            });
            const runNpmInstall = typeof npmInstallConfirm === 'string' &&
                npmInstallConfirm.toLowerCase() === 'y';
            try {
                await common_1.runTask(`Installing Latest NPM Modules.`, () => {
                    return installLatestNPMLibs(runNpmInstall, config);
                });
            }
            catch (ex) {
                log_1.logger.error(`npm install failed. Try deleting node_modules folder and running ${colors_1.default.input('npm install --force')} manually.`);
            }
            try {
                await common_1.runTask(`Migrating @capacitor/storage to @capacitor/preferences.`, () => {
                    return migrateStoragePluginToPreferences(runNpmInstall);
                });
            }
            catch (ex) {
                log_1.logger.error(`@capacitor/preferences failed to install. Try deleting node_modules folder and running ${colors_1.default.input('npm install @capacitor/preferences --force')} manually.`);
            }
            if (allDependencies['@capacitor/ios'] &&
                utils_fs_1.existsSync(config.ios.platformDirAbs)) {
                // Set deployment target to 13.0
                await common_1.runTask(`Migrating deployment target to 13.0.`, () => {
                    return updateFile(config, path_1.join(config.ios.nativeXcodeProjDirAbs, 'project.pbxproj'), 'IPHONEOS_DEPLOYMENT_TARGET = ', ';', '13.0');
                });
                // Update Podfile to 13.0
                await common_1.runTask(`Migrating Podfile to 13.0.`, () => {
                    return updateFile(config, path_1.join(config.ios.nativeProjectDirAbs, 'Podfile'), `platform :ios, '`, `'`, '13.0');
                });
                await common_1.runTask(`Migrating Podfile to use post_install script.`, () => {
                    return podfileAssertDeploymentTarget(path_1.join(config.ios.nativeProjectDirAbs, 'Podfile'));
                });
                // Remove touchesBegan
                await common_1.runTask(`Migrating AppDelegate.swift by removing touchesBegan.`, () => {
                    return updateFile(config, path_1.join(config.ios.nativeTargetDirAbs, 'AppDelegate.swift'), `override func touchesBegan`, `}`, undefined, true);
                });
                // Remove NSAppTransportSecurity
                await common_1.runTask(`Migrating Info.plist by removing NSAppTransportSecurity key.`, () => {
                    return removeKey(path_1.join(config.ios.nativeTargetDirAbs, 'Info.plist'), 'NSAppTransportSecurity');
                });
                // Remove USE_PUSH
                await common_1.runTask(`Migrating by removing USE_PUSH.`, () => {
                    return replacePush(path_1.join(config.ios.nativeXcodeProjDirAbs, 'project.pbxproj'));
                });
                // Remove from App Delegate
                await common_1.runTask(`Migrating App Delegate.`, () => {
                    return replaceIfUsePush(config);
                });
            }
            if (allDependencies['@capacitor/android'] &&
                utils_fs_1.existsSync(config.android.platformDirAbs)) {
                // AndroidManifest.xml add attribute: <activity android:exported="true"
                await common_1.runTask(`Migrating AndroidManifest.xml by adding android:exported attribute to Activity.`, () => {
                    return updateAndroidManifest(path_1.join(config.android.srcMainDirAbs, 'AndroidManifest.xml'));
                });
                // Update build.gradle
                const { leaveJCenterPrompt } = await log_1.logPrompt(`Some projects still require JCenter to function. If your project does, please answer yes below.`, {
                    type: 'text',
                    name: 'leaveJCenterPrompt',
                    message: `Keep JCenter if present? (y/N)`,
                    initial: 'n',
                });
                await common_1.runTask(`Migrating build.gradle file.`, () => {
                    return updateBuildGradle(path_1.join(config.android.platformDirAbs, 'build.gradle'), typeof leaveJCenterPrompt === 'string' &&
                        leaveJCenterPrompt.toLowerCase() === 'y', variablesAndClasspaths);
                });
                // Update app.gradle
                await common_1.runTask(`Migrating app/build.gradle file.`, () => {
                    return updateAppBuildGradle(path_1.join(config.android.appDirAbs, 'build.gradle'));
                });
                // Update gradle-wrapper.properties
                await common_1.runTask(`Migrating gradle-wrapper.properties by updating gradle version from 7.0 to ${gradleVersion}.`, () => {
                    return updateGradleWrapper(path_1.join(config.android.platformDirAbs, 'gradle', 'wrapper', 'gradle-wrapper.properties'));
                });
                // Variables gradle
                await common_1.runTask(`Migrating variables.gradle file.`, () => {
                    return (async () => {
                        const variablesPath = path_1.join(config.android.platformDirAbs, 'variables.gradle');
                        let txt = readFile(variablesPath);
                        if (!txt) {
                            return;
                        }
                        txt = txt.replace(/= {2}'/g, `= '`);
                        utils_fs_1.writeFileSync(variablesPath, txt, { encoding: 'utf-8' });
                        for (const variable of Object.keys(variablesAndClasspaths.variables)) {
                            if (!(await updateFile(config, variablesPath, `${variable} = '`, `'`, variablesAndClasspaths.variables[variable].toString(), true))) {
                                const didWork = await updateFile(config, variablesPath, `${variable} = `, `\n`, variablesAndClasspaths.variables[variable].toString(), true);
                                if (!didWork) {
                                    let file = readFile(variablesPath);
                                    if (file) {
                                        file = file.replace('}', `    ${variable} = '${variablesAndClasspaths.variables[variable].toString()}'\n}`);
                                        utils_fs_1.writeFileSync(variablesPath, file);
                                    }
                                }
                            }
                        }
                        const pluginVariables = {
                            firebaseMessagingVersion: '23.0.5',
                            playServicesLocationVersion: '20.0.0',
                            androidxBrowserVersion: '1.4.0',
                            androidxMaterialVersion: '1.6.1',
                            androidxExifInterfaceVersion: '1.3.3',
                        };
                        for (const variable of Object.keys(pluginVariables)) {
                            await updateFile(config, variablesPath, `${variable} = '`, `'`, pluginVariables[variable], true);
                        }
                    })();
                });
                // remove init
                await common_1.runTask('Migrating MainActivity', () => {
                    return migrateMainActivity(config);
                });
                rimraf_1.default.sync(path_1.join(config.android.appDirAbs, 'build'));
                // add new splashscreen
                await common_1.runTask('Migrate to Android 12 Splashscreen and apply DayNight theme.', () => {
                    return addNewSplashScreen(config);
                });
            }
            // Run Cap Sync
            await common_1.runTask(`Running cap sync.`, () => {
                return subprocess_1.getCommandOutput('npx', ['cap', 'sync']);
            });
            try {
                await common_1.runTask(`Upgrading gradle wrapper files`, () => {
                    return updateGradleWrapperFiles(config.android.platformDirAbs);
                });
            }
            catch (e) {
                if (e.includes('EACCES')) {
                    log_1.logger.error(`gradlew file does not have executable permissions. This can happen if the Android platform was added on a Windows machine. Please run ${colors_1.default.input(`chmod +x ./${config.android.platformDir}/gradlew`)} and ${colors_1.default.input(`cd ${config.android.platformDir} && ./gradlew wrapper --distribution-type all --gradle-version ${gradleVersion} --warning-mode all`)} to update the files manually`);
                }
                else {
                    log_1.logger.error(`gradle wrapper files were not updated`);
                }
            }
            // Write all breaking changes
            await common_1.runTask(`Writing breaking changes.`, () => {
                return writeBreakingChanges();
            });
            log_1.logSuccess(`Migration to Capacitor ${coreVersion} is complete. Run and test your app!`);
        }
        catch (err) {
            errors_1.fatal(`Failed to migrate: ${err}`);
        }
    }
    else {
        errors_1.fatal(`User canceled migration.`);
    }
    //*/
}
exports.migrateCommand = migrateCommand;
async function installLatestNPMLibs(runInstall, config) {
    const pkgJsonPath = path_1.join(config.app.rootDir, 'package.json');
    const pkgJsonFile = readFile(pkgJsonPath);
    if (!pkgJsonFile) {
        return;
    }
    const pkgJson = JSON.parse(pkgJsonFile);
    for (const devDepKey of Object.keys(pkgJson['devDependencies'] || {})) {
        if (libs.includes(devDepKey)) {
            pkgJson['devDependencies'][devDepKey] = coreVersion;
        }
        else if (plugins.includes(devDepKey)) {
            pkgJson['devDependencies'][devDepKey] = pluginVersion;
        }
    }
    for (const depKey of Object.keys(pkgJson['dependencies'])) {
        if (libs.includes(depKey)) {
            pkgJson['dependencies'][depKey] = coreVersion;
        }
        else if (plugins.includes(depKey)) {
            pkgJson['dependencies'][depKey] = pluginVersion;
        }
    }
    utils_fs_1.writeFileSync(pkgJsonPath, JSON.stringify(pkgJson, null, 2), {
        encoding: 'utf-8',
    });
    if (runInstall) {
        rimraf_1.default.sync(path_1.join(config.app.rootDir, 'node_modules/@capacitor/!(cli)'));
        await subprocess_1.runCommand('npm', ['i']);
    }
    else {
        log_1.logger.info(`Please run an install command with your package manager of choice. (ex: yarn install)`);
    }
}
async function migrateStoragePluginToPreferences(runInstall) {
    if (allDependencies['@capacitor/storage']) {
        log_1.logger.info('NOTE: @capacitor/storage was renamed to @capacitor/preferences, please be sure to replace occurances in your code.');
        if (runInstall) {
            await subprocess_1.getCommandOutput('npm', ['uninstall', '@capacitor/storage']);
            await subprocess_1.runCommand('npm', ['i', `@capacitor/preferences@${pluginVersion}`]);
        }
        else {
            log_1.logger.info(`Please manually uninstall @capacitor/storage and replace it with @capacitor/preferences@${pluginVersion}`);
        }
    }
}
async function writeBreakingChanges() {
    const breaking = [
        '@capacitor/storage',
        '@capacitor/camera',
        '@capacitor/push-notifications',
        '@capacitor/local-notifications',
    ];
    const broken = [];
    for (const lib of breaking) {
        if (allDependencies[lib]) {
            broken.push(lib);
        }
    }
    if (broken.length > 0) {
        log_1.logger.info(`IMPORTANT: Review https://capacitorjs.com/docs/updating/4-0#plugins for breaking changes in these plugins that you use: ${broken.join(', ')}.`);
    }
    if (allDependencies['@capacitor/android']) {
        log_1.logger.info('Warning: The Android Gradle plugin was updated and it requires Java 11 to run. You may need to select this in Android Studio.');
    }
}
async function updateAndroidManifest(filename) {
    const txt = readFile(filename);
    if (!txt) {
        return;
    }
    const hasAndroidExportedAlreadySet = new RegExp(/<activity([^>]*(android:exported=")[^>]*)>/g).test(txt);
    let isAndroidExportedSetToFalse = false;
    if (hasAndroidExportedAlreadySet) {
        isAndroidExportedSetToFalse = new RegExp(/<activity([^>]*(android:exported="false")[^>]*)>/g).test(txt);
    }
    // AndroidManifest.xml add attribute: <activity android:exported="true"
    if (hasAndroidExportedAlreadySet && !isAndroidExportedSetToFalse) {
        return; // Probably already updated manually
    }
    let replaced = txt;
    if (!hasAndroidExportedAlreadySet) {
        replaced = setAllStringIn(txt, '<activity', ' ', `\n            android:exported="true"\n`);
    }
    else {
        log_1.logger.info(`Found 'android:exported="false"' in your AndroidManifest.xml, if this is not intentional please update it manually to "true".`);
    }
    if (txt == replaced) {
        log_1.logger.error(`Unable to update Android Manifest. Missing <activity> tag`);
        return;
    }
    utils_fs_1.writeFileSync(filename, replaced, 'utf-8');
}
async function updateBuildGradle(filename, leaveJCenter, variablesAndClasspaths) {
    // In build.gradle add dependencies:
    // classpath 'com.android.tools.build:gradle:7.2.1'
    // classpath 'com.google.gms:google-services:4.3.13'
    const txt = readFile(filename);
    if (!txt) {
        return;
    }
    const neededDeps = {
        'com.android.tools.build:gradle': variablesAndClasspaths['com.android.tools.build:gradle'],
        'com.google.gms:google-services': variablesAndClasspaths['com.google.gms:google-services'],
    };
    let replaced = txt;
    for (const dep of Object.keys(neededDeps)) {
        if (replaced.includes(`classpath '${dep}`)) {
            const semver = await Promise.resolve().then(() => tslib_1.__importStar(require('semver')));
            const firstIndex = replaced.indexOf(dep) + dep.length + 1;
            const existingVersion = '' + replaced.substring(firstIndex, replaced.indexOf("'", firstIndex));
            if (semver.gte(neededDeps[dep], existingVersion)) {
                replaced = setAllStringIn(replaced, `classpath '${dep}:`, `'`, neededDeps[dep]);
                log_1.logger.info(`Set ${dep} = ${neededDeps[dep]}.`);
            }
        }
    }
    // Replace jcenter()
    const lines = replaced.split('\n');
    let inRepositories = false;
    let hasMavenCentral = false;
    let final = '';
    for (const line of lines) {
        if (line.includes('repositories {')) {
            inRepositories = true;
            hasMavenCentral = false;
        }
        else if (line.trim() == '}') {
            // Make sure we have mavenCentral()
            if (inRepositories && !hasMavenCentral) {
                final += '        mavenCentral()\n';
                log_1.logger.info(`Added mavenCentral().`);
            }
            inRepositories = false;
        }
        if (inRepositories && line.trim() === 'mavenCentral()') {
            hasMavenCentral = true;
        }
        if (inRepositories && line.trim() === 'jcenter()' && !leaveJCenter) {
            // skip jCentral()
            log_1.logger.info(`Removed jcenter().`);
        }
        else {
            final += line + '\n';
        }
    }
    if (txt !== final) {
        utils_fs_1.writeFileSync(filename, final, 'utf-8');
        return;
    }
}
async function getAndroidVarriablesAndClasspaths(config) {
    const tempAndroidTemplateFolder = path_1.join(config.cli.assetsDirAbs, 'tempAndroidTemplate');
    await template_1.extractTemplate(config.cli.assets.android.platformTemplateArchiveAbs, tempAndroidTemplateFolder);
    const variablesGradleFile = readFile(path_1.join(tempAndroidTemplateFolder, 'variables.gradle'));
    const buildGradleFile = readFile(path_1.join(tempAndroidTemplateFolder, 'build.gradle'));
    if (!variablesGradleFile || !buildGradleFile) {
        return;
    }
    fs_1.deleteFolderRecursive(tempAndroidTemplateFolder);
    const firstIndxOfCATBGV = buildGradleFile.indexOf(`classpath 'com.android.tools.build:gradle:`) + 42;
    const firstIndxOfCGGGS = buildGradleFile.indexOf(`com.google.gms:google-services:`) + 31;
    const comAndroidToolsBuildGradleVersion = '' +
        buildGradleFile.substring(firstIndxOfCATBGV, buildGradleFile.indexOf("'", firstIndxOfCATBGV));
    const comGoogleGmsGoogleServices = '' +
        buildGradleFile.substring(firstIndxOfCGGGS, buildGradleFile.indexOf("'", firstIndxOfCGGGS));
    const variablesGradleAsJSON = JSON.parse(variablesGradleFile
        .replace('ext ', '')
        .replace(/=/g, ':')
        .replace(/\n/g, ',')
        .replace(/,([^:]+):/g, function (_k, p1) {
        return `,"${p1}":`;
    })
        .replace('{,', '{')
        .replace(',}', '}')
        .replace(/\s/g, '')
        .replace(/'/g, '"'));
    return {
        'variables': variablesGradleAsJSON,
        'com.android.tools.build:gradle': comAndroidToolsBuildGradleVersion,
        'com.google.gms:google-services': comGoogleGmsGoogleServices,
    };
}
function readFile(filename) {
    try {
        if (!utils_fs_1.existsSync(filename)) {
            log_1.logger.error(`Unable to find ${filename}. Try updating it manually`);
            return;
        }
        return utils_fs_1.readFileSync(filename, 'utf-8');
    }
    catch (err) {
        log_1.logger.error(`Unable to read ${filename}. Verify it is not already open. ${err}`);
    }
}
async function updateAppBuildGradle(filename) {
    const txt = readFile(filename);
    if (!txt) {
        return;
    }
    let replaced = txt;
    if (!txt.includes('androidx.coordinatorlayout:coordinatorlayout:')) {
        replaced = replaced.replace('dependencies {', 'dependencies {\n    implementation "androidx.coordinatorlayout:coordinatorlayout:$androidxCoordinatorLayoutVersion"');
    }
    if (!txt.includes('androidx.core:core-splashscreen:')) {
        replaced = replaced.replace('dependencies {', 'dependencies {\n    implementation "androidx.core:core-splashscreen:$coreSplashScreenVersion"');
    }
    // const lines = txt.split('\n');
    if (replaced !== txt) {
        utils_fs_1.writeFileSync(filename, replaced, 'utf-8');
    }
}
async function updateGradleWrapper(filename) {
    const txt = readFile(filename);
    if (!txt) {
        return;
    }
    const replaced = setAllStringIn(txt, 'distributionUrl=', '\n', 
    // eslint-disable-next-line no-useless-escape
    `https\\://services.gradle.org/distributions/gradle-${gradleVersion}-all.zip`);
    utils_fs_1.writeFileSync(filename, replaced, 'utf-8');
}
async function updateGradleWrapperFiles(platformDir) {
    await subprocess_1.runCommand(`./gradlew`, [
        'wrapper',
        '--distribution-type',
        'all',
        '--gradle-version',
        gradleVersion,
        '--warning-mode',
        'all',
    ], {
        cwd: platformDir,
    });
}
async function updateFile(config, filename, textStart, textEnd, replacement, skipIfNotFound) {
    if (config === null) {
        return false;
    }
    const path = filename;
    let txt = readFile(path);
    if (!txt) {
        return false;
    }
    if (txt.includes(textStart)) {
        if (replacement) {
            txt = setAllStringIn(txt, textStart, textEnd, replacement);
            utils_fs_1.writeFileSync(path, txt, { encoding: 'utf-8' });
        }
        else {
            // Replacing in code so we need to count the number of brackets to find the end of the function in swift
            const lines = txt.split('\n');
            let replaced = '';
            let keep = true;
            let brackets = 0;
            for (const line of lines) {
                if (line.includes(textStart)) {
                    keep = false;
                }
                if (!keep) {
                    brackets += (line.match(/{/g) || []).length;
                    brackets -= (line.match(/}/g) || []).length;
                    if (brackets == 0) {
                        keep = true;
                    }
                }
                else {
                    replaced += line + '\n';
                }
            }
            utils_fs_1.writeFileSync(path, replaced, { encoding: 'utf-8' });
        }
        return true;
    }
    else if (!skipIfNotFound) {
        log_1.logger.error(`Unable to find "${textStart}" in ${filename}. Try updating it manually`);
    }
    return false;
}
function setAllStringIn(data, start, end, replacement) {
    let position = 0;
    let result = data;
    let replaced = true;
    while (replaced) {
        const foundIdx = result.indexOf(start, position);
        if (foundIdx == -1) {
            replaced = false;
        }
        else {
            const idx = foundIdx + start.length;
            position = idx + replacement.length;
            result =
                result.substring(0, idx) +
                    replacement +
                    result.substring(result.indexOf(end, idx));
        }
    }
    return result;
}
async function replaceIfUsePush(config) {
    const startLine = '#if USE_PUSH';
    const endLine = '#endif';
    const filename = path_1.join(config.ios.nativeTargetDirAbs, 'AppDelegate.swift');
    const txt = readFile(filename);
    if (!txt) {
        return;
    }
    const lines = txt.split('\n');
    let startLineIndex = null;
    let endLineIndex = null;
    for (const [key, item] of lines.entries()) {
        if (item.includes(startLine)) {
            startLineIndex = key;
            break;
        }
    }
    if (startLineIndex !== null) {
        for (const [key, item] of lines.entries()) {
            if (item.includes(endLine) && key > startLineIndex) {
                endLineIndex = key;
                break;
            }
        }
        if (endLineIndex !== null) {
            lines[endLineIndex] = '';
            lines[startLineIndex] = '';
            utils_fs_1.writeFileSync(filename, lines.join('\n'), 'utf-8');
        }
    }
}
async function replacePush(filename) {
    const txt = readFile(filename);
    if (!txt) {
        return;
    }
    let replaced = txt;
    replaced = replaced.replace('DEBUG USE_PUSH', 'DEBUG');
    replaced = replaced.replace('USE_PUSH', '""');
    if (replaced != txt) {
        utils_fs_1.writeFileSync(filename, replaced, 'utf-8');
    }
}
async function removeKey(filename, key) {
    const txt = readFile(filename);
    if (!txt) {
        return;
    }
    let lines = txt.split('\n');
    let removed = false;
    let removing = false;
    lines = lines.filter(line => {
        if (removing && line.includes('</dict>')) {
            removing = false;
            return false;
        }
        if (line.includes(`<key>${key}</key`)) {
            removing = true;
            removed = true;
        }
        return !removing;
    });
    if (removed) {
        utils_fs_1.writeFileSync(filename, lines.join('\n'), 'utf-8');
    }
}
async function podfileAssertDeploymentTarget(filename) {
    const txt = readFile(filename);
    if (!txt) {
        return;
    }
    let replaced = txt;
    if (!replaced.includes(`require_relative '../../node_modules/@capacitor/ios/scripts/pods_helpers`)) {
        replaced =
            `require_relative '../../node_modules/@capacitor/ios/scripts/pods_helpers'\n\n` +
                txt;
    }
    if (replaced.includes('post_install do |installer|')) {
        if (!replaced.includes(`assertDeploymentTarget(installer)`)) {
            replaced = replaced.replace('post_install do |installer|', `post_install do |installer|\n  assertDeploymentTarget(installer)\n`);
        }
    }
    else {
        replaced =
            replaced +
                `\n\npost_install do |installer|\n  assertDeploymentTarget(installer)\nend\n`;
    }
    utils_fs_1.writeFileSync(filename, replaced, 'utf-8');
}
async function migrateMainActivity(config) {
    const xmlData = await xml_1.readXML(path_1.join(config.android.srcMainDirAbs, 'AndroidManifest.xml'));
    const manifestNode = xmlData.manifest;
    const applicationChildNodes = manifestNode.application;
    let mainActivityClassPath = '';
    applicationChildNodes.find(applicationChildNode => {
        const activityChildNodes = applicationChildNode.activity;
        if (!Array.isArray(activityChildNodes)) {
            return false;
        }
        const mainActivityNode = activityChildNodes.find(activityChildNode => {
            const intentFilterChildNodes = activityChildNode['intent-filter'];
            if (!Array.isArray(intentFilterChildNodes)) {
                return false;
            }
            return intentFilterChildNodes.find(intentFilterChildNode => {
                const actionChildNodes = intentFilterChildNode.action;
                if (!Array.isArray(actionChildNodes)) {
                    return false;
                }
                const mainActionChildNode = actionChildNodes.find(actionChildNode => {
                    const androidName = actionChildNode.$['android:name'];
                    return androidName === 'android.intent.action.MAIN';
                });
                if (!mainActionChildNode) {
                    return false;
                }
                const categoryChildNodes = intentFilterChildNode.category;
                if (!Array.isArray(categoryChildNodes)) {
                    return false;
                }
                return categoryChildNodes.find(categoryChildNode => {
                    const androidName = categoryChildNode.$['android:name'];
                    return androidName === 'android.intent.category.LAUNCHER';
                });
            });
        });
        if (mainActivityNode) {
            mainActivityClassPath = mainActivityNode.$['android:name'];
        }
        return mainActivityNode;
    });
    const mainActivityClassName = mainActivityClassPath.split('.').pop();
    const mainActivityPathArray = mainActivityClassPath.split('.');
    mainActivityPathArray.pop();
    const mainActivityClassFileName = `${mainActivityClassName}.java`;
    const mainActivityClassFilePath = path_1.join(path_1.join(config.android.srcMainDirAbs, 'java'), ...mainActivityPathArray, mainActivityClassFileName);
    let data = readFile(mainActivityClassFilePath);
    if (data) {
        const bindex = data.indexOf('this.init(savedInstanceState');
        if (bindex !== -1) {
            const eindex = data.indexOf('}});', bindex) + 4;
            data = data.replace(data.substring(bindex, eindex), '');
            data = data.replace('// Initializes the Bridge', '');
        }
        const rindex = data.indexOf('registerPlugin');
        const superLine = 'super.onCreate(savedInstanceState);';
        if (rindex !== -1) {
            if (data.indexOf(superLine) < rindex) {
                const linePadding = rindex - data.indexOf(superLine) - superLine.length - 1;
                data = data.replace(`${superLine}\n${' '.repeat(linePadding)}`, '');
                const eindex = data.lastIndexOf('.class);') + 8;
                data = data.replace(data.substring(bindex, eindex), `${data.substring(bindex, eindex)}\n${' '.repeat(linePadding) + superLine.padStart(linePadding)}`);
            }
        }
        if (bindex == -1 && rindex == -1) {
            return;
        }
        utils_fs_1.writeFileSync(mainActivityClassFilePath, data);
    }
}
async function addNewSplashScreen(config) {
    const stylePath = path_1.join(config.android.srcMainDirAbs, 'res', 'values', 'styles.xml');
    let stylesXml = readFile(stylePath);
    if (!stylesXml)
        return;
    stylesXml = stylesXml.replace(`parent="AppTheme.NoActionBar"`, `parent="Theme.SplashScreen"`);
    // revert wrong replaces
    stylesXml = stylesXml.replace(`name="Theme.SplashScreen"`, `name="AppTheme.NoActionBar"`);
    stylesXml = stylesXml.replace(`name="Theme.SplashScreenLaunch"`, `name="AppTheme.NoActionBarLaunch"`);
    // Apply DayNight theme
    stylesXml = stylesXml.replace(`parent="Theme.AppCompat.NoActionBar"`, `parent="Theme.AppCompat.DayNight.NoActionBar"`);
    utils_fs_1.writeFileSync(stylePath, stylesXml);
}
