"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildAndroid = void 0;
const tslib_1 = require("tslib");
const path_1 = require("path");
const colors_1 = tslib_1.__importDefault(require("../colors"));
const common_1 = require("../common");
const log_1 = require("../log");
const subprocess_1 = require("../util/subprocess");
async function buildAndroid(config, buildOptions) {
    var _a;
    const releaseType = (_a = buildOptions.androidreleasetype) !== null && _a !== void 0 ? _a : 'AAB';
    const releaseTypeIsAAB = releaseType === 'AAB';
    const arg = releaseTypeIsAAB ? ':app:bundleRelease' : 'assembleRelease';
    const gradleArgs = [arg];
    if (!buildOptions.keystorepath ||
        !buildOptions.keystorealias ||
        !buildOptions.keystorealiaspass ||
        !buildOptions.keystorepass) {
        throw 'Missing options. Please supply all options for android signing. (Keystore Path, Keystore Password, Keystore Key Alias, Keystore Key Password)';
    }
    try {
        await common_1.runTask('Running Gradle build', async () => subprocess_1.runCommand('./gradlew', gradleArgs, {
            cwd: config.android.platformDirAbs,
        }));
    }
    catch (e) {
        if (e.includes('EACCES')) {
            throw `gradlew file does not have executable permissions. This can happen if the Android platform was added on a Windows machine. Please run ${colors_1.default.strong(`chmod +x ./${config.android.platformDir}/gradlew`)} and try again.`;
        }
        else {
            throw e;
        }
    }
    const releasePath = path_1.join(config.android.appDirAbs, 'build', 'outputs', releaseTypeIsAAB ? 'bundle' : 'apk', 'release');
    const unsignedReleaseName = `app${config.android.flavor ? `-${config.android.flavor}` : ''}-release${releaseTypeIsAAB ? '' : '-unsigned'}.${releaseType.toLowerCase()}`;
    const signedReleaseName = unsignedReleaseName.replace(`-release${releaseTypeIsAAB ? '' : '-unsigned'}.${releaseType.toLowerCase()}`, `-release-signed.${releaseType.toLowerCase()}`);
    const signingArgs = [
        '-sigalg',
        'SHA1withRSA',
        '-digestalg',
        'SHA1',
        '-keystore',
        buildOptions.keystorepath,
        '-keypass',
        buildOptions.keystorealiaspass,
        '-storepass',
        buildOptions.keystorepass,
        `-signedjar`,
        `${path_1.join(releasePath, signedReleaseName)}`,
        `${path_1.join(releasePath, unsignedReleaseName)}`,
        buildOptions.keystorealias,
    ];
    await common_1.runTask('Signing Release', async () => {
        await subprocess_1.runCommand('jarsigner', signingArgs, {
            cwd: config.android.platformDirAbs,
        });
    });
    log_1.logSuccess(`Successfully generated ${signedReleaseName} at: ${releasePath}`);
}
exports.buildAndroid = buildAndroid;
