"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getStartOffsetOfAttribute = exports.findAttributeOnElementWithAttrs = exports.findAttributeOnElementWithTag = exports.findElementsWithAttribute = void 0;
const parse5_1 = require("parse5");
/**
 * Parses a HTML fragment and traverses all AST nodes in order find elements that
 * include the specified attribute.
 */
function findElementsWithAttribute(html, attributeName) {
    const document = (0, parse5_1.parseFragment)(html, { sourceCodeLocationInfo: true });
    const elements = [];
    const visitNodes = (nodes) => {
        nodes.forEach(n => {
            const node = n;
            if (node.childNodes) {
                visitNodes(node.childNodes);
            }
            if (node.attrs?.some(attr => attr.name === attributeName.toLowerCase())) {
                elements.push(node);
            }
        });
    };
    visitNodes(document.childNodes);
    return elements;
}
exports.findElementsWithAttribute = findElementsWithAttribute;
/**
 * Finds elements with explicit tag names that also contain the specified attribute. Returns the
 * attribute start offset based on the specified HTML.
 */
function findAttributeOnElementWithTag(html, name, tagNames) {
    return findElementsWithAttribute(html, name)
        .filter(element => tagNames.includes(element.tagName))
        .map(element => getStartOffsetOfAttribute(element, name));
}
exports.findAttributeOnElementWithTag = findAttributeOnElementWithTag;
/**
 * Finds elements that contain the given attribute and contain at least one of the other
 * specified attributes. Returns the primary attribute's start offset based on the specified HTML.
 */
function findAttributeOnElementWithAttrs(html, name, attrs) {
    return findElementsWithAttribute(html, name)
        .filter(element => attrs.some(attr => hasElementAttribute(element, attr)))
        .map(element => getStartOffsetOfAttribute(element, name));
}
exports.findAttributeOnElementWithAttrs = findAttributeOnElementWithAttrs;
/** Shorthand function that checks if the specified element contains the given attribute. */
function hasElementAttribute(element, attributeName) {
    return element.attrs && element.attrs.some(attr => attr.name === attributeName.toLowerCase());
}
/** Gets the start offset of the given attribute from a Parse5 element. */
function getStartOffsetOfAttribute(element, attributeName) {
    return element.sourceCodeLocation.attrs[attributeName.toLowerCase()].startOffset;
}
exports.getStartOffsetOfAttribute = getStartOffsetOfAttribute;
//# sourceMappingURL=data:application/json;base64,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