import 'package:flutter/material.dart';
import 'package:yaru/icons.dart';

import 'yaru_icon_button.dart';

const _kAnimationDuration = Duration(milliseconds: 250);
const _kAnimationCurve = Curves.easeInOutCubic;

enum YaruExpandableButtonPosition {
  /// Align the button before the header widget
  start,

  /// Align the button at the opposite of the header widget
  end,
}

class YaruExpandable extends StatefulWidget {
  const YaruExpandable({
    super.key,
    required this.header,
    this.expandIcon,
    this.expandIconPadding = EdgeInsets.zero,
    this.expandIconSemanticLabel,
    this.expandButtonPosition = YaruExpandableButtonPosition.end,
    required this.child,
    this.collapsedChild,
    this.gapHeight = 4.0,
    this.isExpanded = false,
    this.onChange,
    this.usePadding = false,
  });

  /// Widget placed in the header, against the expand button.
  final Widget header;

  /// Icon used in the expand button.
  /// Prefer use a "right arrow" icon.
  /// A 25° rotation is used when expanded.
  final Widget? expandIcon;

  /// Optional padding around the expand button.
  final EdgeInsetsGeometry expandIconPadding;

  /// Optional semantic label to add to the expand button.
  final String? expandIconSemanticLabel;

  /// Controls expand button position, see [YaruExpandableButtonPosition].
  final YaruExpandableButtonPosition expandButtonPosition;

  /// Widget show when expanded.
  final Widget child;

  /// Widget show when collapsed.
  final Widget? collapsedChild;

  /// Gap between [header] and [child].
  final double gapHeight;

  /// Optional initial value.
  final bool isExpanded;

  /// Callback called on expand or collapse.
  final ValueChanged<bool>? onChange;

  /// If true, a [Padding] will be used instead of a [Column],
  /// around the [child] widget to add a gap between [header] and [child].
  /// This avoids unexpected behaviour with [Column], especially with a child
  /// that is not well sized.
  ///
  /// See: [#1024](https://github.com/ubuntu/yaru.dart/issues/1024)
  ///
  /// In the next major release, the [Padding] will be the default behaviour,
  /// and this flag will be removed.
  final bool usePadding;

  @override
  State<YaruExpandable> createState() => _YaruExpandableState();
}

class _YaruExpandableState extends State<YaruExpandable> {
  late bool _isExpanded;

  @override
  void initState() {
    _isExpanded = widget.isExpanded;
    super.initState();
  }

  @override
  void didUpdateWidget(YaruExpandable oldWidget) {
    super.didUpdateWidget(oldWidget);
    if (oldWidget.isExpanded != widget.isExpanded) {
      _isExpanded = widget.isExpanded;
    }
  }

  @override
  Widget build(BuildContext context) {
    final iconButton = Padding(
      padding: widget.expandIconPadding,
      child: YaruIconButton(
        iconSize: 36,
        padding: EdgeInsets.zero,
        onPressed: _onTap,
        icon: AnimatedRotation(
          turns: _isExpanded ? .25 : 0,
          duration: _kAnimationDuration,
          curve: _kAnimationCurve,
          child:
              widget.expandIcon ??
              Icon(
                YaruIcons.pan_end,
                semanticLabel: widget.expandIconSemanticLabel,
              ),
        ),
      ),
    );

    final header = Flexible(child: widget.header);

    final MainAxisAlignment expandButtonPosition;
    final List<Widget> headerChildren;

    switch (widget.expandButtonPosition) {
      case YaruExpandableButtonPosition.start:
        expandButtonPosition = MainAxisAlignment.start;
        headerChildren = [iconButton, header];
        break;
      case YaruExpandableButtonPosition.end:
        expandButtonPosition = MainAxisAlignment.spaceBetween;
        headerChildren = [header, iconButton];
        break;
    }

    return Column(
      children: [
        GestureDetector(
          behavior: HitTestBehavior.opaque,
          onTap: _onTap,
          child: Row(
            mainAxisAlignment: expandButtonPosition,
            children: headerChildren,
          ),
        ),
        AnimatedCrossFade(
          firstChild: _buildChild(widget.child),
          secondChild: widget.collapsedChild != null
              ? _buildChild(widget.collapsedChild!)
              : Container(),
          crossFadeState: _isExpanded
              ? CrossFadeState.showFirst
              : CrossFadeState.showSecond,
          sizeCurve: _kAnimationCurve,
          duration: _kAnimationDuration,
        ),
      ],
    );
  }

  void _onTap() {
    setState(() => _isExpanded = !_isExpanded);

    widget.onChange?.call(_isExpanded);
  }

  Widget _buildChild(Widget child) {
    if (widget.usePadding) {
      return Padding(
        padding: EdgeInsetsGeometry.only(top: widget.gapHeight),
        child: child,
      );
    }

    return Column(
      children: [
        SizedBox(height: widget.gapHeight),
        child,
      ],
    );
  }
}
