package com.absinthe.libchecker.features.snapshot.detail.ui.adapter.provider

import android.view.ContextThemeWrapper
import android.view.ViewGroup
import androidx.core.graphics.drawable.toDrawable
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.lifecycleScope
import com.absinthe.libchecker.R
import com.absinthe.libchecker.features.applist.detail.ui.LibDetailDialogFragment
import com.absinthe.libchecker.features.snapshot.detail.bean.ADDED
import com.absinthe.libchecker.features.snapshot.detail.bean.CHANGED
import com.absinthe.libchecker.features.snapshot.detail.bean.MOVED
import com.absinthe.libchecker.features.snapshot.detail.bean.REMOVED
import com.absinthe.libchecker.features.snapshot.detail.ui.adapter.node.SnapshotComponentNode
import com.absinthe.libchecker.features.snapshot.detail.ui.view.SnapshotDetailComponentView
import com.absinthe.libchecker.ui.base.BaseActivity
import com.absinthe.libchecker.utils.extensions.getColor
import com.absinthe.libraries.utils.utils.AntiShakeUtils
import com.absinthe.libraries.utils.utils.UiUtils
import com.absinthe.rulesbundle.LCRules
import com.chad.library.adapter.base.entity.node.BaseNode
import com.chad.library.adapter.base.provider.BaseNodeProvider
import com.chad.library.adapter.base.viewholder.BaseViewHolder
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

const val SNAPSHOT_COMPONENT_PROVIDER = 3

class SnapshotComponentProvider : BaseNodeProvider() {

  override val itemViewType: Int = SNAPSHOT_COMPONENT_PROVIDER
  override val layoutId: Int = 0

  override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): BaseViewHolder {
    return BaseViewHolder(
      SnapshotDetailComponentView(ContextThemeWrapper(context, R.style.AppListMaterialCard)).also {
        it.layoutParams = ViewGroup.LayoutParams(
          ViewGroup.LayoutParams.MATCH_PARENT,
          ViewGroup.LayoutParams.WRAP_CONTENT
        )
      }
    )
  }

  override fun convert(helper: BaseViewHolder, item: BaseNode) {
    (helper.itemView as SnapshotDetailComponentView).container.apply {
      val snapshotItem = (item as SnapshotComponentNode).item

      name.text = snapshotItem.title

      val colorRes = when (snapshotItem.diffType) {
        ADDED -> R.color.material_green_300
        REMOVED -> R.color.material_red_300
        CHANGED -> R.color.material_yellow_300
        MOVED -> R.color.material_blue_300
        else -> throw IllegalArgumentException("wrong diff type")
      }

      typeIcon.setImageResource(
        when (snapshotItem.diffType) {
          ADDED -> R.drawable.ic_add
          REMOVED -> R.drawable.ic_remove
          CHANGED -> R.drawable.ic_changed
          MOVED -> R.drawable.ic_move
          else -> throw IllegalArgumentException("wrong diff type")
        }
      )

      val baseColor = colorRes.getColor(context)
      val alpha = if (UiUtils.isDarkMode()) {
        (0.75f * 255).toInt() and 0xFF
      } else {
        (0.95f * 255).toInt() and 0xFF
      }
      val alphaColor = (baseColor and 0x00FFFFFF) or (alpha shl 24)
      background = alphaColor.toDrawable()

      (this@SnapshotComponentProvider.context as? LifecycleOwner)?.lifecycleScope?.launch {
        val rule = LCRules.getRule(snapshotItem.name, snapshotItem.itemType, true)

        withContext(Dispatchers.Main) {
          setChip(rule, alphaColor)
          if (rule != null) {
            setChipOnClickListener {
              if (AntiShakeUtils.isInvalidClick(it)) {
                return@setChipOnClickListener
              }
              val name = item.item.name
              val fragmentManager =
                (this@SnapshotComponentProvider.context as BaseActivity<*>).supportFragmentManager
              LibDetailDialogFragment.newInstance(name, item.item.itemType, rule.regexName)
                .show(fragmentManager, LibDetailDialogFragment::class.java.name)
            }
          } else {
            setChipOnClickListener(null)
          }
        }
      }
    }
  }
}
