package com.a4a.g8invoicing.data

import android.util.Log
import androidx.compose.ui.text.input.TextFieldValue
import app.cash.sqldelight.coroutines.asFlow
import com.a4a.g8invoicing.Database
import com.a4a.g8invoicing.ui.shared.calculatePriceWithTax
import com.a4a.g8invoicing.ui.states.ClientRef
import com.a4a.g8invoicing.ui.states.DocumentProductState
import com.a4a.g8invoicing.ui.states.ProductPrice
import com.a4a.g8invoicing.ui.states.ProductState
import g8invoicing.DocumentProduct
import g8invoicing.Product
import g8invoicing.ProductPriceQueries
import g8invoicing.TaxRateQueries
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.withContext
import java.math.BigDecimal
import java.math.RoundingMode

// Product is a basic product: name, description, prices
// DocumentProduct is a product added to a document: quantity, name, description, price, currency

class ProductLocalDataSource(
    db: Database,
) : ProductLocalDataSourceInterface {
    private val productQueries = db.productQueries
    private val taxQueries = db.taxRateQueries
    private val documentProductQueries = db.documentProductQueries
    private val productPriceQueries = db.productPriceQueries

    override suspend fun fetchProduct(id: Long): ProductState? {
        return withContext(Dispatchers.IO) {
            productQueries.getProduct(id).executeAsOneOrNull()
                ?.transformIntoEditableProduct(taxQueries, productPriceQueries)
        }
    }

    override fun fetchAllProducts(): Flow<List<ProductState>> {
        return productQueries.getAllProducts()
            .asFlow()
            .map { query ->
                query.executeAsList()
                    .map { it.transformIntoEditableProduct(taxQueries, productPriceQueries) }
            }
    }

    override suspend fun saveProduct(product: ProductState) {
        return withContext(Dispatchers.IO) {
            try {
                // Start a transaction to ensure both product and its price are saved, or neither.
                productQueries.transaction {
                    // 1. Save the Product itself
                    productQueries.saveProduct(
                        id = null, // Assuming id is autogenerated for new products
                        name = product.name.text,
                        description = product.description?.text,
                        product_tax_id = product.taxRate?.let {
                            taxQueries.getTaxRateId(it.toDouble()).executeAsOneOrNull()
                        },
                        unit = product.unit?.text
                    )

                    val lastInsertedProductId = productQueries.lastInsertRowId().executeAsOne()

                    // 2. Save the default price
                    product.defaultPriceWithoutTax?.let { price ->
                        productPriceQueries.saveProductPrice(
                            id = null,
                            product_id = lastInsertedProductId,
                            client_id = null,
                            price_without_tax = price.toDouble()
                        )
                    }

                    // 3. Save other client-specific prices
                    product.additionalPrices?.forEach { price ->
                        price.clients.forEach { client ->
                            productPriceQueries.saveProductPrice(
                                id = null,
                                product_id = lastInsertedProductId,
                                client_id = client.id.toLong(),
                                price_without_tax = price.priceWithoutTax?.toDouble()
                            )
                        }
                    }
                }
            } catch (cause: Throwable) {
                println("Error saving product: ${cause.message}")
            }
        }
    }

    override suspend fun duplicateProducts(
        products: List<ProductState>,
        duplicateNameSuffix: String,
    ) {
        return withContext(Dispatchers.IO) {
            try {
                products.forEach { productToDuplicate ->
                    val originalProduct =
                        productToDuplicate.id?.let { fetchProduct(it.toLong()) }

                    originalProduct?.let { productToDuplicate ->
                        // Start a transaction for duplicating the product and all its prices
                        productQueries.transaction {
                            // 1. Save the new Product (core details)
                            productQueries.saveProduct(
                                id = null,
                                name = "${productToDuplicate.name.text}$duplicateNameSuffix",
                                description = productToDuplicate.description?.text,
                                product_tax_id = productToDuplicate.taxRate?.let {
                                    taxQueries.getTaxRateId(it.toDouble()).executeAsOneOrNull()
                                },
                                unit = productToDuplicate.unit?.text
                            )

                            // Get the ID of the newly duplicated product
                            val newProductId = productQueries.lastInsertRowId().executeAsOne()

                            // 2. Save the Default Price for the duplicated product
                            // Assuming 'defaultPriceWithoutTax' is available in your fetched ProductState
                            productToDuplicate.defaultPriceWithoutTax?.let { price ->
                                productPriceQueries.saveProductPrice(
                                    id = null,
                                    product_id = newProductId,
                                    client_id = null, // No client for this price means it's the default price
                                    price_without_tax = price.toDouble()
                                )
                            }

                            // 3. Save the Additional (Client-Specific) Prices for the duplicated product
                            productToDuplicate.additionalPrices?.forEach { price ->
                                price.clients.forEach { client ->
                                    productPriceQueries.saveProductPrice(
                                        id = null,
                                        product_id = newProductId,
                                        client_id = client.id.toLong(),
                                        price_without_tax = price.priceWithoutTax?.toDouble()
                                    )
                                }
                            }

                        }
                    } ?: run {
                        println("Warning: Product with id ${productToDuplicate.id} not found for duplication.")
                    }
                }
            } catch (cause: Throwable) {
                println("Error duplicating products: ${cause.message}")
            }
        }
    }

    override suspend fun updateProduct(product: ProductState) {
        return withContext(Dispatchers.IO) {
            // Ensure the product has an ID, as we are updating an existing one.
            val productId = product.id?.toLong() ?: run {
                println("Error: Product ID is null, cannot update.")
                // Optionally throw an exception or return a failure result
                return@withContext
            }

            try {
                productQueries.transaction {
                    // 1. Update the core Product details
                    productQueries.updateProduct(
                        id = productId,
                        name = product.name.text,
                        description = product.description?.text,
                        product_tax_id = product.taxRate?.let {
                            taxQueries.getTaxRateId(it.toDouble()).executeAsOneOrNull()
                        },
                        unit = product.unit?.text
                    )

                    // 2. Delete all existing prices for this product
                    productPriceQueries.deleteAllPricesForProduct(productId)

                    // 3. Save the new Default Price
                    product.defaultPriceWithoutTax?.let { price ->
                        productPriceQueries.saveProductPrice(
                            id = null,
                            product_id = productId, // Use the existing product's ID
                            client_id = null, // Default price
                            price_without_tax = price.toDouble()
                        )
                    }

                    // 4. Save the new Additional (Client-Specific) Prices
                    product.additionalPrices?.forEach { price ->
                        price.clients.forEach { client ->
                            productPriceQueries.saveProductPrice(
                                id = null,
                                product_id = productId,
                                client_id = client.id.toLong(),
                                price_without_tax = price.priceWithoutTax?.toDouble()
                            )
                        }
                    }
                }
            } catch (cause: Throwable) {
                println("Error updating product with id $productId: ${cause.message}")
            }
        }
    }


    override suspend fun updateDocumentProduct(documentProduct: DocumentProductState) {
        return withContext(Dispatchers.IO) {
            try {
                documentProduct.id?.toLong()?.let {
                    documentProductQueries.updateDocumentProduct(
                        document_product_id = it,
                        name = documentProduct.name.text,
                        quantity = documentProduct.quantity.toDouble(),
                        description = documentProduct.description?.text,
                        unit = documentProduct.unit?.text,
                        tax_rate = documentProduct.taxRate?.toDouble(),
                        price_without_tax = documentProduct.priceWithoutTax?.toDouble()
                    )
                }
            } catch (cause: Throwable) {
                println("Error updating documentProduct $documentProduct: ${cause.message}")
            }
        }
    }

    override suspend fun deleteProduct(id: Long) {
        return withContext(Dispatchers.IO) {
            try {
                productQueries.transaction {
                    productPriceQueries.deleteAllPricesForProduct(id)
                    productQueries.deleteProduct(id)
                }
            } catch (cause: Throwable) {
                println("Error deleting product with id $id: ${cause.message}")
            }
        }
    }

    override suspend fun deleteAdditionalPrice(productId: Long, priceWithoutTax: BigDecimal) {
        return withContext(Dispatchers.IO) {
            try {
                productQueries.transaction {
                    productPriceQueries.deletePriceForClients(productId, priceWithoutTax.toDouble())
                }
            } catch (cause: Throwable) {
                println("Error deleting price with value $priceWithoutTax : ${cause.message}")
            }
        }
    }

    override suspend fun removeClientFromAdditionalPrice(productId: Long, clientId: Long) {
        return withContext(Dispatchers.IO) {
            try {
                productQueries.transaction {
                    productPriceQueries.deletePriceForClient(productId, clientId)
                }
            } catch (cause: Throwable) {
                println("Error deleting price for client $clientId : ${cause.message}")
            }
        }
    }


    override suspend fun deleteDocumentProducts(ids: List<Long>) {
        return withContext(Dispatchers.IO) {
            ids.forEach {
                documentProductQueries.deleteDocumentProduct(it)
            }
        }
    }
}

fun Product.transformIntoEditableProduct(
    taxQueries: TaxRateQueries,
    productPriceQueries: ProductPriceQueries,
): ProductState {

    val taxRate = this.product_tax_id
        ?.let { taxQueries.getTaxRate(it).executeAsOneOrNull() }
        ?.toBigDecimal()

    // ─────────────────────────────
    // Récupère TOUS les prix (avec et sans clients)
    // ─────────────────────────────
    val rows = productPriceQueries
        .getAdditionalPricesWithClients(this.id)
        .executeAsList()

    // ─────────────────────────────
    // Sépare prix par défaut vs additionnels
    // ─────────────────────────────
    val defaultPriceRow = rows.firstOrNull { it.client_id == null }

    val defaultPriceWithoutTax = defaultPriceRow
        ?.price_without_tax
        ?.toBigDecimal()
        ?.setScale(2, RoundingMode.HALF_UP)

    val defaultPriceWithTax = defaultPriceWithoutTax?.let { price ->
        taxRate?.let { tax ->
            calculatePriceWithTax(price, tax)
        } ?: price // Si pas de taxe, prix TTC = prix HT
    }

    // ─────────────────────────────
    // Additional prices: ceux qui ont un client_id
    // ─────────────────────────────
    val additionalRows = rows.filter { it.client_id != null }

    val additionalPrices = if (additionalRows.isNotEmpty()) {
        // Groupe par montant de prix
        val priceToClients = mutableMapOf<BigDecimal, MutableList<ClientRef>>()

        additionalRows.forEach { row ->
            val price = row.price_without_tax
                ?.toBigDecimal()
                ?.setScale(2, RoundingMode.HALF_UP)
                ?: return@forEach

            val clientId = row.client_id ?: return@forEach

            val displayName = listOfNotNull(
                row.first_name?.takeIf { it.isNotBlank() },
                row.name
            ).joinToString(" ")

            val clientRef = ClientRef(
                id = clientId.toInt(),
                name = displayName
            )

            priceToClients
                .getOrPut(price) { mutableListOf() }
                .add(clientRef)
        }

        println("additionalPrices size = ${priceToClients.size}")

        priceToClients.map { (price, clients) ->
            val productPrice = ProductPrice(
                priceWithoutTax = price,
                priceWithTax = taxRate?.let {
                    calculatePriceWithTax(price, it)
                } ?: price, // Si pas de taxe, prix TTC = prix HT
                clients = clients
            )
            println(productPrice)
            productPrice
        }.ifEmpty { null }
    } else {
        null
    }

    // ─────────────────────────────
    // Final state
    // ─────────────────────────────
    return ProductState(
        id = this.id.toInt(),
        name = TextFieldValue(this.name),
        description = this.description?.let { TextFieldValue(it) },
        defaultPriceWithoutTax = defaultPriceWithoutTax,
        defaultPriceWithTax = defaultPriceWithTax,
        taxRate = taxRate,
        unit = this.unit?.let { TextFieldValue(it) } ?: TextFieldValue(""),
        additionalPrices = additionalPrices
    )
}



fun DocumentProduct.transformIntoEditableDocumentProduct(
    linkedDate: String? = null,
    linkedDocNumber: String? = null,
    sortOrder: Int?
): DocumentProductState {
    return DocumentProductState(
        id = this.id.toInt(),
        name = TextFieldValue(this.name),
        description = this.description?.let { TextFieldValue(it) },
        priceWithoutTax = this.price_without_tax?.toBigDecimal(),
        priceWithTax = this.price_without_tax?.toBigDecimal()?.let { priceWithoutTax ->
            this.tax_rate?.toBigDecimal()?.let { tax ->
                calculatePriceWithTax(priceWithoutTax, tax)
            }
        },
        taxRate = this.tax_rate?.toBigDecimal(),
        quantity = this.quantity.toBigDecimal(),
        unit = TextFieldValue(this.unit ?: ""),
        productId = this.product_id?.toInt(),
        linkedDate = linkedDate,
        linkedDocNumber = linkedDocNumber,
        sortOrder = sortOrder
    )
}

