package com.a4a.g8invoicing.ui.navigation

import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import androidx.navigation.NavController
import androidx.navigation.NavGraphBuilder
import androidx.navigation.compose.composable
import androidx.navigation.navArgument
import com.a4a.g8invoicing.ui.viewmodels.ClientOrIssuerAddEditViewModel
import com.a4a.g8invoicing.ui.viewmodels.ClientOrIssuerListViewModel
import com.a4a.g8invoicing.ui.viewmodels.ClientOrIssuerType
import com.a4a.g8invoicing.ui.screens.shared.DocumentAddEdit
import com.a4a.g8invoicing.ui.viewmodels.ProductAddEditViewModel
import com.a4a.g8invoicing.ui.viewmodels.ProductListViewModel
import com.a4a.g8invoicing.ui.viewmodels.ProductType
import com.a4a.g8invoicing.ui.screens.shared.DocumentBottomSheetTypeOfForm
import com.a4a.g8invoicing.ui.shared.ScreenElement
import com.a4a.g8invoicing.ui.viewmodels.CreditNoteAddEditViewModel
import com.itextpdf.kernel.pdf.PdfName.a
import kotlinx.coroutines.launch

fun NavGraphBuilder.creditNoteAddEdit(
    navController: NavController,
    onClickBack: () -> Unit,
) {
    composable(
        route = Screen.CreditNoteAddEdit.name + "?itemId={itemId}",
        arguments = listOf(
            navArgument("itemId") { nullable = true },
        )
    ) { backStackEntry ->
        val scope = rememberCoroutineScope()

        val creditNoteViewModel: CreditNoteAddEditViewModel = hiltViewModel()
        val uiState by creditNoteViewModel.documentUiState.collectAsStateWithLifecycle()

        val clientOrIssuerListViewModel: ClientOrIssuerListViewModel = hiltViewModel()
        val clientListUiState by clientOrIssuerListViewModel.clientsUiState
            .collectAsStateWithLifecycle()
        val issuerListUiState by clientOrIssuerListViewModel.issuersUiState
            .collectAsStateWithLifecycle()

        val clientOrIssuerAddEditViewModel: ClientOrIssuerAddEditViewModel = hiltViewModel()
        val documentClientUiState by clientOrIssuerAddEditViewModel.documentClientUiState.collectAsState()
        val documentIssuerUiState by clientOrIssuerAddEditViewModel.documentIssuerUiState.collectAsState()

        val productListViewModel: ProductListViewModel = hiltViewModel()
        val productListUiState by productListViewModel.productsUiState
            .collectAsStateWithLifecycle()

        val productAddEditViewModel: ProductAddEditViewModel = hiltViewModel()
        val documentProduct by productAddEditViewModel.documentProductUiState.collectAsState()

        var showDocumentForm by remember { mutableStateOf(false) }
        var moveDocumentPagerToLastPage by remember { mutableStateOf(false) }

        // Get result from "Add new" screen, to know if it's
        // a client or issuer that has been added
        DocumentAddEdit(
            navController = navController,
            document = uiState,
            onClickBack = onClickBack,
            clientList = clientListUiState.clientsOrIssuerList.toMutableList(),
            issuerList = issuerListUiState.clientsOrIssuerList.toMutableList(),
            documentClientUiState = documentClientUiState,
            documentIssuerUiState = documentIssuerUiState,
            documentProductUiState = documentProduct, // Used when choosing a product or creating new product from the bottom sheet
            taxRates = productAddEditViewModel.fetchTaxRatesFromLocalDb(),
            products = productListUiState.products.toMutableList(), // The list of products to display when adding a product
            onValueChange = { pageElement, value ->
                creditNoteViewModel.updateUiState(pageElement, value)
            },
            onSelectProduct = { product, clientId ->
                // Initialize documentProductUiState to display it in the bottomSheet form
                productAddEditViewModel.setDocumentProductUiStateWithProduct(
                    product,
                    clientId
                )
            },
            onClickNewDocumentProduct = {
                productAddEditViewModel.clearProductNameAndDescription()
            },
            onSelectClientOrIssuer = { clientOrIssuer ->
                // Conserver l'ID original du client avant de le transformer en document client
                clientOrIssuer.originalClientId = clientOrIssuer.id
                if (clientOrIssuer.type == ClientOrIssuerType.CLIENT) {
                    documentClientUiState.type = ClientOrIssuerType.DOCUMENT_CLIENT
                    clientOrIssuer.type = ClientOrIssuerType.DOCUMENT_CLIENT
                } else {
                    documentIssuerUiState.type = ClientOrIssuerType.DOCUMENT_ISSUER
                    clientOrIssuer.type = ClientOrIssuerType.DOCUMENT_ISSUER
                }
                creditNoteViewModel.saveDocumentClientOrIssuerInLocalDb(clientOrIssuer)
                creditNoteViewModel.saveDocumentClientOrIssuerInUiState(clientOrIssuer)

            },
            onClickEditDocumentProduct = {// Edit a document product
                productAddEditViewModel.setDocumentProductUiState(it)
            },
            onClickNewDocumentClientOrIssuer = {
                clientOrIssuerAddEditViewModel.clearClientOrIssuerUiState(it)
            },
            onClickDocumentClientOrIssuer = {// Edit a document client/issuer
                clientOrIssuerAddEditViewModel.setDocumentClientOrIssuerUiState(it)
            },
            onClickDeleteDocumentProduct = {
                creditNoteViewModel.removeDocumentProductFromUiState(it)
                creditNoteViewModel.removeDocumentProductFromLocalDb(it)
            },
            onClickDeleteDocumentClientOrIssuer = { type ->
                creditNoteViewModel.removeDocumentClientOrIssuerFromUiState(type)
                creditNoteViewModel.removeDocumentClientOrIssuerFromLocalDb(type)
            },
            placeCursorAtTheEndOfText = { pageElement ->
                if (pageElement == ScreenElement.DOCUMENT_NUMBER ||
                    pageElement == ScreenElement.DOCUMENT_REFERENCE
                ) {
                    creditNoteViewModel.updateTextFieldCursorOfCreditNoteState(pageElement)
                }
            },
            bottomFormOnValueChange = { pageElement, value, type ->
                if (pageElement.name.contains("PRODUCT")) {
                    productAddEditViewModel.updateProductState(
                        pageElement,
                        value,
                        ProductType.DOCUMENT_PRODUCT
                    )
                } else {
                    type?.let {
                        clientOrIssuerAddEditViewModel.updateClientOrIssuerState(
                            pageElement,
                            value,
                            it
                        )
                    }
                }
            },
            bottomFormPlaceCursor = { pageElement, clientOrIssuer ->
                if (pageElement.name.contains(ProductType.DOCUMENT_PRODUCT.name)) {
                    productAddEditViewModel.updateCursor(pageElement, ProductType.DOCUMENT_PRODUCT)
                } else if (clientOrIssuer == ClientOrIssuerType.DOCUMENT_ISSUER) {
                    clientOrIssuerAddEditViewModel.updateCursor(
                        pageElement,
                        ClientOrIssuerType.DOCUMENT_ISSUER
                    )
                } else if (clientOrIssuer == ClientOrIssuerType.DOCUMENT_CLIENT) {
                    clientOrIssuerAddEditViewModel.updateCursor(
                        pageElement,
                        ClientOrIssuerType.DOCUMENT_CLIENT
                    )
                }
            },
            onClickDoneForm = { typeOfCreation ->
                scope.launch {
                    when (typeOfCreation) {
                        // NEW = create new & save in clients list too
                        DocumentBottomSheetTypeOfForm.NEW_CLIENT -> {
                            if (clientOrIssuerAddEditViewModel.validateInputs(ClientOrIssuerType.DOCUMENT_CLIENT)) {
                                createNewClientOrIssuer(
                                    clientOrIssuerAddEditViewModel,
                                    ClientOrIssuerType.DOCUMENT_CLIENT
                                )
                                documentClientUiState.type = ClientOrIssuerType.DOCUMENT_CLIENT
                                creditNoteViewModel.saveDocumentClientOrIssuerInUiState(
                                    documentClientUiState
                                )
                                creditNoteViewModel.saveDocumentClientOrIssuerInLocalDb(
                                    documentClientUiState
                                )
                                showDocumentForm = false
                            }
                        }
                        // EDIT = edit the chosen item (will only impact the document, doesn't change
                        // the initial object)
                        DocumentBottomSheetTypeOfForm.EDIT_CLIENT -> {
                            if (clientOrIssuerAddEditViewModel.validateInputs(ClientOrIssuerType.DOCUMENT_CLIENT)) {
                                creditNoteViewModel.updateUiState(
                                    ScreenElement.DOCUMENT_CLIENT,
                                    documentClientUiState
                                )
                                clientOrIssuerAddEditViewModel.updateClientOrIssuerInLocalDb(
                                    ClientOrIssuerType.DOCUMENT_ISSUER,
                                    documentIssuerUiState
                                )
                                showDocumentForm = false
                            }
                        }

                        DocumentBottomSheetTypeOfForm.NEW_ISSUER -> {
                            if (clientOrIssuerAddEditViewModel.validateInputs(ClientOrIssuerType.DOCUMENT_ISSUER)) {
                                createNewClientOrIssuer(
                                    clientOrIssuerAddEditViewModel,
                                    ClientOrIssuerType.DOCUMENT_ISSUER
                                )
                                documentIssuerUiState.type = ClientOrIssuerType.DOCUMENT_ISSUER
                                creditNoteViewModel.saveDocumentClientOrIssuerInUiState(
                                    documentIssuerUiState
                                )
                                creditNoteViewModel.saveDocumentClientOrIssuerInLocalDb(
                                    documentIssuerUiState
                                )
                                showDocumentForm = false
                            }
                        }

                        DocumentBottomSheetTypeOfForm.EDIT_ISSUER -> {
                            if (clientOrIssuerAddEditViewModel.validateInputs(ClientOrIssuerType.DOCUMENT_ISSUER)) {
                                creditNoteViewModel.updateUiState(
                                    ScreenElement.DOCUMENT_ISSUER,
                                    documentIssuerUiState
                                )
                                clientOrIssuerAddEditViewModel.updateClientOrIssuerInLocalDb(
                                    ClientOrIssuerType.DOCUMENT_ISSUER,
                                    documentIssuerUiState
                                )
                                showDocumentForm = false
                            }
                        }

                        DocumentBottomSheetTypeOfForm.ADD_EXISTING_PRODUCT -> {
                            if (productAddEditViewModel.validateInputs(ProductType.DOCUMENT_PRODUCT)) {
                                // 1. Save DocumentProduct in db and get the id
                                val documentProductId =
                                    creditNoteViewModel.saveDocumentProductInLocalDbAndGetId(documentProduct)
                                // 2. Update UI only if ID is obtained
                                if (documentProductId != null) {
                                    creditNoteViewModel.saveDocumentProductInUiState(documentProduct.copy(id = documentProductId))

                                    // 5. Close bottom sheet
                                    showDocumentForm = false
                                } else {
                                    // Les validations ont échoué
                                    //println("Validation failed for new product.")
                                    // Le message d'erreur de validation devrait déjà être affiché dans le formulaire
                                    // Ne pas fermer le formulaire.
                                }
                            }
                        }

                        DocumentBottomSheetTypeOfForm.NEW_PRODUCT -> {
                            if (productAddEditViewModel.validateInputs(ProductType.DOCUMENT_PRODUCT)) {
                                // 1. Updating product viewModel
                                productAddEditViewModel.setProductUiState()
                                productAddEditViewModel.saveProductInLocalDb()

                                // 2. Save DocumentProduct in db and get the id
                                val documentProductId =
                                    creditNoteViewModel.saveDocumentProductInLocalDbAndGetId(documentProduct)
                                // 3. Update UI only if ID is obtained
                                if (documentProductId != null) {
                                    creditNoteViewModel.saveDocumentProductInUiState(documentProduct.copy(id = documentProductId))
                                    // 4. Clear product viewModel
                                    productAddEditViewModel.clearProductUiState()

                                    // 5. Close bottom sheet
                                    showDocumentForm = false
                                } else {
                                    // Les validations ont échoué
                                    //println("Validation failed for new product.")
                                    // Le message d'erreur de validation devrait déjà être affiché dans le formulaire
                                    // Ne pas fermer le formulaire.
                                }
                            }
                        }

                        DocumentBottomSheetTypeOfForm.EDIT_PRODUCT -> {
                            if (productAddEditViewModel.validateInputs(ProductType.DOCUMENT_PRODUCT)) {
                                creditNoteViewModel.updateUiState(
                                    ScreenElement.DOCUMENT_PRODUCT,
                                    documentProduct
                                )
                                showDocumentForm = false
                            }
                        }

                        else -> null
                    }
                }
            },
            onClickCancelForm = {
            },
            onSelectTaxRate = {
                productAddEditViewModel.updateTaxRate(
                    it,
                    ProductType.DOCUMENT_PRODUCT
                )
            },
            showDocumentForm = showDocumentForm,
            onShowDocumentForm = {
                showDocumentForm = it
            },
            onClickDeleteAddress = {
                clientOrIssuerAddEditViewModel.removeAddressFromClientOrIssuerState(it)
            },
            onOrderChange = creditNoteViewModel::updateDocumentProductsOrderInUiStateAndDb
        )
    }
}
