package com.arturo254.opentune.ui.screens.playlist

import androidx.activity.compose.BackHandler
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.background
import androidx.compose.foundation.combinedClickable
import androidx.compose.foundation.gestures.ScrollableDefaults
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.asPaddingValues
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.ime
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.union
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.windowInsetsPadding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.material3.TextFieldDefaults
import androidx.compose.material3.TopAppBar
import androidx.compose.material3.TopAppBarScrollBehavior
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.derivedStateOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.alpha
import androidx.compose.ui.draw.clip
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.hapticfeedback.HapticFeedbackType
import androidx.compose.ui.platform.LocalHapticFeedback
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.pluralStringResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.LinkAnnotation
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.input.ImeAction
import androidx.compose.ui.text.input.TextFieldValue
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.text.withLink
import androidx.compose.ui.text.withStyle
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.compose.ui.util.fastAny
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.navigation.NavController
import coil.compose.AsyncImage
import com.arturo254.innertube.models.SongItem
import com.arturo254.innertube.models.WatchEndpoint
import com.arturo254.opentune.LocalDatabase
import com.arturo254.opentune.LocalPlayerAwareWindowInsets
import com.arturo254.opentune.LocalPlayerConnection
import com.arturo254.opentune.R
import com.arturo254.opentune.constants.AlbumThumbnailSize
import com.arturo254.opentune.constants.HideExplicitKey
import com.arturo254.opentune.constants.ThumbnailCornerRadius
import com.arturo254.opentune.db.entities.PlaylistEntity
import com.arturo254.opentune.db.entities.PlaylistSongMap
import com.arturo254.opentune.extensions.metadata
import com.arturo254.opentune.extensions.toMediaItem
import com.arturo254.opentune.extensions.togglePlayPause
import com.arturo254.opentune.models.toMediaMetadata
import com.arturo254.opentune.playback.queues.YouTubeQueue
import com.arturo254.opentune.ui.component.AutoResizeText
import com.arturo254.opentune.ui.component.FontSizeRange
import com.arturo254.opentune.ui.component.LocalMenuState
import com.arturo254.opentune.ui.component.VerticalFastScroller
import com.arturo254.opentune.ui.component.YouTubeListItem
import com.arturo254.opentune.ui.component.shimmer.ButtonPlaceholder
import com.arturo254.opentune.ui.component.shimmer.ListItemPlaceHolder
import com.arturo254.opentune.ui.component.shimmer.ShimmerHost
import com.arturo254.opentune.ui.component.shimmer.TextPlaceholder
import com.arturo254.opentune.ui.menu.SelectionMediaMetadataMenu
import com.arturo254.opentune.ui.menu.YouTubePlaylistMenu
import com.arturo254.opentune.ui.menu.YouTubeSongMenu
import com.arturo254.opentune.ui.utils.ItemWrapper
import com.arturo254.opentune.utils.rememberPreference
import com.arturo254.opentune.viewmodels.OnlinePlaylistViewModel

@OptIn(ExperimentalMaterial3Api::class, ExperimentalFoundationApi::class)
@Composable
fun OnlinePlaylistScreen(
    navController: NavController,
    scrollBehavior: TopAppBarScrollBehavior,
    viewModel: OnlinePlaylistViewModel = hiltViewModel(),
) {
    val menuState = LocalMenuState.current
    val database = LocalDatabase.current
    val haptic = LocalHapticFeedback.current
    val playerConnection = LocalPlayerConnection.current ?: return
    val isPlaying by playerConnection.isPlaying.collectAsState()
    val mediaMetadata by playerConnection.mediaMetadata.collectAsState()

    val playlist by viewModel.playlist.collectAsState()
    val songs by viewModel.playlistSongs.collectAsState()
    val dbPlaylist by viewModel.dbPlaylist.collectAsState()

    var selection by remember {
        mutableStateOf(false)
    }
    val hideExplicit by rememberPreference(key = HideExplicitKey, defaultValue = false)

    val lazyListState = rememberLazyListState()
    val coroutineScope = rememberCoroutineScope()
    val snackbarHostState = remember { SnackbarHostState() }

    var isSearching by rememberSaveable { mutableStateOf(false) }
    var query by rememberSaveable(stateSaver = TextFieldValue.Saver) {
        mutableStateOf(TextFieldValue())
    }
    val filteredSongs =
        remember(songs, query) {
            if (query.text.isEmpty()) {
                songs.mapIndexed { index, song -> index to song }
            } else {
                songs
                    .mapIndexed { index, song -> index to song }
                    .filter { (_, song) ->
                        song.title.contains(query.text, ignoreCase = true) ||
                                song.artists.fastAny {
                                    it.name.contains(
                                        query.text,
                                        ignoreCase = true
                                    )
                                }
                    }
            }
        }
    val focusRequester = remember { FocusRequester() }
    LaunchedEffect(isSearching) {
        if (isSearching) {
            focusRequester.requestFocus()
        }
    }
    if (isSearching) {
        BackHandler {
            isSearching = false
            query = TextFieldValue()
        }
    }








    LaunchedEffect(playlist?.id) {
        // Restaurar posición si cambias de playlist
        if (playlist?.id != null) {
            lazyListState.scrollToItem(0)
        }
    }

    val showTopBarTitle by remember {
        derivedStateOf {
            lazyListState.firstVisibleItemIndex > 0
        }
    }

    // Calcular el padding del contenido considerando WindowInsets
    val contentPadding = LocalPlayerAwareWindowInsets.current.union(WindowInsets.ime)
        .asPaddingValues()
    val wrappedSongs = filteredSongs.map { item -> ItemWrapper(item) }.toMutableList()

    Box(
        modifier = Modifier.fillMaxSize(),
    ) {
        VerticalFastScroller(
            listState = lazyListState,
            topContentPadding = contentPadding.calculateTopPadding(),
            endContentPadding = 0.dp
        ) {
            LazyColumn(
                state = lazyListState,
                contentPadding = contentPadding,
                flingBehavior = ScrollableDefaults.flingBehavior(),
                userScrollEnabled = true,
            ) {
                playlist.let { playlist ->
                    if (playlist != null) {
                        if (!isSearching) {
                            item {
                                Column(
                                    modifier =
                                        Modifier
                                            .padding(12.dp)
                                            .animateItem(),
                                ) {
                                    Row(
                                        verticalAlignment = Alignment.CenterVertically,
                                    ) {
                                        Box(
                                            contentAlignment = Alignment.Center,
                                            modifier = Modifier
                                                .size(AlbumThumbnailSize)
                                                .clip(RoundedCornerShape(ThumbnailCornerRadius))
                                                .fillMaxWidth(),
                                        ) {
                                            AsyncImage(
                                                model = playlist.thumbnail,
                                                contentDescription = null,
                                                modifier = Modifier
                                                    .fillMaxWidth()
                                                    .clip(RoundedCornerShape(ThumbnailCornerRadius)),
                                            )
                                        }

                                        Spacer(Modifier.width(16.dp))

                                        Column(
                                            verticalArrangement = Arrangement.Center,
                                        ) {
                                            AutoResizeText(
                                                text = playlist.title,
                                                fontWeight = FontWeight.Bold,
                                                maxLines = 2,
                                                overflow = TextOverflow.Ellipsis,
                                                fontSizeRange = FontSizeRange(16.sp, 22.sp),
                                            )

                                            playlist.author?.let { artist ->
                                                Text(
                                                    buildAnnotatedString {
                                                        withStyle(
                                                            style =
                                                                MaterialTheme.typography.titleMedium
                                                                    .copy(
                                                                        fontWeight = FontWeight.Normal,
                                                                        color = MaterialTheme.colorScheme.onBackground,
                                                                    ).toSpanStyle(),
                                                        ) {
                                                            if (artist.id != null) {
                                                                val link =
                                                                    LinkAnnotation.Clickable(artist.id!!) {
                                                                        navController.navigate("artist/${artist.id!!}")
                                                                    }
                                                                withLink(link) {
                                                                    append(artist.name)
                                                                }
                                                            } else {
                                                                append(artist.name)
                                                            }
                                                        }
                                                    },
                                                )
                                            }

                                            playlist.songCountText?.let { songCountText ->
                                                Text(
                                                    text = songCountText,
                                                    style = MaterialTheme.typography.titleMedium,
                                                    fontWeight = FontWeight.Normal,
                                                )
                                            }

                                            Row {
                                                if (playlist.id != "LM") {
                                                    IconButton(
                                                        onClick = {
                                                            if (dbPlaylist?.playlist == null) {
                                                                database.transaction {
                                                                    val playlistEntity =
                                                                        PlaylistEntity(
                                                                            name = playlist.title,
                                                                            browseId = playlist.id,
                                                                            isEditable = playlist.isEditable,
                                                                            playEndpointParams = playlist.playEndpoint?.params,
                                                                            shuffleEndpointParams = playlist.shuffleEndpoint?.params,
                                                                            radioEndpointParams = playlist.radioEndpoint?.params
                                                                        ).toggleLike()
                                                                    insert(playlistEntity)
                                                                    songs.map(SongItem::toMediaMetadata)
                                                                        .onEach(::insert)
                                                                        .mapIndexed { index, song ->
                                                                            PlaylistSongMap(
                                                                                songId = song.id,
                                                                                playlistId = playlistEntity.id,
                                                                                position = index
                                                                            )
                                                                        }
                                                                        .forEach(::insert)
                                                                }
                                                            } else {
                                                                database.transaction {
                                                                    update(dbPlaylist!!.playlist.toggleLike())
                                                                }
                                                            }
                                                        }
                                                    ) {
                                                        Icon(
                                                            painter = painterResource(
                                                                if (dbPlaylist?.playlist?.bookmarkedAt != null) R.drawable.favorite else R.drawable.favorite_border
                                                            ),
                                                            contentDescription = null,
                                                            tint = if (dbPlaylist?.playlist?.bookmarkedAt != null) MaterialTheme.colorScheme.error else LocalContentColor.current
                                                        )
                                                    }
                                                }

                                                IconButton(
                                                    onClick = {
                                                        menuState.show {
                                                            YouTubePlaylistMenu(
                                                                playlist = playlist,
                                                                songs = songs,
                                                                coroutineScope = coroutineScope,
                                                                onDismiss = menuState::dismiss,
                                                                selectAction = { selection = true },
                                                                canSelect = true,
                                                            )
                                                        }
                                                    },
                                                ) {
                                                    Icon(
                                                        painter = painterResource(R.drawable.more_vert),
                                                        contentDescription = null,
                                                    )
                                                }
                                            }
                                        }
                                    }

                                    Spacer(Modifier.height(12.dp))

                                    Row(horizontalArrangement = Arrangement.spacedBy(12.dp)) {
                                        playlist.shuffleEndpoint?.let { shuffleEndpoint ->
                                            Button(
                                                onClick = {
                                                    playerConnection.playQueue(
                                                        YouTubeQueue(
                                                            shuffleEndpoint
                                                        )
                                                    )
                                                },
                                                contentPadding = ButtonDefaults.ButtonWithIconContentPadding,
                                                modifier = Modifier.weight(1f)
                                            ) {
                                                Icon(
                                                    painter = painterResource(R.drawable.shuffle),
                                                    contentDescription = null,
                                                    modifier = Modifier.size(ButtonDefaults.IconSize)
                                                )
                                                Spacer(Modifier.size(ButtonDefaults.IconSpacing))
                                                Text(stringResource(R.string.shuffle))
                                            }
                                        }

                                        playlist.radioEndpoint?.let { radioEndpoint ->
                                            OutlinedButton(
                                                onClick = {
                                                    playerConnection.playQueue(
                                                        YouTubeQueue(
                                                            radioEndpoint
                                                        )
                                                    )
                                                },
                                                contentPadding = ButtonDefaults.ButtonWithIconContentPadding,
                                                modifier = Modifier.weight(1f),
                                            ) {
                                                Icon(
                                                    painter = painterResource(R.drawable.radio),
                                                    contentDescription = null,
                                                    modifier = Modifier.size(ButtonDefaults.IconSize),
                                                )
                                                Spacer(Modifier.size(ButtonDefaults.IconSpacing))
                                                Text(stringResource(R.string.radio))
                                            }
                                        }
                                    }
                                }
                            }
                        }

                        items(
                            items = wrappedSongs,
                        ) { song ->
                            YouTubeListItem(
                                item = song.item.second,
                                isActive = mediaMetadata?.id == song.item.second.id,
                                isPlaying = isPlaying,
                                isSelected = song.isSelected && selection,
                                trailingContent = {
                                    IconButton(
                                        onClick = {
                                            menuState.show {
                                                YouTubeSongMenu(
                                                    song = song.item.second,
                                                    navController = navController,
                                                    onDismiss = menuState::dismiss,
                                                )
                                            }
                                        },
                                    ) {
                                        Icon(
                                            painter = painterResource(R.drawable.more_vert),
                                            contentDescription = null,
                                        )
                                    }
                                },
                                modifier =
                                    Modifier
                                        .combinedClickable(
                                            enabled = !hideExplicit || !song.item.second.explicit,
                                            onClick = {
                                                if (!selection) {
                                                    if (song.item.second.id == mediaMetadata?.id) {
                                                        playerConnection.player.togglePlayPause()
                                                    } else {
                                                        playerConnection.service.getAutomix(
                                                            playlistId = playlist.id
                                                        )
                                                        playerConnection.playQueue(
                                                            YouTubeQueue(
                                                                song.item.second.endpoint
                                                                    ?: WatchEndpoint(
                                                                        videoId =
                                                                            song.item.second
                                                                                .id,
                                                                    ),
                                                                song.item.second.toMediaMetadata(),
                                                            ),
                                                        )
                                                    }
                                                } else {
                                                    song.isSelected = !song.isSelected
                                                }
                                            },
                                            onLongClick = {
                                                haptic.performHapticFeedback(HapticFeedbackType.LongPress)
                                                if (!selection) {
                                                    selection = true
                                                }
                                                wrappedSongs.forEach { it.isSelected = false }
                                                song.isSelected = true
                                            },
                                        )
                                        .alpha(if (hideExplicit && song.item.second.explicit) 0.3f else 1f)
                                        .animateItem(),
                            )
                        }
                    } else {
                        item {
                            ShimmerHost {
                                Column(Modifier.padding(12.dp)) {
                                    Row(verticalAlignment = Alignment.CenterVertically) {
                                        Spacer(
                                            modifier =
                                                Modifier
                                                    .size(AlbumThumbnailSize)
                                                    .clip(RoundedCornerShape(ThumbnailCornerRadius))
                                                    .background(MaterialTheme.colorScheme.onSurface),
                                        )

                                        Spacer(Modifier.width(16.dp))

                                        Column(
                                            verticalArrangement = Arrangement.Center,
                                        ) {
                                            TextPlaceholder()
                                            TextPlaceholder()
                                            TextPlaceholder()
                                        }
                                    }

                                    Spacer(Modifier.padding(8.dp))

                                    Row {
                                        ButtonPlaceholder(Modifier.weight(1f))

                                        Spacer(Modifier.width(12.dp))

                                        ButtonPlaceholder(Modifier.weight(1f))
                                    }
                                }

                                repeat(6) {
                                    ListItemPlaceHolder()
                                }
                            }
                        }
                    }
                }
            }
        }

        TopAppBar(
            title = {
                if (selection) {
                    val count = wrappedSongs.count { it.isSelected }
                    Text(
                        text = pluralStringResource(R.plurals.n_song, count, count),
                        style = MaterialTheme.typography.titleLarge
                    )
                } else if (isSearching) {
                    TextField(
                        value = query,
                        onValueChange = { query = it },
                        placeholder = {
                            Text(
                                text = stringResource(R.string.search),
                                style = MaterialTheme.typography.titleLarge
                            )
                        },
                        singleLine = true,
                        textStyle = MaterialTheme.typography.titleLarge,
                        keyboardOptions = KeyboardOptions(imeAction = ImeAction.Search),
                        colors = TextFieldDefaults.colors(
                            focusedContainerColor = Color.Transparent,
                            unfocusedContainerColor = Color.Transparent,
                            focusedIndicatorColor = Color.Transparent,
                            unfocusedIndicatorColor = Color.Transparent,
                            disabledIndicatorColor = Color.Transparent,
                        ),
                        modifier = Modifier
                            .fillMaxWidth()
                            .focusRequester(focusRequester)
                    )
                } else if (showTopBarTitle) {
                    Text(playlist?.title.orEmpty())
                }
            },
            navigationIcon = {
                IconButton(
                    onClick = {
                        if (isSearching) {
                            isSearching = false
                            query = TextFieldValue()
                        } else if (selection) {
                            selection = false
                        } else {
                            navController.navigateUp()
                        }
                    },
                ) {
                    Icon(
                        painter = painterResource(
                            if (selection) R.drawable.close else R.drawable.arrow_back
                        ),
                        contentDescription = null
                    )
                }
            },
            actions = {
                if (selection) {
                    val count = wrappedSongs.count { it.isSelected }
                    IconButton(
                        onClick = {
                            if (count == wrappedSongs.size) {
                                wrappedSongs.forEach { it.isSelected = false }
                            } else {
                                wrappedSongs.forEach { it.isSelected = true }
                            }
                        },
                    ) {
                        Icon(
                            painter = painterResource(
                                if (count == wrappedSongs.size) R.drawable.deselect else R.drawable.select_all
                            ),
                            contentDescription = null
                        )
                    }
                    IconButton(
                        onClick = {
                            menuState.show {
                                SelectionMediaMetadataMenu(
                                    songSelection = wrappedSongs.filter { it.isSelected }
                                        .map { it.item.second.toMediaItem().metadata!! },
                                    onDismiss = menuState::dismiss,
                                    clearAction = { selection = false },
                                    currentItems = emptyList()
                                )
                            }
                        },
                    ) {
                        Icon(
                            painter = painterResource(R.drawable.more_vert),
                            contentDescription = null
                        )
                    }
                } else {
                    if (!isSearching) {
                        IconButton(
                            onClick = { isSearching = true }
                        ) {
                            Icon(
                                painter = painterResource(R.drawable.search),
                                contentDescription = null
                            )
                        }
                    }
                }
            }
        )

        SnackbarHost(
            hostState = snackbarHostState,
            modifier =
                Modifier
                    .windowInsetsPadding(LocalPlayerAwareWindowInsets.current.union(WindowInsets.ime))
                    .align(Alignment.BottomCenter),
        )
    }
}