/*
 * Copyright 2023, 2024 New Vector Ltd.
 *
 * SPDX-License-Identifier: AGPL-3.0-only OR LicenseRef-Element-Commercial
 * Please see LICENSE files in the repository root for full details.
 */

package io.element.android.libraries.pushproviders.unifiedpush

import chat.schildi.lib.preferences.ScAppStateStore
import dev.zacsweers.metro.AppScope
import dev.zacsweers.metro.ContributesBinding
import dev.zacsweers.metro.Inject
import io.element.android.libraries.core.extensions.flatMap
import io.element.android.libraries.core.log.logger.LoggerTag
import io.element.android.libraries.matrix.api.MatrixClientProvider
import io.element.android.libraries.pushproviders.api.PusherSubscriber
import io.element.android.libraries.pushstore.api.UserPushStoreFactory
import io.element.android.libraries.pushstore.api.clientsecret.PushClientSecret
import timber.log.Timber

private val loggerTag = LoggerTag("DefaultUnifiedPushNewGatewayHandler", LoggerTag.PushLoggerTag)

/**
 * Handle new endpoint received from UnifiedPush. Will update the session matching the client secret.
 */
interface UnifiedPushNewGatewayHandler {
    suspend fun handle(endpoint: String, pushGateway: String, clientSecret: String): Result<Unit>
}

@ContributesBinding(AppScope::class)
@Inject
class DefaultUnifiedPushNewGatewayHandler(
    private val pusherSubscriber: PusherSubscriber,
    private val userPushStoreFactory: UserPushStoreFactory,
    private val pushClientSecret: PushClientSecret,
    private val scAppStateStore: ScAppStateStore,
    private val matrixClientProvider: MatrixClientProvider,
) : UnifiedPushNewGatewayHandler {
    override suspend fun handle(endpoint: String, pushGateway: String, clientSecret: String): Result<Unit> {
        // Register the pusher for the session with this client secret, if is it using UnifiedPush.
        val userId = pushClientSecret.getUserIdFromSecret(clientSecret) ?: return Result.failure<Unit>(
            IllegalStateException("Unable to retrieve session")
        ).also {
            Timber.w("Unable to retrieve session")
        }
        val userDataStore = userPushStoreFactory.getOrCreate(userId)
        scAppStateStore.setPushGateway(pushGateway)
        return if (userDataStore.getPushProviderName() == UnifiedPushConfig.NAME) {
            matrixClientProvider
                .getOrRestore(userId)
                .flatMap { client ->
                    pusherSubscriber.registerPusher(client, endpoint, pushGateway)
                }
        } else {
            Timber.tag(loggerTag.value).d("This session is not using UnifiedPush pusher")
            Result.failure(
                IllegalStateException("This session is not using UnifiedPush pusher")
            )
        }
    }
}
