use super::*;
use crate::chat;
use crate::chat::ChatId;
use crate::config::Config;
use crate::constants;
use crate::contact::Contact;
use crate::contact::ContactId;
use crate::contact::Origin;
use crate::test_utils::TestContext;
use crate::tools;

#[tokio::test(flavor = "multi_thread", worker_threads = 2)]
async fn test_clear_config_cache() -> anyhow::Result<()> {
    // Some migrations change the `config` table in SQL.
    // This test checks that the config cache is invalidated in `execute_migration()`.

    let t = TestContext::new().await;
    assert_eq!(t.get_config_bool(Config::IsChatmail).await?, false);

    t.sql
        .execute_migration(
            "INSERT INTO config (keyname, value) VALUES ('is_chatmail', '1')",
            1000,
        )
        .await?;
    assert_eq!(t.get_config_bool(Config::IsChatmail).await?, true);
    assert_eq!(t.sql.get_raw_config_int(VERSION_CFG).await?.unwrap(), 1000);

    Ok(())
}

#[tokio::test(flavor = "multi_thread", worker_threads = 2)]
async fn test_key_contacts_migration_autocrypt() -> Result<()> {
    let t = STOP_MIGRATIONS_AT
        .scope(131, async move { TestContext::new_alice().await })
        .await;

    t.sql.call_write(|conn| Ok(conn.execute_batch(r#"
        INSERT INTO acpeerstates VALUES(1,'bob@example.net',0,0,NULL,1,1745589039,X'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',NULL,'CCCB5AA9F6E1141C943165F1DB18B18CBCF70487',NULL,NULL,'',NULL,NULL,'',NULL);
        INSERT INTO contacts VALUES(10,'','bob@example.net',16384,0,0,'','',1745589041,'',0);
        INSERT INTO chats VALUES(10,100,'bob@example.net',0,'',0,'','',0,0,0,0,0,1745589039,0,NULL,0);
        INSERT INTO chats_contacts VALUES(10,10,0,0);"#,
    )?)).await?;
    t.sql.run_migrations(&t).await?;

    //std::thread::sleep(std::time::Duration::from_secs(1000));
    let pgp_bob_id = Contact::lookup_id_by_addr(&t, "bob@example.net", Origin::Hidden)
        .await?
        .unwrap();
    let pgp_bob = Contact::get_by_id(&t, pgp_bob_id).await?;
    assert_eq!(pgp_bob.is_key_contact(), true);
    assert_eq!(pgp_bob.origin, Origin::OutgoingTo);
    assert_eq!(pgp_bob.e2ee_avail(&t).await?, true);
    assert_eq!(
        pgp_bob.fingerprint().unwrap(),
        pgp_bob.public_key(&t).await?.unwrap().dc_fingerprint()
    );
    assert_eq!(pgp_bob.get_verifier_id(&t).await?, None);

    // Hidden address-contact can't be looked up.
    assert!(
        Contact::get_all(&t, constants::DC_GCL_ADDRESS, Some("bob@example.net"))
            .await?
            .is_empty()
    );

    let bob_chat_contacts = chat::get_chat_contacts(&t, ChatId::new(10)).await?;
    assert_eq!(tools::single_value(bob_chat_contacts).unwrap(), pgp_bob_id);

    Ok(())
}

#[tokio::test(flavor = "multi_thread", worker_threads = 2)]
async fn test_key_contacts_migration_email1() -> Result<()> {
    let t = STOP_MIGRATIONS_AT
        .scope(131, async move { TestContext::new_alice().await })
        .await;

    t.sql.call_write(|conn| Ok(conn.execute_batch(r#"
        INSERT INTO acpeerstates VALUES(1,'bob@example.net',0,0,NULL,1,1745589039,X'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',NULL,'CCCB5AA9F6E1141C943165F1DB18B18CBCF70487',NULL,NULL,'',NULL,NULL,'',NULL);
        INSERT INTO contacts VALUES(10,'','bob@example.net',16384,0,0,'','',1745589041,'',0);
        INSERT INTO chats VALUES(10,120,'Group',0,'',0,'','g=1745609548',0,0,0,0,0,1745609547,0,NULL,1);
        INSERT INTO chats_contacts VALUES(10,1,1745609547,0);
        INSERT INTO chats_contacts VALUES(10,10,1745609547,0);"#,
    )?)).await?;
    t.sql.run_migrations(&t).await?;

    let email_bob_id = *Contact::get_all(&t, constants::DC_GCL_ADDRESS, Some("bob@example.net"))
        .await?
        .first()
        .unwrap();
    let email_bob = Contact::get_by_id(&t, email_bob_id).await?;
    assert_eq!(email_bob.is_key_contact(), false);
    assert_eq!(email_bob.origin, Origin::OutgoingTo);
    assert_eq!(email_bob.e2ee_avail(&t).await?, false);
    assert_eq!(email_bob.fingerprint(), None);
    assert_eq!(email_bob.get_verifier_id(&t).await?, None);

    let bob_chat_contacts = chat::get_chat_contacts(&t, ChatId::new(10)).await?;
    dbg!(&bob_chat_contacts);

    Ok(())
}

#[tokio::test(flavor = "multi_thread", worker_threads = 2)]
async fn test_key_contacts_migration_email2() -> Result<()> {
    let t = STOP_MIGRATIONS_AT
        .scope(131, async move { TestContext::new_alice().await })
        .await;

    t.sql.call_write(|conn| Ok(conn.execute_batch(r#"
        INSERT INTO acpeerstates VALUES(1,'bob@example.net',0,0,NULL,20,1745589039,X'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',NULL,'CCCB5AA9F6E1141C943165F1DB18B18CBCF70487',NULL,NULL,'',NULL,NULL,'',NULL);
        INSERT INTO contacts VALUES(10,'','bob@example.net',16384,0,0,'','',1745589041,'',0);
        INSERT INTO chats VALUES(10,100,'bob@example.net',0,'',0,'','',0,0,0,0,0,1745589039,0,NULL,0);
        INSERT INTO chats_contacts VALUES(10,10,0,0);"#,
    )?)).await?;
    t.sql.run_migrations(&t).await?;

    // Hidden key-contact can't be looked up.
    assert!(
        Contact::get_all(&t, 0, Some("bob@example.net"))
            .await?
            .is_empty()
    );
    let pgp_bob = Contact::get_by_id(&t, ContactId::new(11001)).await?;
    assert_eq!(pgp_bob.is_key_contact(), true);
    assert_eq!(pgp_bob.origin, Origin::Hidden);

    let email_bob_id = *Contact::get_all(&t, constants::DC_GCL_ADDRESS, Some("bob@example.net"))
        .await?
        .first()
        .unwrap();
    let email_bob = Contact::get_by_id(&t, email_bob_id).await?;
    assert_eq!(email_bob.is_key_contact(), false);
    assert_eq!(email_bob.origin, Origin::OutgoingTo);
    assert_eq!(email_bob.e2ee_avail(&t).await?, false);
    assert_eq!(email_bob.fingerprint(), None);
    assert_eq!(email_bob.get_verifier_id(&t).await?, None);

    let bob_chat_contacts = chat::get_chat_contacts(&t, ChatId::new(10)).await?;
    dbg!(&bob_chat_contacts);

    Ok(())
}

#[tokio::test(flavor = "multi_thread", worker_threads = 2)]
async fn test_key_contacts_migration_verified() -> Result<()> {
    let t = STOP_MIGRATIONS_AT
        .scope(131, async move { TestContext::new_alice().await })
        .await;

    t.sql.call_write(|conn| Ok(conn.execute_batch(r#"
        INSERT INTO acpeerstates VALUES(1,'bob@example.net',0,0,X'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',1,1745609547,X'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','CCCB5AA9F6E1141C943165F1DB18B18CBCF70487','CCCB5AA9F6E1141C943165F1DB18B18CBCF70487',X'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','CCCB5AA9F6E1141C943165F1DB18B18CBCF70487','',NULL,NULL,'',1);
        INSERT INTO contacts VALUES(10,'','bob@example.net',16384,0,0,'','',1745609549,'',0);
        INSERT INTO msgs VALUES(10,'29b4af31-1560-4bc8-9b2b-083f2a3d0432@localhost','',0,10,2,2,1745609547,10,13,1,0,'Messages are end-to-end encrypted.','','S=11',0,0,0,0,NULL,'',NULL,1,0,'',0,0,0,'',0,NULL,0,NULL,0);
        INSERT INTO msgs VALUES(11,'411b3fdd-a20c-48c7-b94d-19c04654a1c5@localhost','',0,10,1,0,1745609548,10,26,1,0,'Hello!','',replace('A=1\nc=1','\n',char(10)),0,0,0,0,X'','','411b3fdd-a20c-48c7-b94d-19c04654a1c5@localhost',1,0,'',0,0,0,'Group',0,NULL,1,NULL,0);
        INSERT INTO chats VALUES(10,120,'Group',0,'',0,'-PYdPTYhrEl9L_C6osfpEpQu','g=1745609548',0,0,0,0,0,1745609547,0,NULL,1);
        INSERT INTO chats_contacts VALUES(10,1,1745609547,0);
        INSERT INTO chats_contacts VALUES(10,10,1745609547,0);
    "#,
    )?)).await?;

    t.sql.run_migrations(&t).await?;

    // Hidden address-contact can't be looked up.
    assert!(
        Contact::get_all(&t, constants::DC_GCL_ADDRESS, Some("bob@example.net"))
            .await?
            .is_empty()
    );

    let mut bob_chat_contacts = chat::get_chat_contacts(&t, ChatId::new(10)).await?;
    assert_eq!(bob_chat_contacts.len(), 2);
    bob_chat_contacts.retain(|c| *c != ContactId::SELF);
    assert_eq!(bob_chat_contacts.len(), 1);
    let pgp_bob_id = bob_chat_contacts[0];

    let pgp_bob = Contact::get_by_id(&t, pgp_bob_id).await?;
    dbg!(&pgp_bob);
    assert_eq!(pgp_bob.origin, Origin::OutgoingTo);
    assert_eq!(pgp_bob.e2ee_avail(&t).await?, true);
    assert_eq!(
        pgp_bob.fingerprint().unwrap(),
        pgp_bob.public_key(&t).await?.unwrap().dc_fingerprint()
    );
    assert_eq!(pgp_bob.get_verifier_id(&t).await?, Some(None));

    Ok(())
}
