/*  _____ _
 * |_   _| |_  _ _ ___ ___ _ __  __ _
 *   | | | ' \| '_/ -_) -_) '  \/ _` |_
 *   |_| |_||_|_| \___\___|_|_|_\__,_(_)
 *
 * Threema for Android
 * Copyright (c) 2013-2025 Threema GmbH
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License, version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package ch.threema.app.preference;

import android.app.Activity;
import android.app.KeyguardManager;
import android.content.Context;
import android.content.Intent;
import android.os.AsyncTask;
import android.os.Bundle;
import android.provider.Settings;
import android.text.InputType;
import android.view.View;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.biometric.BiometricPrompt;
import androidx.fragment.app.DialogFragment;
import androidx.preference.DropDownPreference;
import androidx.preference.Preference;
import androidx.preference.TwoStatePreference;

import com.google.android.material.snackbar.Snackbar;

import org.slf4j.Logger;

import java.security.MessageDigest;
import java.util.Arrays;

import ch.threema.app.AppConstants;
import ch.threema.app.R;
import ch.threema.app.ThreemaApplication;
import ch.threema.app.activities.ThreemaActivity;
import ch.threema.app.passphrase.PassphraseUnlockActivity;
import ch.threema.app.dialogs.GenericAlertDialog;
import ch.threema.app.dialogs.GenericProgressDialog;
import ch.threema.app.dialogs.PasswordEntryDialog;
import ch.threema.app.dialogs.ThreemaDialogFragment;
import ch.threema.app.preference.service.PreferenceService;
import ch.threema.app.services.ConversationCategoryService;
import ch.threema.app.utils.BiometricUtil;
import ch.threema.app.utils.DialogUtil;
import ch.threema.app.utils.HiddenChatUtil;
import ch.threema.app.utils.RuntimeUtil;
import ch.threema.app.utils.Toaster;
import ch.threema.base.utils.LoggingUtil;

import static ch.threema.app.di.DIJavaCompat.getMasterKeyManager;
import static ch.threema.app.preference.service.PreferenceService.LockingMech_NONE;
import static ch.threema.app.utils.ActiveScreenLoggerKt.logScreenVisibility;

public class SettingsSecurityFragment extends ThreemaPreferenceFragment implements PasswordEntryDialog.PasswordEntryDialogClickListener, GenericAlertDialog.DialogClickListener {
    private static final Logger logger = LoggingUtil.getThreemaLogger("SettingsSecurityFragment");

    private Preference pinPreference;
    private TwoStatePreference uiLockSwitchPreference;
    private DropDownPreference gracePreference, lockMechanismPreference;
    private Preference masterkeyPreference;
    private TwoStatePreference masterkeySwitchPreference;
    private PreferenceService preferenceService;
    private ConversationCategoryService conversationCategoryService;

    private View fragmentView;

    private static final String ID_DIALOG_PASSPHRASE = "mkpw";
    private static final String ID_DIALOG_PROGRESS = "pogress";
    private static final String ID_DIALOG_PIN = "dpin";
    private static final String ID_UNHIDE_CHATS_CONFIRM = "uh";
    private static final String DIALOG_TAG_PASSWORD_REMINDER_PIN = "emin";
    private static final String DIALOG_TAG_PASSWORD_REMINDER_PASSPHRASE = "eminpass";

    private static final int ID_ENABLE_SYSTEM_LOCK = 7780;
    private static final int ACTIVITY_ID_CHANGE_PASSPHRASE_UNLOCK = 20032;
    private static final int ACTIVITY_ID_SET_PASSPHRASE = 20013;
    private static final int ACTIVITY_ID_CHANGE_PASSPHRASE = 20014;
    private static final int ACTIVITY_ID_RESET_PASSPHRASE = 20015;

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        logScreenVisibility(this, logger);
    }

    @Override
    protected void initializePreferences() {
        logger.debug("### initializePreferences");

        super.initializePreferences();

        preferenceService = requirePreferenceService();
        conversationCategoryService = requireConversationCategoryService();
    }

    private void onCreateUnlocked() {
        logger.debug("### onCreateUnlocked");
        fragmentView.setVisibility(View.VISIBLE);

        uiLockSwitchPreference = findPreference(getResources().getString(R.string.preferences__lock_ui_switch));
        lockMechanismPreference = findPreference(getResources().getString(R.string.preferences__lock_mechanism));
        pinPreference = findPreference(getResources().getString(R.string.preferences__pin_lock_code));
        gracePreference = findPreference(getResources().getString(R.string.preferences__pin_lock_grace_time));

        //get pin switch pref from service!
        uiLockSwitchPreference.setChecked(preferenceService.isAppLockEnabled());

        CharSequence[] entries = lockMechanismPreference.getEntries();
        if (preferenceService.getLockMechanism().equals(PreferenceService.LockingMech_BIOMETRIC)) {
            if (!BiometricUtil.isHardwareSupported(getContext())) {
                preferenceService.setLockMechanism(LockingMech_NONE);
                // remove biometric option
                lockMechanismPreference.setEntries(Arrays.copyOf(entries, 3));
            }
        } else {
            if (!BiometricUtil.isHardwareSupported(getContext())) {
                // remove biometric option
                lockMechanismPreference.setEntries(Arrays.copyOf(entries, 3));
            }
        }

        if (preferenceService.isPinSet()) {
            pinPreference.setSummary(getString(R.string.click_here_to_change_pin));
        }

        switch (preferenceService.getLockMechanism()) {
            case PreferenceService.LockingMech_PIN:
                lockMechanismPreference.setValueIndex(1);
                break;
            case PreferenceService.LockingMech_SYSTEM:
                lockMechanismPreference.setValueIndex(2);
                break;
            case PreferenceService.LockingMech_BIOMETRIC:
                lockMechanismPreference.setValueIndex(3);
                break;
            default:
                lockMechanismPreference.setValueIndex(0);
        }

        updateLockPreferences();

        setGraceTime();

        lockMechanismPreference.setOnPreferenceChangeListener(new Preference.OnPreferenceChangeListener() {
            @Override
            public boolean onPreferenceChange(@NonNull Preference preference, Object newValue) {
                switch ((String) newValue) {
                    case LockingMech_NONE:
                        if (conversationCategoryService.hasPrivateChats()) {
                            GenericAlertDialog dialog = GenericAlertDialog.newInstance(
                                R.string.hide_chat,
                                R.string.access_protection_cannot_be_removed,
                                R.string.cancel,
                                0
                            );
                            dialog.setTargetFragment(SettingsSecurityFragment.this, 0);
                            dialog.show(getFragmentManager(), ID_UNHIDE_CHATS_CONFIRM);
                        } else {
                            removeAccessProtection();
                        }
                        break;
                    case PreferenceService.LockingMech_PIN:
                        GenericAlertDialog dialog = GenericAlertDialog.newInstance(R.string.warning, getString(R.string.password_remember_warning, getString(R.string.app_name)), R.string.ok, R.string.cancel);
                        dialog.setTargetFragment(SettingsSecurityFragment.this, 0);
                        dialog.show(getParentFragmentManager(), DIALOG_TAG_PASSWORD_REMINDER_PIN);
                        break;
                    case PreferenceService.LockingMech_SYSTEM:
                        setSystemScreenLock();
                        break;
                    case PreferenceService.LockingMech_BIOMETRIC:
                        setBiometricLock();
                }

                updateLockPreferences();
                return false;
            }
        });

        uiLockSwitchPreference.setOnPreferenceChangeListener(new Preference.OnPreferenceChangeListener() {
            public boolean onPreferenceChange(@NonNull Preference preference, Object newValue) {
                boolean newCheckedValue = newValue.equals(true);

                if (((TwoStatePreference) preference).isChecked() != newCheckedValue) {
                    preferenceService.setAppLockEnabled(false);

                    if (newCheckedValue) {
                        if (lockMechanismPreference.getValue() == null || PreferenceService.LockingMech_NONE.equals(lockMechanismPreference.getValue())) {
                            return false;
                        }

                        setGraceTime();
                        preferenceService.setAppLockEnabled(true);
                    }
                }
                return true;
            }
        });

        pinPreference.setOnPreferenceClickListener(new Preference.OnPreferenceClickListener() {
            @Override
            public boolean onPreferenceClick(@NonNull Preference preference) {
                if (preference.getKey().equals(getResources().getString(R.string.preferences__pin_lock_code))) {
                    if (preferenceService.isPinSet()) {
                        setPin();
                    }
                }
                return false;
            }
        });

        this.updateGracePreferenceSummary(gracePreference.getValue());
        gracePreference.setOnPreferenceChangeListener(new Preference.OnPreferenceChangeListener() {
            public boolean onPreferenceChange(@NonNull Preference preference, Object newValue) {
                updateGracePreferenceSummary(newValue.toString());
                return true;
            }
        });

        masterkeyPreference = getPref(getResources().getString(R.string.preferences__masterkey_passphrase));
        masterkeyPreference.setOnPreferenceClickListener(new Preference.OnPreferenceClickListener() {
            @Override
            public boolean onPreferenceClick(@NonNull Preference preference) {
                if (MessageDigest.isEqual(preference.getKey().getBytes(), getResources().getString(R.string.preferences__masterkey_passphrase).getBytes())) {
                    Intent intent = PassphraseUnlockActivity.createIntent(requireContext(), true, true);
                    startActivityForResult(intent, ACTIVITY_ID_CHANGE_PASSPHRASE_UNLOCK);

                }
                return false;
            }
        });

        masterkeySwitchPreference = findPreference(getResources().getString(R.string.preferences__masterkey_switch));
        masterkeySwitchPreference.setChecked(getMasterKeyManager().isProtectedWithPassphrase());

        setMasterKeyPreferenceText();

        masterkeySwitchPreference.setOnPreferenceChangeListener(new Preference.OnPreferenceChangeListener() {
            public boolean onPreferenceChange(@NonNull Preference preference, Object newValue) {
                boolean newCheckedValue = newValue.equals(true);

                if (((TwoStatePreference) preference).isChecked() != newCheckedValue) {
                    if (!newCheckedValue) {
                        Intent intent = PassphraseUnlockActivity.createIntent(requireContext(), true, true);
                        startActivityForResult(intent, ACTIVITY_ID_RESET_PASSPHRASE);

                        setMasterKeyPreferenceText();
                    } else {
                        startSetPassphraseActivity();
                        return false;
                    }
                }
                return true;
            }
        });

        logger.debug("### onCreateUnlocked end");
    }

    private void updateGracePreferenceSummary(String value) {
        String[] existingValues = getResources().getStringArray(R.array.list_pin_grace_time_values);
        for (int n = 0; n < existingValues.length; n++) {
            if (existingValues[n].equals(value)) {
                gracePreference.setSummary(getResources().getStringArray(R.array.list_pin_grace_time)[n]);
            }
        }
    }

    private void removeAccessProtection() {
        lockMechanismPreference.setValue(LockingMech_NONE);
        preferenceService.setPrivateChatsHidden(false);
    }

    private void updateLockPreferences() {
        pinPreference.setSummary(preferenceService.isPinSet() ? getString(R.string.click_here_to_change_pin) : getString(R.string.prefs_title_pin_code));
        lockMechanismPreference.setSummary(lockMechanismPreference.getEntry());

        switch (lockMechanismPreference.getValue()) {
            case LockingMech_NONE:
                pinPreference.setEnabled(false);
                gracePreference.setEnabled(false);
                uiLockSwitchPreference.setChecked(false);
                uiLockSwitchPreference.setEnabled(false);
                preferenceService.setPin(null);
                preferenceService.setAppLockEnabled(false);
                break;
            case PreferenceService.LockingMech_PIN:
                pinPreference.setEnabled(true);
                gracePreference.setEnabled(true);
                uiLockSwitchPreference.setEnabled(true);
                break;
            case PreferenceService.LockingMech_SYSTEM:
            case PreferenceService.LockingMech_BIOMETRIC:
                pinPreference.setEnabled(false);
                gracePreference.setEnabled(true);
                uiLockSwitchPreference.setEnabled(true);
                preferenceService.setPin(null);
                break;
        }
    }

    @Override
    public void onViewCreated(@NonNull View view, Bundle savedInstanceState) {
        logger.debug("### onViewCreated");

        // we make the complete fragment invisible until it's been unlocked
        fragmentView = view;
        fragmentView.setVisibility(View.INVISIBLE);

        super.onViewCreated(view, savedInstanceState);

        // ask for pin before entering
        if (preferenceService.getLockMechanism().equals(LockingMech_NONE)) {
            onCreateUnlocked();
        } else {
            if ((preferenceService.getLockMechanism().equals(PreferenceService.LockingMech_PIN)) && !preferenceService.isPinSet()) {
                // fix misconfiguration
                preferenceService.setLockMechanism(LockingMech_NONE);
                onCreateUnlocked();
            } else {
                if (savedInstanceState == null) {
                    HiddenChatUtil.launchLockCheckDialog(this, preferenceService);
                }
            }
        }
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        if (resultCode == Activity.RESULT_OK) {
            switch (requestCode) {
                case ThreemaActivity.ACTIVITY_ID_CHECK_LOCK:
                    requireScreenLockService().setAuthenticated(true);
                    onCreateUnlocked();
                    break;
                case ID_ENABLE_SYSTEM_LOCK:
                    lockMechanismPreference.setValue(PreferenceService.LockingMech_SYSTEM);
                    if (uiLockSwitchPreference.isChecked()) {
                        preferenceService.setAppLockEnabled(true);
                    }
                    updateLockPreferences();
                    break;
                case ACTIVITY_ID_RESET_PASSPHRASE:
                    var passphrase = PassphraseUnlockActivity.getPassphrase(data);
                    if (passphrase != null) {
                        RuntimeUtil.runOnWorkerThread(() -> {
                            try {
                                getMasterKeyManager().removePassphrase(passphrase);
                            } catch (Exception e) {
                                logger.error("Failed to remove passphrase", e);
                                new Toaster(requireContext()).showToast(R.string.an_error_occurred);
                            }
                        });
                    } else {
                        logger.error("Passphrase not received, cannot remove");
                    }
                    break;
                case ACTIVITY_ID_CHANGE_PASSPHRASE_UNLOCK:
                    var oldPassphrase = PassphraseUnlockActivity.getPassphrase(data);
                    if (oldPassphrase != null) {
                        startChangePassphraseActivity(oldPassphrase);
                    } else {
                        logger.error("Passphrase not received, cannot change");
                    }
                    break;
                case ACTIVITY_ID_SET_PASSPHRASE:
                case ACTIVITY_ID_CHANGE_PASSPHRASE:
                    //do not handle event
                    setMasterKeyPreferenceText();
                    break;
                default:
                    super.onActivityResult(requestCode, resultCode, data);
            }
        } else {
            switch (requestCode) {
                case ThreemaActivity.ACTIVITY_ID_CHECK_LOCK:
                    requireScreenLockService().setAuthenticated(false);
                    requireActivity().onBackPressed();
                    break;
                case ACTIVITY_ID_SET_PASSPHRASE:
                    //only switch back on set
                    masterkeySwitchPreference.setChecked(false);
                    break;
                case ACTIVITY_ID_RESET_PASSPHRASE:
                    //only switch back on set
                    masterkeySwitchPreference.setChecked(true);
                    break;
                default:
                    super.onActivityResult(requestCode, resultCode, data);
            }
        }
    }

    private void setPin() {
        DialogFragment dialogFragment = PasswordEntryDialog.newInstance(
            R.string.set_pin_menu_title,
            R.string.set_pin_summary_intro,
            R.string.set_pin_hint,
            R.string.ok,
            R.string.cancel,
            AppConstants.MIN_PIN_LENGTH,
            AppConstants.MAX_PIN_LENGTH,
            R.string.set_pin_again_summary,
            InputType.TYPE_CLASS_NUMBER | InputType.TYPE_NUMBER_VARIATION_PASSWORD,
            0, PasswordEntryDialog.ForgotHintType.NONE);
        dialogFragment.setTargetFragment(this, 0);
        dialogFragment.show(getParentFragmentManager(), ID_DIALOG_PIN);
    }

    private void setSystemScreenLock() {
        KeyguardManager keyguardManager = (KeyguardManager) requireActivity().getSystemService(Context.KEYGUARD_SERVICE);
        if (keyguardManager.isDeviceSecure()) {
            BiometricUtil.showUnlockDialog(null, this, true, ID_ENABLE_SYSTEM_LOCK, PreferenceService.LockingMech_SYSTEM);
        } else {
            Snackbar snackbar = Snackbar.make(fragmentView, R.string.no_lockscreen_set, Snackbar.LENGTH_LONG);
            snackbar.setAction(R.string.configure, new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    startActivity(new Intent(Settings.ACTION_SECURITY_SETTINGS));
                }
            });
            snackbar.show();
        }
    }

    private void setBiometricLock() {
        if (BiometricUtil.isBiometricsSupported(requireContext())) {
            BiometricPrompt.PromptInfo promptInfo = new BiometricPrompt.PromptInfo.Builder()
                .setTitle(getString(R.string.prefs_title_access_protection))
                .setSubtitle(getString(R.string.biometric_enter_authentication))
                .setNegativeButtonText(getString(R.string.cancel))
                .build();
            BiometricPrompt biometricPrompt = new BiometricPrompt(requireActivity(), new RuntimeUtil.MainThreadExecutor(), new BiometricPrompt.AuthenticationCallback() {
                @Override
                public void onAuthenticationError(int errorCode, @NonNull CharSequence errString) {
                    if (errorCode != BiometricPrompt.ERROR_USER_CANCELED && errorCode != BiometricPrompt.ERROR_NEGATIVE_BUTTON) {
                        String text = errString + " (" + errorCode + ")";
                        try {
                            Snackbar.make(fragmentView, text, Snackbar.LENGTH_LONG).show();
                        } catch (IllegalArgumentException e) {
                            logger.error("Exception", e);
                            Toast.makeText(ThreemaApplication.getAppContext(), text, Toast.LENGTH_LONG).show();
                        }
                    }
                }

                @Override
                public void onAuthenticationSucceeded(@NonNull BiometricPrompt.AuthenticationResult result) {
                    super.onAuthenticationSucceeded(result);
                    Snackbar.make(fragmentView, R.string.biometric_authentication_successful, Snackbar.LENGTH_LONG).show();

                    lockMechanismPreference.setValue(PreferenceService.LockingMech_BIOMETRIC);
                    if (uiLockSwitchPreference.isChecked()) {
                        preferenceService.setLockMechanism(PreferenceService.LockingMech_BIOMETRIC);
                    }
                    updateLockPreferences();
                }
            });
            biometricPrompt.authenticate(promptInfo);
        }
    }

    private void setGraceTime() {
        String graceTime = gracePreference.getValue();
        if (Integer.parseInt(graceTime) >= 0 && Integer.parseInt(graceTime) < 30) {
            //set default grace time to never
            gracePreference.setValue("-1");
            updateGracePreferenceSummary(gracePreference.getValue());
        }
    }

    private void startSetPassphraseActivity() {
        GenericAlertDialog dialog = GenericAlertDialog.newInstance(R.string.warning, getString(R.string.password_remember_warning, getString(R.string.app_name)), R.string.ok, R.string.cancel);
        dialog.setTargetFragment(this, 0);
        dialog.show(getParentFragmentManager(), DIALOG_TAG_PASSWORD_REMINDER_PASSPHRASE);
    }

    private void startChangePassphraseActivity(@Nullable char[] oldPassphrase) {
        ThreemaDialogFragment dialogFragment = PasswordEntryDialog.newInstance(
            R.string.masterkey_passphrase_title,
            R.string.masterkey_passphrase_summary,
            R.string.masterkey_passphrase_hint,
            R.string.ok,
            R.string.cancel,
            8, 0,
            R.string.masterkey_passphrase_again_summary,
            0, 0, PasswordEntryDialog.ForgotHintType.NONE);
        dialogFragment.setTargetFragment(this, 0);
        dialogFragment.setData(oldPassphrase);
        dialogFragment.show(getParentFragmentManager(), ID_DIALOG_PASSPHRASE);
    }

    private void setMasterKeyPreferenceText() {
        masterkeyPreference.setSummary(
            getMasterKeyManager().isProtectedWithPassphrase()
                ? getString(R.string.click_here_to_change_passphrase)
                : getString(R.string.prefs_masterkey_passphrase)
        );
    }

    @Override
    public void onYes(String tag, final String text, final boolean isChecked, Object data) {
        switch (tag) {
            case ID_DIALOG_PIN:
                if (preferenceService.setPin(text)) {
                    lockMechanismPreference.setValue(PreferenceService.LockingMech_PIN);
                    if (uiLockSwitchPreference.isChecked()) {
                        preferenceService.setAppLockEnabled(true);
                    }
                    updateLockPreferences();
                } else {
                    Toast.makeText(getActivity(), getString(R.string.pin_invalid_not_set), Toast.LENGTH_SHORT).show();
                }
                break;
            case ID_DIALOG_PASSPHRASE:
                var oldPassphrase = data != null ? (char[]) data : null;
                new AsyncTask<Void, Void, Boolean>() {
                    @Override
                    protected void onPreExecute() {
                        GenericProgressDialog.newInstance(R.string.setting_masterkey_passphrase, R.string.please_wait)
                            .show(getParentFragmentManager(), ID_DIALOG_PROGRESS);
                    }

                    @Override
                    protected Boolean doInBackground(Void... voids) {
                        try {
                            int pl = text.length();
                            char[] passphrase = new char[pl];
                            text.getChars(0, pl, passphrase, 0);

                            getMasterKeyManager().setPassphrase(passphrase, oldPassphrase);
                            RuntimeUtil.runOnUiThread(new Runnable() {
                                @Override
                                public void run() {
                                    setMasterKeyPreferenceText();
                                }
                            });
                        } catch (Exception e) {
                            logger.error("Failed to set passphrase", e);
                            return false;
                        }
                        return true;
                    }

                    @Override
                    protected void onPostExecute(Boolean success) {
                        if (isAdded()) {
                            DialogUtil.dismissDialog(getFragmentManager(), ID_DIALOG_PROGRESS, true);
                        }
                        if (success) {
                            masterkeySwitchPreference.setChecked(true);
                        } else {
                            Toaster.Companion.showToast(R.string.an_error_occurred);
                        }
                    }
                }.execute();
                break;
        }
    }

    @Override
    public void onNo(String tag) {
        switch (tag) {
            case ID_DIALOG_PASSPHRASE:
                masterkeySwitchPreference.setChecked(getMasterKeyManager().isProtectedWithPassphrase());
                setMasterKeyPreferenceText();
                break;
            case ID_DIALOG_PIN:
                // workaround to reset dropdown state
                lockMechanismPreference.setEnabled(false);
                lockMechanismPreference.setEnabled(true);
                break;
            default:
                break;
        }
    }

    @Override
    public void onYes(String tag, Object data) {
        switch (tag) {
            case DIALOG_TAG_PASSWORD_REMINDER_PIN:
                setPin();
                break;
            case DIALOG_TAG_PASSWORD_REMINDER_PASSPHRASE:
                startChangePassphraseActivity(null);
                break;
            default:
                break;
        }
    }

    @Override
    public void onNo(String tag, Object data) {
        if (DIALOG_TAG_PASSWORD_REMINDER_PIN.equals(tag)) {
            // workaround to reset dropdown state
            lockMechanismPreference.setEnabled(false);
            lockMechanismPreference.setEnabled(true);
        }
    }

    @Override
    public int getPreferenceTitleResource() {
        return R.string.prefs_security;
    }

    @Override
    public int getPreferenceResource() {
        return R.xml.preference_security;
    }
}
