/* eslint-disable prettier/prettier */
/*
To find example transactions:
for h in `./bitcoin-cli getblock 00000000025a711e6cd4bce9138dc852232a4494afbf36d8bb80499a786da2a4|jq '.tx[]'`; do ./bitcoin-cli getrawtransaction `echo $h | sed 's/"//g'` 1; done > 00000000025a711e6cd4bce9138dc852232a4494afbf36d8bb80499a786da2a4
*/
export type CoreInput = {txid: string, vout: number, scriptSig: {hex: string, [x: string]: unknown}, txinwitness?: string[], sequence: number}
export type CoreOutput = {value: number, scriptPubKey: {hex: string, [x: string]: unknown}, [x: string]: unknown}
export type CoreTx = {txid: string, vin: CoreInput[], vout: CoreOutput[], hex: string, locktime: number, [x: string]: unknown}
export const spentTxs = {
  "5c02391a5b8f55ebb0db0e4e75900ace70f7e8e992115b3931379ab32b69da40": "020000000001016dac615a27b99c5e82af284cc01b410e789047709de08026c861bb6675fbf81800000000171600140fc5a02184f752c41e7f5da947f48646a213e32dfeffffff02ab5b990000000000160014080b627107f555d511730262f3c7167168e0db46a086010000000000220020de4cbf3bfa733a93fc7db03ef7aef2d3aae4e6c04c0ed3a9ebfe6aae259bb7e8024730440220740e00a9cde06677429018b6cadadbdbe7edd2bcd20f8255be2d83560de1490b02206be17eb882b02b25db2367d3afe8b67e6fb62fee2cd8244285403f16efc1886c012103ad62cd5bce8dab660f74c6bdcd4fc2695acdaa3687d8d6901c1cfb35b1211f3b87032000",
  "80453b180c6616431d28d395a6a89cb331d7ecb21bf3ac1cbb68c90dba6e95ff": "02000000015e50f6bfe2d91dc2927ba425fd1617cc2b68d2d7d2b70a3c795c15a9c86f43f1010000006a47304402203c9bf4759c850c43d7c25936c4163a3e6b869cc8a4cce107475fb0447442814c02203630d862ad274ba227ddd8d7ca85580139398b53101d4c4eec9d54712bc4b0420121037ed58c914720772c59f7a1e7e76fba0ef95d7c5667119798586301519b9ad2cfffffffff020000000000000000366a34696f6e3a322e516d66316f75486a46474b48674652716262794a4463454c617369414c595a344d5769797542715877674376695494eded07000000001976a9147628046f6ec479c031d03a47b6a003c4dedb1cad88ac00000000",
  "0d41be343ced2023ff112b123a7bb5c8f345becddb8f1cc7070e1273a21d3710": "0200000000010112fad068799db7637882b643557fb48fd5a7d733276a0e8e96cb44c28e3633230100000000fdffffff01b23f010000000000225120aaeed4185487b0fa6939914cf851cfbe84f5ef098bdc79408e8201b98339a0d80400473044022037ac3cbec286964f3fda9fdd4a73e510d76e829b267aafab392ee23b72d888f602200cc2f93f9251efd6ff1d511384fd0f7d0eb62924444f3806c5ac5e82977e546f0147304402203d5f1ef5cf53bce95e10731375b625dbaf4f6a344cb858d203b422aa421e59e902206fab1e6ee80420a46b5bafbf5ceecd06fd0971080b4c3d0c69d24f953f30ce160169522102bec882e74ae50d719b95087050feb223d9af025d381700c02eb27e8efc5aa50d21038be5e2af3e885abb51d4bd2dbe7bde16d24e1bc3fbe74781ff09645786c7cc262103b135afbeb5dd40aab1fd5e0b19aa537713af8507f01cbfc123a21a750e0efd8753aeb1f61f00",
  "5742c24f1485958d0e36d0ae758b1e33c01ba99107eb54baa539cc7db233a330": "02000000000101b7ff921fcf988d2f394b23f7a9e4648a11e64040d9cbe140d68e9f0fa63b52bd00000000171600142d62a39a40a30228c77de441c1f0993fe1707ddaffffffff02c38f934c0200000017a9145a102ac5376429003168cad88c80331daa3a747187102700000000000017a914c18c7cc194e62a004de3427199417f42d32cd81b870247304402205b615920b281070e99804ef9c5fe57e0595cb0c7968dd5d803cc505f7d28413102202b51ebf4ba70d7f987a98d101e1d60a6b40fba5bcb6ba9261eca9ed3777761cd012103efc6b990c1626d08bd176aab0e545a4f55c627c7ddee878d12bbbc46a126177a00000000",
  "9075ef8fd97f92ff0db344322873f12c42b29661c3960801e05114ba8adcabd6": "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",
  "5512d5788d4c26117f093de91223ef384c3fb22799810a92e3304bb6f0819224": "0200000000010145965c4228caf68517d945edd6011b9ed3acacc25e671d5ff17de63c3f0429bb0000000000feffffff02a4800cb3080000001600149102187fea0ada893fd32a7a823374ea81f428e8102700000000000017a91428b5e7e041bb35cc78793ff6741953a13090ab3d87024730440220303ff0212c80305b78651adf5f4d2de152f093b64afa4b0528f45d1864c2715302207ca67132b1cde92914d8a59664d8cd468358ed90ef7c6110d096d91e1cde02fc01210225682ef649fb604da92c7131baf6c5858b3f06d21e3b8f2d602bb488dbebf05e37052000",
  "d36d3ba59981dab0dad0e02dafa9fa97ad51f4e5a47ffad3ad8544bdb251b70f": "020000000001055aa7ebaae0ed0a026bf72011467deb0d21607714d1787d671a2457db72014267000000001716001463e4abed42b06909dc626f9f14e364d377e3e358fdffffff4fa37ac8b379296682cb4d529f6ea264e7e35c3f595c8c596692c91c0c85ddfd01000000171600147ac8f21860d4a529025b494597174eb1a441a57cfdffffff0c69fc1d2c9a3ae75004eaad986cad6b84a64db33913cb94103037f879da171c00000000171600146f6c84387636a119bdaf92cad81396c2ce7a4d58fdfffffff4640d1312f962506978643ccafb843f0230eb9583d9e9e505035b01bf0dc9720000000017160014fad372f775bbc24ab9a6ea3613a8ef81871f78b1fdffffff8c4ca258a1f248d19b0def11b5e046a3fe623c0d06e65c035f858ffcb267d05a0000000017160014f58147b16bf214794f87b534c00a680fe74135e5fdffffff0230341b000000000017a91443e08be49ad8303db84342b24e7ba49f6b0f487487fa1aa3010000000017a914c5c22cf7cccf3021156ddaded63559437c36fe2c8702483045022100d855db0fadb0d705356bf3d8ee333f3e563bff04cda0f56ff5c7b5c15e2c624602200b5db6961d6e8ceef87da9403305bd099ac3ddad0c9d99e2649dfebbc7242ee201210313092d082133d2219380f9f3829db3a0b177e33bb5e5a5b09d84db0969d2ca2a02483045022100b24f394ac5f7ec7270a03a8c20bb423a6bec4122d261c17bd6b1286e0a9706d402204d3158f865b93a54c0432f5d4ed2ce96ed04aa3ed8561eb60b0cdfedd02747de0121032e12e4ab6dcbca1e3aceaf0ce8621fb21571c067015e181f4b9ddfffc176182c0247304402200296b7f4c81291755b32d4c46b8765bdc4716afba20ba57fb51405a60826c1d002205fd1dbd492472e5754f43e19f224a8ec1c309024b012dbb84197817c844e25f8012102fb6077e1a918e96d2ac9301eeee5cd47929a340bf0639ff2ef0a4a55c743bbd3024830450221009089c07ee3e7232c30e1705c74d4f4e23fca2cb130765d7edc8de37f187e4f8202204f513450265596bd00c1271735f9bbdcc380564773881a4d2dd77d3f506855c7012103f001a06f9bbdbe1a479111232193479d4d1939039a25a678d82bd7b718501a7e024730440220073fb0fde51f53689429044c69758a4ca1963ac857114a4764679fe6197934a10220461aedccc59727a2efd0f8ca63b2f27187c2ca766274e499fb4ac500828c23b5012103147ad367eba82a193f79ed2ad8595ee2adcce26fdda1c2640707abfb9589abe900000000",
  "28ad5054e029252d72da37f13fce66212d7f7763845b4a8c4aaf78e897b2bf9f": "0200000000010129b8fe8705de9e342ab80e5ae4237f05dc467586c8d85fbeb29f750d670a849a0200000017160014c1ac0d63d0258ea1b6fe90ef72d0c35d8d773dd3feffffff030000000000000000166a146f6d6e6900000000800005dd00000017d784000097d31c000000000017a91428b5e7e041bb35cc78793ff6741953a13090ab3d871c0200000000000017a914693e66873dedeaed521e3d2dc646fccfdfa4f3978702473044022063e0349bbb6f6b5346bc12cc94289b3dfe3eedee21c03b9f52cef488ec5fe9e5022027f44fb819331ea3c97be06e85d46a797f91315626895c70fb9d2616a927cb1f012102fb255ed920db5c2f507289202eb60a160e5a067ee7e30199a4ed81b74c22e4412e052000",
  // Integration tests:
  // For speculosP2wpkh:
  "13958e34f31f3afd87178ec383ace1a8a0579c2cb9754175553e82e6cdc22251": "02000000000101ec230e53095256052a2428270eec0498944b10f6f1c578f431c23d0098b4ae5a0100000017160014281539820e2de973ae41ba6004b431c921c4d86dfeffffff02727275000000000017a914c8b906af298c70e603a28c3efc2fae19e6ab280f8740420f00000000001976a914cbae5b50cf939e6f531b8a6b7abd788fe14b029788ac02473044022037ecb4248361aafd4f8c11e705f0fa7a5fbdcd595172fcd5643f3b11beff5d400220020c6d326f6c37d63cecadaf4eb335faedf7c44e05f5ef1d2b68140b023bd13d012103dac82fc0acfcfc36348d4a48a46f01cea77f2b9ece3f8c3b4c99d0b0b2f995d284f21c00",
  "6da700fe7d0c0eeb613942d9889941f82cdeec5baaeaf3b9fa4a3f08ffad5b89": "020000000001027a940a941927042501fcd00d6607690db6c563550ba40dcc099628bf797bbd150000000000fdffffff3a010201983755fd33faaaf0e1d0ef19582cbcf72a66f35bf4c5fb94bf540f5e0100000000fdffffff029cdf010000000000225120740ee64e452e3baee127b03c195bcc21ad3edded2ef26c5af483d9c56304d1e560ea000000000000160014e6e988f7292c9452e834bb8fe81d6c5e92aa0aa202473044022011ba8f1377e63b4103a4e9e2a555984995cd1f2e9786287f212854059145e9b5022017ae57782290e4d5cf25c0937fcec06b427ad3dd12bac8c60519ed107b1e8d4a012103b8df3cc65ad612fe50d9ef072252ea048e6521c8df22eab646ffd9cd7ca621d4024730440220564f7f55e6fc220345f3e1292d1954514e351ac6ecca87296245f446c7b4141302205735a33b3069bee78ba5c518b6ace2ddb92ad43472225f3a755934c710ecbdae01210328732312fd061fe78552929d6a6418e448844a9f178604e4c99cae7d254732f165ef1f00",
  "891013bb58b215330eef5bf5141991ed51604fbdf56e9df0fbbf9172b2c0d353": "02000000000101c3945b596fd88fd0d3031a2df8963a68adf9e9f0dfef4f3c4dd72df7d80777ae000000001716001485d33a279d6ae91e59e0c43de832fd4c3398252dfdffffff02909f07000000000017a914f0464d9fa0ea42d80e4d5f1457883982e23b8eec8720a107000000000017a91423e6f63c476f49e8acf6aa583a0fdec7b8aca6458702483045022100da88128e41ae9ea8d6c7fe54a32ab89bbd558993924acf7d86e7039675dd3e0002207228d510d4d8e16f77ef5b82af051a6e5bd5f0263e4464984db8be777dfc6387012102549c187d80a2c8d26760d1646b40e3dce3b9e6042579dcdb9fdeee85208ceb7c00041d00",
  "0dcdf12e61a22c3c2d376941782ab9b76883602fc425cd4598687b3120724fac": "020000000001012c4ca2f29958ecf3e14ddbd449e7dfc716400d7bfee8a9d7eec1647c3a14a7100000000000fdffffff02d02cf800000000001600142318d66f84fef5c4875f933b038dc63831f8da130087930300000000160014eee49309a5ce45d01f691aec8e5ce206194f9a9f02483045022100badae1c06849c8ba6616af3f306e5a2fccd433f23c1fa621e0dc7114f3c84a4002205ca637953d0b6fc2431f3cf6368fa00dbcb328d46c4778d8173cdb51b22a60760121027cb75d34b005c4eb9f62bbf2c457d7638e813e757efcec8fa68677d950b63662f1f61c00",
  "8ea9e17ad9c972028dc9403e7c9312ea97a558f1f3d4c6294a06c3ccda124227": "020000000001046154ff8383f32c78a1d43167c7ee173af0264fa8c1ea4532bb6175e70749b6450100000000fdffffff15e2d13c7111b18d70b87ca855f9ab5cd311337b36a84a5052629ebc309bd5120000000000fdffffff9ca025a9248bf9c8bc5cfd7c4221584d5ddbdea28592f4938e2f28c86adf062e0000000000fdffffffda203b6ec84392527092b694df00c7fe751f2a873358704d482a3098dc2bab4b0000000000fdffffff024e61bc0000000000225120d6583481e19894e6302b0dce260c9953fbd69e5f86eea18dea181132c2dbadf6655b100000000000160014c8f8acc90a9c5d393c6918dc75a5f089855a996802473044022051837752b74082b91a3445eed0edf9ff84e2d86e5379c310cbfd5782e617ed58022015757395225e167df639f68ab92d0185a9665e91819b025107937d3884bb9e8601210229ec47727131ed2588a20c46eda9abb7daa6f49fd50bafe70b9c5aa6961c4ecc02473044022012aaca5b20497831aab971726cb0ff8399033b9877ff7f96c909e1670954256c0220191bc3556e2a1ba279339eee8969322b83ec7841d91903157547dbea577ceb530121027a31bd83824bb0a547cc6b90ce7cf1641f3aff9a9fedf1c2b23133654c47ce8d02473044022063efaa70fe8ad77411af1ac299a4aa17ee77fd1a2fd5faa9b9d7e237c8729eea02204a52690d395d66e1c572132db97a837b3736908baad6eec771b4962ac6a0ad7e012103cf8e2acd4dffe88d6a144966487f781ae822e614b83f89d21215c76e6365201402473044022044c96441703e239fd6d49c9ec2297dda4e19b858f3c0600ab1a643b2016e6b27022021b509d0494388c74f1f5e89be05f7fede0b63195328da7dd9a87ffaa515196f01210232804f473db3282d5baca4c53713463645729335b90d738e18e2286b603d3fc500000000",
  "347efbd10c20a540f677bdf616f06652215497a444a20127dad1520ce2b9f75d": "0200000001f4c2b478dc9cea2df43c8bb94938ff43f3bed5aeb3af9e36cdb7016104626bf2000000006b483045022100b7b8244b3f83648397fa3c37349ed1827bf40bfca22d4f87cc019d3fded63e1d02205434ce222de35e04ab98be54af57d2c8b464f0da76a0275443b33c7b1df79344012103462f1e48396ce2a4d7694d107fef5123fe4d1e394394a370fa4901e744b4884efdffffff02dcdb0600000000001976a914b242ba04b8526b533b17b168d27eaf0da7bacb8588ac70640800000000001976a91426c680ec6ac9c0ed760177f92096f1c5571b72ef88ac20041d00",
  "de390b0c4ac67b6cdcbe84611463cc42b6774f2bf11d8f7144e9601e3591f857": "02000000015df7b9e20c52d1da2701a244a49754215266f016f6bd77f640a5200cd1fb7e34010000006b483045022100c806957429935c5d61f548223c3536dc739fe351814bce9edc1694e5a5c25abf022013e307b0b4885acb64ccffaf5f4794f205040325ebc36c258787c488b038c2bb0121038ba462d7dd65917ece454142903bb362a0b412d1841d52087ad881646b6cdccffdffffff02140c0300000000001976a914b3e17c4def8a39148ed6fad7dc6febe30dd324cb88ac30570500000000001976a914e0a89380635f18b7ee09546d963232bcb992412b88acfe041d00",
};

// Taken from bitcoin-core testnet
export const p2tr: CoreTx = {
  "txid": "c77a7c5338ed59b934193209f16ea74ec6d846ef61e9fb9705d99e9896d79e25",
  "hash": "d5b0513aeb462fb52cf977a8d367021fa3e66002194a8b8dd2fe97e6e52b81e4",
  "version": 2,
  "size": 150,
  "vsize": 99,
  "weight": 396,
  "locktime": 2097065,
  "vin": [
    {
      "txid": "0d41be343ced2023ff112b123a7bb5c8f345becddb8f1cc7070e1273a21d3710",
      "vout": 0,
      "scriptSig": {
        "asm": "",
        "hex": ""
      },
      "txinwitness": [
        "8d370b53421e0ae033f736a73af72b35562b584143adca592fd918a0bb252ca904b379393446c8dddba08974a2122802844afcc03ef24bcb2312b81a5c5d65ac"
      ],
      "sequence": 4294967293
    }
  ],
  "vout": [
    {
      "value": 0.00081742,
      "n": 0,
      "scriptPubKey": {
        "asm": "0 f5d7911e8d7c560dfc3cf6e3079a4c7c0dce70e9",
        "hex": "0014f5d7911e8d7c560dfc3cf6e3079a4c7c0dce70e9",
        "reqSigs": 1,
        "type": "witness_v0_keyhash",
        "addresses": [
          "tb1q7htez85d03tqmlpu7m3s0xjv0sxuuu8fg40ght"
        ]
      }
    }
  ],
  "hex": "0200000000010110371da273120e07c71c8fdbcdbe45f3c8b57b3a122b11ff2320ed3c34be410d0000000000fdffffff014e3f010000000000160014f5d7911e8d7c560dfc3cf6e3079a4c7c0dce70e901408d370b53421e0ae033f736a73af72b35562b584143adca592fd918a0bb252ca904b379393446c8dddba08974a2122802844afcc03ef24bcb2312b81a5c5d65aca9ff1f00",
  "blockhash": "0000000000000024a3605855f3b669e9e165ab242e53761ecbaae4b6167829d3",
  "confirmations": 1017,
  "time": 1633037023,
  "blocktime": 1633037023
};

export const p2wpkh = {
    "txid": "062c623b6a9486aaa11b60e5c37bddcd6214aa642172d81f485faf3b2fc2996e",
    "hash": "6fabbc622e0d8c9b099fce3a2ac92334a1902302f9f572525b563e3ad97aeab0",
    "version": 2,
    "size": 226,
    "vsize": 145,
    "weight": 577,
    "locktime": 2098057,
    "vin": [
      {
        "txid": "5c02391a5b8f55ebb0db0e4e75900ace70f7e8e992115b3931379ab32b69da40",
        "vout": 0,
        "scriptSig": {
          "asm": "",
          "hex": ""
        },
        "txinwitness": [
          "30440220667c54bc57770a563ef74e68331ee6a27d3a1b073d22e867fda346273d768da50220506609d58f78ab6dfe78d1635cf1974a234347f080da86ed1ee9b0ec345b7c1901",
          "02a236cce923cfb073fd4bfdaf0a8c5ae2f2144052a98549810cd3fe39dcf1e1d7"
        ],
        "sequence": 4294967294
      }
    ],
    "vout": [
      {
        "value": 0.09950330,
        "n": 0,
        "scriptPubKey": {
          "asm": "OP_HASH160 afe05670e110d5203e0544137d2ede18338ced8b OP_EQUAL",
          "hex": "a914afe05670e110d5203e0544137d2ede18338ced8b87",
          "reqSigs": 1,
          "type": "scripthash",
          "addresses": [
            "2N9HAz7E4itqt78dDkN2ZNkAYaKp1pmvnxH"
          ]
        }
      },
      {
        "value": 0.00100000,
        "n": 1,
        "scriptPubKey": {
          "asm": "OP_DUP OP_HASH160 c025f47a88d88b09aa9b4cb7f92010b1d513e300 OP_EQUALVERIFY OP_CHECKSIG",
          "hex": "76a914c025f47a88d88b09aa9b4cb7f92010b1d513e30088ac",
          "reqSigs": 1,
          "type": "pubkeyhash",
          "addresses": [
            "my2wW3Yic3huFYiHWSXy1sMX9DgeQRHmg9"
          ]
        }
      }
    ],
    "hex": "0200000000010140da692bb39a3731395b1192e9e8f770ce0a90754e0edbb0eb558f5b1a39025c0000000000feffffff027ad497000000000017a914afe05670e110d5203e0544137d2ede18338ced8b87a0860100000000001976a914c025f47a88d88b09aa9b4cb7f92010b1d513e30088ac024730440220667c54bc57770a563ef74e68331ee6a27d3a1b073d22e867fda346273d768da50220506609d58f78ab6dfe78d1635cf1974a234347f080da86ed1ee9b0ec345b7c19012102a236cce923cfb073fd4bfdaf0a8c5ae2f2144052a98549810cd3fe39dcf1e1d789032000",
  "blockhash": "0000000000000012201f4dd0b06cdbc3b8a037a5c2c0486c4523688555613a53",
  "confirmations": 16,
  "time": 1633594907,
  "blocktime": 1633594907
};

export const wrappedP2wpkh: CoreTx = {
  "txid": "3f3b8e65eb666dc9950ddf1b4b9a7438e9baae710a6a7c06d2582bd2d750ed54",
  "hash": "a3035b0af8eb2176e4e4f459233c7e669e4a8becbf32cee7d1639ba52b671c5b",
  "version": 2,
  "size": 248,
  "vsize": 166,
  "weight": 662,
  "locktime": 0,
  "vin": [
    {
      "txid": "5742c24f1485958d0e36d0ae758b1e33c01ba99107eb54baa539cc7db233a330",
      "vout": 0,
      "scriptSig": {
        "asm": "00142d62a39a40a30228c77de441c1f0993fe1707dda",
        "hex": "1600142d62a39a40a30228c77de441c1f0993fe1707dda"
      },
      "txinwitness": [
        "3045022100d4a981592a3e314ee20c662b7a34339d26855c0ecff1588154f0f41856a17a990220444364570c93d5a7a82de8d863dc51868e69ff357901926eedd43c305e51818701",
        "03efc6b990c1626d08bd176aab0e545a4f55c627c7ddee878d12bbbc46a126177a"
      ],
      "sequence": 4294967295
    }
  ],
  "vout": [
    {
      "value": 98.74662473,
      "n": 0,
      "scriptPubKey": {
        "asm": "OP_HASH160 5a102ac5376429003168cad88c80331daa3a7471 OP_EQUAL",
        "hex": "a9145a102ac5376429003168cad88c80331daa3a747187",
        "reqSigs": 1,
        "type": "scripthash",
        "addresses": [
          "2N1TSArdd2pt9RoqE3LXY55ixpRE9e5aot8"
        ]
      }
    },
    {
      "value": 0.00010000,
      "n": 1,
      "scriptPubKey": {
        "asm": "OP_HASH160 e5776c571979173607c28d5c1882465014b777dc OP_EQUAL",
        "hex": "a914e5776c571979173607c28d5c1882465014b777dc87",
        "reqSigs": 1,
        "type": "scripthash",
        "addresses": [
          "2NEAXntA12Zoa7NdsNyC9ncw7HLKH66jKNH"
        ]
      }
    }
  ],
  "hex": "0200000000010130a333b27dcc39a5ba54eb0791a91bc0331e8b75aed0360e8d9585144fc2425700000000171600142d62a39a40a30228c77de441c1f0993fe1707ddaffffffff024964934c0200000017a9145a102ac5376429003168cad88c80331daa3a747187102700000000000017a914e5776c571979173607c28d5c1882465014b777dc8702483045022100d4a981592a3e314ee20c662b7a34339d26855c0ecff1588154f0f41856a17a990220444364570c93d5a7a82de8d863dc51868e69ff357901926eedd43c305e518187012103efc6b990c1626d08bd176aab0e545a4f55c627c7ddee878d12bbbc46a126177a00000000",
  "blockhash": "00000000c117e8cb6cb65ef6afc22dda3ab906dc6a42669a154fea124ecec3ca",
  "confirmations": 3,
  "time": 1633609698,
  "blocktime": 1633609698
};

export const p2pkh: CoreTx = {
  "txid": "03717ce615625148a3a3aad38a68fa2aa68e54633259cb98a7a16c16c33a71c3",
  "hash": "03717ce615625148a3a3aad38a68fa2aa68e54633259cb98a7a16c16c33a71c3",
  "version": 2,
  "size": 254,
  "vsize": 254,
  "weight": 1016,
  "locktime": 0,
  "vin": [
    {
      "txid": "80453b180c6616431d28d395a6a89cb331d7ecb21bf3ac1cbb68c90dba6e95ff",
      "vout": 1,
      "scriptSig": {
        "asm": "304402205c809d58dadb3dbe1b9cf129549036b00f7dfa876f7f6c1686d8df77b69cef2c02206512480ee394c3298c8b66340c0ccddddc3c245b4e870a31bf52c9eec9ad69b6[ALL] 037ed58c914720772c59f7a1e7e76fba0ef95d7c5667119798586301519b9ad2cf",
        "hex": "47304402205c809d58dadb3dbe1b9cf129549036b00f7dfa876f7f6c1686d8df77b69cef2c02206512480ee394c3298c8b66340c0ccddddc3c245b4e870a31bf52c9eec9ad69b60121037ed58c914720772c59f7a1e7e76fba0ef95d7c5667119798586301519b9ad2cf"
      },
      "sequence": 4294967295
    }
  ],
  "vout": [
    {
      "value": 0.00000000,
      "n": 0,
      "scriptPubKey": {
        "asm": "OP_RETURN 696f6e3a322e516d59386b56553674567364344a65764d64656d5a744143635152647675776b546158344d455a4e336778677072",
        "hex": "6a34696f6e3a322e516d59386b56553674567364344a65764d64656d5a744143635152647675776b546158344d455a4e336778677072",
        "type": "nulldata"
      }
    },
    {
      "value": 1.33032112,
      "n": 1,
      "scriptPubKey": {
        "asm": "OP_DUP OP_HASH160 7628046f6ec479c031d03a47b6a003c4dedb1cad OP_EQUALVERIFY OP_CHECKSIG",
        "hex": "76a9147628046f6ec479c031d03a47b6a003c4dedb1cad88ac",
        "reqSigs": 1,
        "type": "pubkeyhash",
        "addresses": [
          "mrHhy9DgpBbDLoJsACv4QXXY7f2B5Fq5o1"
        ]
      }
    }
  ],
  "hex": "0200000001ff956eba0dc968bb1cacf31bb2ecd731b39ca8a695d3281d4316660c183b4580010000006a47304402205c809d58dadb3dbe1b9cf129549036b00f7dfa876f7f6c1686d8df77b69cef2c02206512480ee394c3298c8b66340c0ccddddc3c245b4e870a31bf52c9eec9ad69b60121037ed58c914720772c59f7a1e7e76fba0ef95d7c5667119798586301519b9ad2cfffffffff020000000000000000366a34696f6e3a322e516d59386b56553674567364344a65764d64656d5a744143635152647675776b546158344d455a4e336778677072b0e8ed07000000001976a9147628046f6ec479c031d03a47b6a003c4dedb1cad88ac00000000",
  "blockhash": "0000000000000040f9117341ca31f40e4f440fc6f6552a3b3f15e96ed9edeb3e",
  "confirmations": 1,
  "time": 1633611385,
  "blocktime": 1633611385
};
// eslint-disable-next-line @typescript-eslint/no-unused-vars
export const wrappedP2wpkhTwoInputs = {
  "txid": "c03119b538c78f56c8ce2e6cc5fc6998d447eeef42e34c12692764a3f1a3da7c",
  "hash": "6b3812304554a6964e43a6971ac533046f4be101e39609f72179856916e20268",
  "version": 2,
  "size": 420,
  "vsize": 257,
  "weight": 1026,
  "locktime": 0,
  "vin": [
    {
      "txid": "9075ef8fd97f92ff0db344322873f12c42b29661c3960801e05114ba8adcabd6",
      "vout": 0,
      "scriptSig": {
        "asm": "0014e4a7ff7c7e16cb6f15914938e2b92e2801220250",
        "hex": "160014e4a7ff7c7e16cb6f15914938e2b92e2801220250"
      },
      "txinwitness": [
        "30450221008a777087167aaeda51cf3532da368a7541630cd7274068ae4353aa1e9e32d7850220628ad0f414ae4cfb7a6dcd590b0e21e003b2ff638c5fc9aaad9b411783b03e3301",
        "02de057221383ed65635568e38d4305d9120a4e68e205734a5f95a8feea3dd5f53"
      ],
      "sequence": 4294967293
    },
    {
      "txid": "d36d3ba59981dab0dad0e02dafa9fa97ad51f4e5a47ffad3ad8544bdb251b70f",
      "vout": 0,
      "scriptSig": {
        "asm": "001442721355859d8f2a461a5badfb19e59a61935692",
        "hex": "16001442721355859d8f2a461a5badfb19e59a61935692"
      },
      "txinwitness": [
        "3045022100e49eac220605239d702261c5929e4544e2d3ea70f3372527cee7f6cb31dabd24022015b67591a07e15a7a35ac5cc775e2abbc7fdf41983ad14438961e76029aee52c01",
        "02783edf31f3a8845c6350d17a254a73561c02f1d00c69a8972fd91472c1920105"
      ],
      "sequence": 4294967293
    }
  ],
  "vout": [
    {
      "value": 0.03532507,
      "n": 0,
      "scriptPubKey": {
        "asm": "OP_HASH160 deb26fa74cb3bb95cd00213172397ec91c5fc9ba OP_EQUAL",
        "hex": "a914deb26fa74cb3bb95cd00213172397ec91c5fc9ba87",
        "address": "2NDYjkqA8jUpMAqU2rBaSumb5bqa41Ri5BM",
        "type": "scripthash"
      }
    },
    {
      "value": 0.00069944,
      "n": 1,
      "scriptPubKey": {
        "asm": "OP_HASH160 c0a4669d0b6fd3ce7b22e158ac160f2e656f2592 OP_EQUAL",
        "hex": "a914c0a4669d0b6fd3ce7b22e158ac160f2e656f259287",
        "address": "2NAopftKM5sNgM7AueL7Cin9NsFv8ykBkUA",
        "type": "scripthash"
      }
    }
  ],
  "hex": "02000000000102d6abdc8aba1451e0010896c36196b2422cf173283244b30dff927fd98fef75900000000017160014e4a7ff7c7e16cb6f15914938e2b92e2801220250fdffffff0fb751b2bd4485add3fa7fa4e5f451ad97faa9af2de0d0dab0da8199a53b6dd3000000001716001442721355859d8f2a461a5badfb19e59a61935692fdffffff02dbe635000000000017a914deb26fa74cb3bb95cd00213172397ec91c5fc9ba87381101000000000017a914c0a4669d0b6fd3ce7b22e158ac160f2e656f259287024830450221008a777087167aaeda51cf3532da368a7541630cd7274068ae4353aa1e9e32d7850220628ad0f414ae4cfb7a6dcd590b0e21e003b2ff638c5fc9aaad9b411783b03e33012102de057221383ed65635568e38d4305d9120a4e68e205734a5f95a8feea3dd5f5302483045022100e49eac220605239d702261c5929e4544e2d3ea70f3372527cee7f6cb31dabd24022015b67591a07e15a7a35ac5cc775e2abbc7fdf41983ad14438961e76029aee52c012102783edf31f3a8845c6350d17a254a73561c02f1d00c69a8972fd91472c192010500000000",
  "blockhash": "00000000025a711e6cd4bce9138dc852232a4494afbf36d8bb80499a786da2a4",
  "confirmations": 1,
  "time": 1633944124,
  "blocktime": 1633944124
};

export const speculosP2pkh = {
  "txid": "f26b62046101b7cd369eafb3aed5bef343ff3849b98b3cf42dea9cdc78b4c2f4",
  "hash": "f26b62046101b7cd369eafb3aed5bef343ff3849b98b3cf42dea9cdc78b4c2f4",
  "version": 2,
  "size": 192,
  "vsize": 192,
  "weight": 768,
  "locktime": 1901594,
  "vin": [
    {
      "txid": "13958e34f31f3afd87178ec383ace1a8a0579c2cb9754175553e82e6cdc22251",
      "vout": 1,
      "scriptSig": {
        "asm": "3045022100e55b3ca788721aae8def2eadff710e524ffe8c9dec1764fdaa89584f9726e196022012a30fbcf9e1a24df31a1010356b794ab8de438b4250684757ed5772402540f4[ALL] 02ee8608207e21028426f69e76447d7e3d5e077049f5e683c3136c2314762a4718",
        "hex": "483045022100e55b3ca788721aae8def2eadff710e524ffe8c9dec1764fdaa89584f9726e196022012a30fbcf9e1a24df31a1010356b794ab8de438b4250684757ed5772402540f4012102ee8608207e21028426f69e76447d7e3d5e077049f5e683c3136c2314762a4718"
      },
      "sequence": 4294967293
    }
  ],
  "vout": [
    {
      "value": 0.00999800,
      "n": 0,
      "scriptPubKey": {
        "asm": "OP_DUP OP_HASH160 13d7d58166946c3ec022934066d8c0d111d1bb41 OP_EQUALVERIFY OP_CHECKSIG",
        "hex": "76a91413d7d58166946c3ec022934066d8c0d111d1bb4188ac",
        "address": "mhKsh7EzJo1gSU1vrpyejS1qsJAuKyaWWg",
        "type": "pubkeyhash"
      }
    }
  ],
  "hex": "02000000015122c2cde6823e55754175b92c9c57a0a8e1ac83c38e1787fd3a1ff3348e9513010000006b483045022100e55b3ca788721aae8def2eadff710e524ffe8c9dec1764fdaa89584f9726e196022012a30fbcf9e1a24df31a1010356b794ab8de438b4250684757ed5772402540f4012102ee8608207e21028426f69e76447d7e3d5e077049f5e683c3136c2314762a4718fdffffff0178410f00000000001976a91413d7d58166946c3ec022934066d8c0d111d1bb4188ac1a041d00",
  "blockhash": "00000000000000103db7f31cf5c6a0acdc8391d1a7c9e999b89180d2be20cafc",
  "confirmations": 197323,
  "time": 1609422887,
  "blocktime": 1609422887
};

export const speculosP2tr = {
  "txid": "4bab2bdc98302a484d705833872a1f75fec700df94b69270529243c86e3b20da",
  "hash": "fd34d81f333bd228bbcba39e1d7fa53932613d5e139357eb514b264254d158a7",
  "version": 2,
  "size": 151,
  "vsize": 100,
  "weight": 397,
  "locktime": 0,
  "vin": [
    {
      "txid": "6da700fe7d0c0eeb613942d9889941f82cdeec5baaeaf3b9fa4a3f08ffad5b89",
      "vout": 0,
      "scriptSig": {
        "asm": "",
        "hex": ""
      },
      "txinwitness": [
        "58738807af5acb436cb8e15dde483dc52ac8e5b5e4f17c0309439ea4462405b276fd5fdd8f4610abb5acadb7b6bb570cbb9a2349f5522bd3493fab2d6690e2c901"
      ],
      "sequence": 4294967295
    }
  ],
  "vout": [
    {
      "value": 0.00110000,
      "n": 0,
      "scriptPubKey": {
        "asm": "0 3318e04fae6c12afcb009c69cd57e5b2504ae6b4",
        "hex": "00143318e04fae6c12afcb009c69cd57e5b2504ae6b4",
        "address": "tb1qxvvwqnawdsf2ljcqn35u64l9kfgy4e45v3uamu",
        "type": "witness_v0_keyhash"
      }
    }
  ],
  "hex": "02000000000101895badff083f4afab9f3eaaa5becde2cf8419988d9423961eb0e0c7dfe00a76d0000000000ffffffff01b0ad0100000000001600143318e04fae6c12afcb009c69cd57e5b2504ae6b4014158738807af5acb436cb8e15dde483dc52ac8e5b5e4f17c0309439ea4462405b276fd5fdd8f4610abb5acadb7b6bb570cbb9a2349f5522bd3493fab2d6690e2c90100000000",
  "blockhash": "00000000000000439711ed1340c4988588f9a40d140839f8d6b99c461160787a",
  "confirmations": 5417,
  "time": 1631612662,
  "blocktime": 1631612662
};

export const speculosP2wpkh2inWithChange = {
  "txid": "5c09c82a4efa271adb692917aa35e92b1b360e333fade6cf081952bc7fd16a7a",
  "hash": "3d2c075e64f97800397962f9054ebc5f3d464afa9090d32a5a4791c92d2bed6e",
  "version": 2,
  "size": 370,
  "vsize": 208,
  "weight": 832,
  "locktime": 1901123,
  "vin": [
    {
      "txid": "0dcdf12e61a22c3c2d376941782ab9b76883602fc425cd4598687b3120724fac",
      "vout": 0,
      "scriptSig": {
        "asm": "",
        "hex": ""
      },
      "txinwitness": [
        "304402203aed260c81d21e36cfdd5a095ad289366103ae3a2690011b20e15001173e44d902206f17e60fc169db8bc21820dfefe51271446a6305284bcfb73738ca38a2715d9f01",
        "03455ee7cedc97b0ba435b80066fc92c963a34c600317981d135330c4ee43ac7a3"
      ],
      "sequence": 4294967293
    },
    {
      "txid": "0dcdf12e61a22c3c2d376941782ab9b76883602fc425cd4598687b3120724fac",
      "vout": 1,
      "scriptSig": {
        "asm": "",
        "hex": ""
      },
      "txinwitness": [
        "30440220223a3f37f85ed5e42d6ed463ca36932db28799ff12d0391b5e02bcfef829007302206c42e9cb5817bff184e8854848032867f129814df07a96afc49baa5542ede8a401",
        "03affafaf410b74f019028055618695f89c23478aa5fca3ba531f44d382ad7791a"
      ],
      "sequence": 4294967293
    }
  ],
  "vout": [
    {
      "value": 0.16264190,
      "n": 0,
      "scriptPubKey": {
        "asm": "0 c132f90de4a19728e35d8af8b3f9ec4e43cf1948",
        "hex": "0014c132f90de4a19728e35d8af8b3f9ec4e43cf1948",
        "address": "tb1qcye0jr0y5xtj3c6a3tut870vfepu7x2ge8sapk",
        "type": "witness_v0_keyhash"
      }
    },
    {
      "value": 0.60000001,
      "n": 1,
      "scriptPubKey": {
        "asm": "0 3318e04fae6c12afcb009c69cd57e5b2504ae6b4",
        "hex": "00143318e04fae6c12afcb009c69cd57e5b2504ae6b4",
        "address": "tb1qxvvwqnawdsf2ljcqn35u64l9kfgy4e45v3uamu",
        "type": "witness_v0_keyhash"
      }
    }
  ],
  "hex": "02000000000102ac4f7220317b689845cd25c42f608368b7b92a784169372d3c2ca2612ef1cd0d0000000000fdffffffac4f7220317b689845cd25c42f608368b7b92a784169372d3c2ca2612ef1cd0d0100000000fdffffff02fe2bf80000000000160014c132f90de4a19728e35d8af8b3f9ec4e43cf194801879303000000001600143318e04fae6c12afcb009c69cd57e5b2504ae6b40247304402203aed260c81d21e36cfdd5a095ad289366103ae3a2690011b20e15001173e44d902206f17e60fc169db8bc21820dfefe51271446a6305284bcfb73738ca38a2715d9f012103455ee7cedc97b0ba435b80066fc92c963a34c600317981d135330c4ee43ac7a3024730440220223a3f37f85ed5e42d6ed463ca36932db28799ff12d0391b5e02bcfef829007302206c42e9cb5817bff184e8854848032867f129814df07a96afc49baa5542ede8a4012103affafaf410b74f019028055618695f89c23478aa5fca3ba531f44d382ad7791a43021d00",
  "blockhash": "0000000019b9a194c4b54d494499106e1a06d87403a3c65f8eaaf70e871ed161",
  "confirmations": 198326,
  "time": 1609193209,
  "blocktime": 1609193209
};

export const p2wpkhWrapped2in2out = {
  "txid": "f5ff97147f8c7cca717536325eb157c696ef9630934271905b3f8c564b18e708",
  "hash": "3480427ed1da3802b1a7326ac283c13b8319689035890066334db7203928a7e4",
  "version": 2,
  "size": 420,
  "vsize": 257,
  "weight": 1026,
  "locktime": 1901520,
  "vin": [
    {
      "txid": "891013bb58b215330eef5bf5141991ed51604fbdf56e9df0fbbf9172b2c0d353",
      "vout": 0,
      "scriptSig": {
        "asm": "0014cb078087eff485aaa2260e94a53d7d6d1c5dd151",
        "hex": "160014cb078087eff485aaa2260e94a53d7d6d1c5dd151"
      },
      "txinwitness": [
        "3045022100a0173bbd3dcdfe51c24386fee905b8197f36c2024f582b1a838e7bf09a1ec16902206bcd803041b13738f7d8896561e6a8ccec9a3f5d243be8645c403e83beb19b3401",
        "024ba3b77d933de9fa3f9583348c40f3caaf2effad5b6e244ece8abbfcc7244f67"
      ],
      "sequence": 4294967293
    },
    {
      "txid": "891013bb58b215330eef5bf5141991ed51604fbdf56e9df0fbbf9172b2c0d353",
      "vout": 1,
      "scriptSig": {
        "asm": "00143f8f2b556915a9306fd92eb5ae72217be9dd593f",
        "hex": "1600143f8f2b556915a9306fd92eb5ae72217be9dd593f"
      },
      "txinwitness": [
        "3045022100ac5701f0cc3cb4b490269674d3463c4fab74f6ee9faf7c29b3d0dea22875dd0002206ba4f4195dce69ffd3e7d832ee5d67ee0d037bad03f4e00f841fbf84649375b501",
        "03a1989dadf0bce57f4c96cfc4d9e6aa9abeba7ac0733e99176f69432e852666f1"
      ],
      "sequence": 4294967293
    }
  ],
  "vout": [
    {
      "value": 0.00199300,
      "n": 0,
      "scriptPubKey": {
        "asm": "OP_HASH160 ffc91a30e33fc6d6ecff42e4d9bd6c7e115d8498 OP_EQUAL",
        "hex": "a914ffc91a30e33fc6d6ecff42e4d9bd6c7e115d849887",
        "address": "2NGZhD2vqXhdeCYmfGcqaYFXzgmwqyfVhtk",
        "type": "scripthash"
      }
    },
    {
      "value": 0.00800000,
      "n": 1,
      "scriptPubKey": {
        "asm": "OP_HASH160 2f5864a8acd23fa85977d73e9ac30fd6b341b78c OP_EQUAL",
        "hex": "a9142f5864a8acd23fa85977d73e9ac30fd6b341b78c87",
        "address": "2MwZZdy3bCvEhiPsBycreCFiqwDhykAcWN9",
        "type": "scripthash"
      }
    }
  ],
  "hex": "0200000000010253d3c0b27291bffbf09d6ef5bd4f6051ed911914f55bef0e3315b258bb1310890000000017160014cb078087eff485aaa2260e94a53d7d6d1c5dd151fdffffff53d3c0b27291bffbf09d6ef5bd4f6051ed911914f55bef0e3315b258bb13108901000000171600143f8f2b556915a9306fd92eb5ae72217be9dd593ffdffffff02840a03000000000017a914ffc91a30e33fc6d6ecff42e4d9bd6c7e115d84988700350c000000000017a9142f5864a8acd23fa85977d73e9ac30fd6b341b78c8702483045022100a0173bbd3dcdfe51c24386fee905b8197f36c2024f582b1a838e7bf09a1ec16902206bcd803041b13738f7d8896561e6a8ccec9a3f5d243be8645c403e83beb19b340121024ba3b77d933de9fa3f9583348c40f3caaf2effad5b6e244ece8abbfcc7244f6702483045022100ac5701f0cc3cb4b490269674d3463c4fab74f6ee9faf7c29b3d0dea22875dd0002206ba4f4195dce69ffd3e7d832ee5d67ee0d037bad03f4e00f841fbf84649375b5012103a1989dadf0bce57f4c96cfc4d9e6aa9abeba7ac0733e99176f69432e852666f1d0031d00",
  "blockhash": "00000000000000103db7f31cf5c6a0acdc8391d1a7c9e999b89180d2be20cafc",
  "confirmations": 197745,
  "time": 1609422887,
  "blocktime": 1609422887
};

/*
speculos p2wpkh addresses:
    m/84'/1'/0'/0/0:tb1qzdr7s2sr0dwmkwx033r4nujzk86u0cy6fmzfjk                          
    m/84'/1'/0'/0/1:tb1qyvvdvmuylm6ufp6ljvas8rwx8qcl3ksnad49ra                          
    m/84'/1'/0'/0/2:tb1qdqcp6uv54vwx6m9z40wreyusw67p4us6kn26ra                          
    m/84'/1'/0'/0/3:tb1qxvvwqnawdsf2ljcqn35u64l9kfgy4e45v3uamu
    m/84'/1'/0'/0/4:tb1q738p97ezlrkwy3sz7umrpu62037x95leext42g
    m/84'/1'/0'/0/5:tb1qfnsfmnrvvkys0qt9a6x4xjcjmyxw3c9rlx0nhm
    m/84'/1'/0'/0/6:tb1qzwvm3gycg7srnvp49avdyykpszatzlmh8pvkl5
    m/84'/1'/0'/0/7:tb1qu8hysfeazlzd000gqdvqa6qvgdlqf5lhkppap5
    m/84'/1'/0'/0/8:tb1qcu5tezxx6yjxl0wxs4sffhanff38rvkykx5rq0
    m/84'/1'/0'/0/9:tb1q7l0u7vda2u90rwcwn0lkvnaruc24hj3qru9mrp
    m/84'/1'/0'/0/10:tb1qp4j23ldhxxjyw76h82pfksp9emz80pvq49d8hh
    m/84'/1'/0'/0/11:tb1qyevmjsysv9pv4thuha599v8sxkcw4nvkvpclh2
    m/84'/1'/0'/0/12:tb1qqefalksn3lj0372qee9z38z3erqpzgvfmp3x7e
    m/84'/1'/0'/0/13:tb1qfudg6ctvxww7yzp9mt8wxdwm7kyurfe66srxz7
    m/84'/1'/0'/0/14:tb1q7qhl6yqpdtg3lkqdnwpk3tc8y7z2dkrpx4cnn2
    m/84'/1'/0'/0/15:tb1qpsplrg6va5vpdwp4dvcqsgqjunts3zhtm09cu2
    m/84'/1'/0'/0/16:tb1q57cjn3qvv40mjuxhtge0xpacxwgs9csev3pkgh
    m/84'/1'/0'/0/17:tb1qcdj07hklfcsmfvkujcu54w3txhpa6v8x79hj9z
    m/84'/1'/0'/0/18:tb1quev9d5nmpqraptqt9ardzswww4aet2dkwm5ltj
    m/84'/1'/0'/0/19:tb1q3xrvvs2yanmsse2s35gcckttfqa808ss8xnct9
    m/84'/1'/0'/1/0:tb1qxhrwphtd8jztz6agshw2wwkgv0gttylvumsz3m
    m/84'/1'/0'/1/1:tb1q9wfn7609avjy58cyhm2farkfunkf80yl4tc2jj
    m/84'/1'/0'/1/2:tb1qeru2ejg2n3wnj0rfrrw8tf0s3xz44xtgdvqkvv
    m/84'/1'/0'/1/3:tb1qzr0vg9wkn23eslu9z9ys5kh96xehmjqrec07a3
    m/84'/1'/0'/1/4:tb1qamjfxzd9eezaq8mfrtkguh8zqcv5lx5lp3kvrz
    m/84'/1'/0'/1/5:tb1qcye0jr0y5xtj3c6a3tut870vfepu7x2ge8sapk
    m/84'/1'/0'/1/6:tb1q4e9fuyk89wh60sya5tgdjf8q25hldrp9wacd07
    m/84'/1'/0'/1/7:tb1qms5xurxju67n0n47nmp5n76lpzg9vuhuvanyjz
    m/84'/1'/0'/1/8:tb1q8tuy9xk4j492tm52x0yc8lv2r6r8nyjtskt34a
    m/84'/1'/0'/1/9:tb1qastfds4f069l48dkzppl7m5h40nr7dkyxfyept
    m/84'/1'/0'/1/10:tb1qavu04xup9rup7fhftmdschl6a2pkjrlygawmw3
    m/84'/1'/0'/1/11:tb1qh3gal0s6wwx2c4hd05gl6c8rwfmzv0khnamrjk
    m/84'/1'/0'/1/12:tb1q2rhdh0gesesqrn4a4wuakuk68n0tlaelcj35pt
    m/84'/1'/0'/1/13:tb1qf4kgkt5l7zjksn6d7rxttuq39uczmeu2uudzyk
    m/84'/1'/0'/1/14:tb1qr58lz3vldcqlaty8sdc0ys76du0r58246wdu09
    m/84'/1'/0'/1/15:tb1qlrvzyx8jcjfj2xuy69du9trtxnsvjuped7e289
    m/84'/1'/0'/1/16:tb1q2h8dd834vvy2vx30yggeft7ftahvxx7we9sk30
    m/84'/1'/0'/1/17:tb1qk4kn5ke7d9fkqplka0k2dj65rupehtchq29syh
    m/84'/1'/0'/1/18:tb1qyhm3ww9762scr2qv94n7n6qxddpexr7l85tnhh
    m/84'/1'/0'/1/19:tb1q734uqw9aa70vsq0ewwyf3rmr9cvwsfyy6232up

    speculos p2tr transactions:
    "6dfea6a9c7cb2e2ab09da2a83d51057ae17d5b6e9bef34494b696d33d223b8a1"
      in: tb1p6evrfq0pnz2wvvptph8zvrye20aad8jlsmh2rr02rqgn9skm4hmqks9rnm 86'/1'/0'/0/1
      out: tb1q738p97ezlrkwy3sz7umrpu62037x95leext42g
           tb1pmr60r5vfjmdkrwcu4a2z8h39mzs7a6wf2rfhuml6qgcp940x9cxs7t9pdy 86'/1'/0'/1/0
    "05724d38a82c61a16fae335d28f4e197cba459328dd072170196cee459792148"
      in: tb1pmr60r5vfjmdkrwcu4a2z8h39mzs7a6wf2rfhuml6qgcp940x9cxs7t9pdy
      out: tb1ppw88fp4uxp4q4mah5lqq4uzq24pdkxkdw0hve2p8w8x0tlj7kc8sx2ph6j
           tb1qzwvm3gycg7srnvp49avdyykpszatzlmh8pvkl5
    speculos p2tr addresses:


*/
export const speculosP2trWithChange = {
  "txid": "6dfea6a9c7cb2e2ab09da2a83d51057ae17d5b6e9bef34494b696d33d223b8a1",
  "hash": "a9d04857a6410e573b2e8b4fa0da4d7a2973a20af137be595f555b0f85874108",
  "version": 2,
  "size": 194,
  "vsize": 143,
  "weight": 569,
  "locktime": 0,
  "vin": [
    {
      "txid": "8ea9e17ad9c972028dc9403e7c9312ea97a558f1f3d4c6294a06c3ccda124227",
      "vout": 0,
      "scriptSig": {
        "asm": "",
        "hex": ""
      },
      "txinwitness": [
        "0ab0333ae38de68f02057a4523be1ff372130cbf9b43d5c76400d8bb74033c1fa100d572814cdfbc74bc450425f16a86e59f79ffad554423497f6507f9c6d6f401"
      ],
      "sequence": 4294967293
    }
  ],
  "vout": [
    {
      "value": 0.01234567,
      "n": 0,
      "scriptPubKey": {
        "asm": "0 f44e12fb22f8ece24602f73630f34a7c7c62d3f9",
        "hex": "0014f44e12fb22f8ece24602f73630f34a7c7c62d3f9",
        "address": "tb1q738p97ezlrkwy3sz7umrpu62037x95leext42g",
        "type": "witness_v0_keyhash"
      }
    },
    {
      "value": 0.11110055,
      "n": 1,
      "scriptPubKey": {
        "asm": "1 d8f4f1d18996db61bb1caf5423de25d8a1eee9c950d37e6ffa023012d5e62e0d",
        "hex": "5120d8f4f1d18996db61bb1caf5423de25d8a1eee9c950d37e6ffa023012d5e62e0d",
        "address": "tb1pmr60r5vfjmdkrwcu4a2z8h39mzs7a6wf2rfhuml6qgcp940x9cxs7t9pdy",
        "type": "witness_v1_taproot"
      }
    }
  ],
  "hex": "02000000000101274212daccc3064a29c6d4f3f158a597ea12937c3e40c98d0272c9d97ae1a98e0000000000fdffffff0287d6120000000000160014f44e12fb22f8ece24602f73630f34a7c7c62d3f9a786a90000000000225120d8f4f1d18996db61bb1caf5423de25d8a1eee9c950d37e6ffa023012d5e62e0d01410ab0333ae38de68f02057a4523be1ff372130cbf9b43d5c76400d8bb74033c1fa100d572814cdfbc74bc450425f16a86e59f79ffad554423497f6507f9c6d6f40100000000",
  "blockhash": "00000000000000264e792f0a5626ec3f3a1bc53fe63d3a88d920945499946164",
  "confirmations": 4698,
  "time": 1631881064,
  "blocktime": 1631881064
};

export const speculosP2pkhWithChange = {
  "txid": "6eb7ece9e847574ca460414b6c2db535cec29ba803cd79d2c0718bd66c289fba",
  "hash": "6eb7ece9e847574ca460414b6c2db535cec29ba803cd79d2c0718bd66c289fba",
  "version": 2,
  "size": 372,
  "vsize": 372,
  "weight": 1488,
  "locktime": 1901785,
  "vin": [
    {
      "txid": "347efbd10c20a540f677bdf616f06652215497a444a20127dad1520ce2b9f75d",
      "vout": 0,
      "scriptSig": {
        "asm": "30440220186ec195f04cfedc62d6d383a855e4b0fec550efedd22c177d2031b293556c16022039f05b1b8cc40c48d07323fdf3cc0453909d54409551c7ddc259210b2862bc54[ALL] 03ed3cf038f00b7ad1c3998e66cf22a688ab5dbaed8400784cbf962d54cd42c2bf",
        "hex": "4730440220186ec195f04cfedc62d6d383a855e4b0fec550efedd22c177d2031b293556c16022039f05b1b8cc40c48d07323fdf3cc0453909d54409551c7ddc259210b2862bc54012103ed3cf038f00b7ad1c3998e66cf22a688ab5dbaed8400784cbf962d54cd42c2bf"
      },
      "sequence": 4294967293
    },
    {
      "txid": "de390b0c4ac67b6cdcbe84611463cc42b6774f2bf11d8f7144e9601e3591f857",
      "vout": 1,
      "scriptSig": {
        "asm": "304402201592969826e01baba12c9833584e0e958c1562736192bd1264d079383e8f277902204d5aafb9566a6de419b9446536ccce62f750361530d2a8f9a3bbc5e47af68590[ALL] 035879ca173a9c1b3f300ec587fb4cc6d54d618e30584e425c1b53b98828708f1d",
        "hex": "47304402201592969826e01baba12c9833584e0e958c1562736192bd1264d079383e8f277902204d5aafb9566a6de419b9446536ccce62f750361530d2a8f9a3bbc5e47af685900121035879ca173a9c1b3f300ec587fb4cc6d54d618e30584e425c1b53b98828708f1d"
      },
      "sequence": 4294967293
    }
  ],
  "vout": [
    {
      "value": 0.00299100,
      "n": 0,
      "scriptPubKey": {
        "asm": "OP_DUP OP_HASH160 d8bb03d84a3aa993ae6db1114488488dab43d6db OP_EQUALVERIFY OP_CHECKSIG",
        "hex": "76a914d8bb03d84a3aa993ae6db1114488488dab43d6db88ac",
        "address": "n1GvHX9wu2cYM9RSh8cwXiGWgEyXBx8BL3",
        "type": "pubkeyhash"
      }
    },
    {
      "value": 0.00500000,
      "n": 1,
      "scriptPubKey": {
        "asm": "OP_DUP OP_HASH160 cbae5b50cf939e6f531b8a6b7abd788fe14b0297 OP_EQUALVERIFY OP_CHECKSIG",
        "hex": "76a914cbae5b50cf939e6f531b8a6b7abd788fe14b029788ac",
        "address": "mz5vLWdM1wHVGSmXUkhKVvZbJ2g4epMXSm",
        "type": "pubkeyhash"
      }
    }
  ],
  "hex": "02000000025df7b9e20c52d1da2701a244a49754215266f016f6bd77f640a5200cd1fb7e34000000006a4730440220186ec195f04cfedc62d6d383a855e4b0fec550efedd22c177d2031b293556c16022039f05b1b8cc40c48d07323fdf3cc0453909d54409551c7ddc259210b2862bc54012103ed3cf038f00b7ad1c3998e66cf22a688ab5dbaed8400784cbf962d54cd42c2bffdffffff57f891351e60e944718f1df12b4f77b642cc63146184bedc6c7bc64a0c0b39de010000006a47304402201592969826e01baba12c9833584e0e958c1562736192bd1264d079383e8f277902204d5aafb9566a6de419b9446536ccce62f750361530d2a8f9a3bbc5e47af685900121035879ca173a9c1b3f300ec587fb4cc6d54d618e30584e425c1b53b98828708f1dfdffffff025c900400000000001976a914d8bb03d84a3aa993ae6db1114488488dab43d6db88ac20a10700000000001976a914cbae5b50cf939e6f531b8a6b7abd788fe14b029788acd9041d00",
  "blockhash": "00000000000000194f45881fa2d21b1cb631398a20cad5656675e59b7c60bbc3",
  "confirmations": 197740,
  "time": 1609532404,
  "blocktime": 1609532404
};
