"use strict";
// Copyright (c) 2023 Jose-Luis Landabaso - https://bitcoinerlab.com
// Distributed under the MIT software license
Object.defineProperty(exports, "__esModule", { value: true });
exports.ledgerPolicyFromState = exports.comparePolicies = exports.ledgerPolicyFromStandard = exports.registerLedgerWallet = exports.descriptorToLedgerFormat = exports.getLedgerXpub = exports.getLedgerMasterFingerPrint = exports.assertLedgerApp = exports.importAndValidateLedgerBitcoin = void 0;
const bitcoinjs_lib_1 = require("bitcoinjs-lib");
const re_1 = require("./re");
/**
 * Dynamically imports the 'ledger-bitcoin' module and, if provided, checks if `ledgerClient` is an instance of `AppClient`.
 *
 * @async
 * @param {unknown} ledgerClient - An optional parameter that, if provided, is checked to see if it's an instance of `AppClient`.
 * @throws {Error} Throws an error if `ledgerClient` is provided but is not an instance of `AppClient`.
 * @throws {Error} Throws an error if the 'ledger-bitcoin' module cannot be imported. This typically indicates that the 'ledger-bitcoin' peer dependency is not installed.
 * @returns {Promise<unknown>} Returns a promise that resolves with the entire 'ledger-bitcoin' module if it can be successfully imported. We force it to return an unknown type so that the declaration of this function won't break projects that don't use ledger-bitcoin as dependency
 *
 * @example
 *
 * importAndValidateLedgerBitcoin(ledgerClient)
 *   .then((module) => {
 *     const { AppClient, PsbtV2, DefaultWalletPolicy, WalletPolicy, DefaultDescriptorTemplate, PartialSignature } = module;
 *     // Use the imported objects...
 *   })
 *   .catch((error) => console.error(error));
 */
async function importAndValidateLedgerBitcoin(ledgerClient) {
    let ledgerBitcoinModule;
    try {
        // Originally, the code used dynamic imports:
        // ledgerBitcoinModule = await import('ledger-bitcoin');
        // However, in React Native with the Metro bundler, there's an issue with
        // recognizing dynamic imports inside try-catch blocks. For details, refer to:
        // https://github.com/react-native-community/discussions-and-proposals/issues/120
        // The dynamic import gets transpiled to:
        // ledgerBitcoinModule = Promise.resolve().then(() => __importStar(require('ledger-bitcoin')));
        // Metro bundler fails to recognize the above as conditional. Hence, it tries
        // to require 'ledger-bitcoin' unconditionally, leading to potential errors if
        // 'ledger-bitcoin' is not installed (given it's an optional peerDependency).
        // To bypass this, we directly use require:
        ledgerBitcoinModule = require('ledger-bitcoin');
    }
    catch (error) {
        throw new Error('Could not import "ledger-bitcoin". This is a peer dependency and needs to be installed explicitly. Please run "npm install ledger-bitcoin" to use Ledger Hardware Wallet functionality.');
    }
    const { AppClient } = ledgerBitcoinModule;
    if (ledgerClient !== undefined && !(ledgerClient instanceof AppClient)) {
        throw new Error('Error: invalid AppClient instance');
    }
    return ledgerBitcoinModule;
}
exports.importAndValidateLedgerBitcoin = importAndValidateLedgerBitcoin;
// eslint-disable-next-line @typescript-eslint/no-explicit-any
async function ledgerAppInfo(transport) {
    const r = await transport.send(0xb0, 0x01, 0x00, 0x00);
    let i = 0;
    const format = r[i++];
    const nameLength = r[i++];
    const name = r.slice(i, (i += nameLength)).toString('ascii');
    const versionLength = r[i++];
    const version = r.slice(i, (i += versionLength)).toString('ascii');
    const flagLength = r[i++];
    const flags = r.slice(i, (i += flagLength));
    return { name, version, flags, format };
}
async function assertLedgerApp({ transport, name, minVersion }) {
    const { name: openName, version } = await ledgerAppInfo(transport);
    if (openName !== name) {
        throw new Error(`Open the ${name} app and try again`);
    }
    else {
        const [mVmajor, mVminor, mVpatch] = minVersion.split('.').map(Number);
        const [major, minor, patch] = version.split('.').map(Number);
        if (mVmajor === undefined ||
            mVminor === undefined ||
            mVpatch === undefined) {
            throw new Error(`Pass a minVersion using semver notation: major.minor.patch`);
        }
        if (major < mVmajor ||
            (major === mVmajor && minor < mVminor) ||
            (major === mVmajor && minor === mVminor && patch < mVpatch))
            throw new Error(`Error: please upgrade ${name} to version ${minVersion}`);
    }
}
exports.assertLedgerApp = assertLedgerApp;
function isLedgerStandard({ ledgerTemplate, keyRoots, network = bitcoinjs_lib_1.networks.bitcoin }) {
    if (keyRoots.length !== 1)
        return false;
    const originPath = keyRoots[0]?.match(re_1.reOriginPath)?.[1];
    if (!originPath)
        return false;
    //Network is the 6th character: /44'/0'
    if (originPath[5] !== (network === bitcoinjs_lib_1.networks.bitcoin ? '0' : '1'))
        return false;
    if ((ledgerTemplate === 'pkh(@0/**)' &&
        originPath.match(/^\/44'\/[01]'\/(\d+)'$/)) ||
        (ledgerTemplate === 'wpkh(@0/**)' &&
            originPath.match(/^\/84'\/[01]'\/(\d+)'$/)) ||
        (ledgerTemplate === 'sh(wpkh(@0/**))' &&
            originPath.match(/^\/49'\/[01]'\/(\d+)'$/)) ||
        (ledgerTemplate === 'tr(@0/**)' &&
            originPath.match(/^\/86'\/[01]'\/(\d+)'$/)))
        return true;
    return false;
}
async function getLedgerMasterFingerPrint({ ledgerClient, ledgerState }) {
    const { AppClient } = (await importAndValidateLedgerBitcoin(ledgerClient));
    if (!(ledgerClient instanceof AppClient))
        throw new Error(`Error: pass a valid ledgerClient`);
    let masterFingerprint = ledgerState.masterFingerprint;
    if (!masterFingerprint) {
        masterFingerprint = Buffer.from(await ledgerClient.getMasterFingerprint(), 'hex');
        ledgerState.masterFingerprint = masterFingerprint;
    }
    return masterFingerprint;
}
exports.getLedgerMasterFingerPrint = getLedgerMasterFingerPrint;
async function getLedgerXpub({ originPath, ledgerClient, ledgerState }) {
    const { AppClient } = (await importAndValidateLedgerBitcoin(ledgerClient));
    if (!(ledgerClient instanceof AppClient))
        throw new Error(`Error: pass a valid ledgerClient`);
    if (!ledgerState.xpubs)
        ledgerState.xpubs = {};
    let xpub = ledgerState.xpubs[originPath];
    if (!xpub) {
        try {
            //Try getting the xpub without user confirmation
            xpub = await ledgerClient.getExtendedPubkey(`m${originPath}`, false);
        }
        catch (err) {
            xpub = await ledgerClient.getExtendedPubkey(`m${originPath}`, true);
        }
        if (typeof xpub !== 'string')
            throw new Error(`Error: ledgerClient did not return a valid xpub`);
        ledgerState.xpubs[originPath] = xpub;
    }
    return xpub;
}
exports.getLedgerXpub = getLedgerXpub;
/**
 * Takes a descriptor and gets its Ledger Wallet Policy, that is, its keyRoots and template.
 * keyRoots and template follow Ledger's specifications:
 * https://github.com/LedgerHQ/app-bitcoin-new/blob/develop/doc/wallet.md
 *
 * keyRoots and template are a generalization of a descriptor and serve to
 * describe internal and external addresses and any index.
 *
 * So, this function starts from a descriptor and obtains generalized Ledger
 * wallet policy.
 *
 * keyRoots is an array of strings, encoding xpub-type key expressions up to the origin.
 * F.ex.: [76223a6e/48'/1'/0'/2']tpubDE7NQymr4AFtewpAsWtnreyq9ghkzQBXpCZjWLFVRAvnbf7vya2eMTvT2fPapNqL8SuVvLQdbUbMfWLVDCZKnsEBqp6UK93QEzL8Ck23AwF
 *
 * Template encodes the descriptor script expression, where its key
 * expressions are represented using variables for each keyRoot and finished with "/**"
 * (for change 1 or 0 and any index). F.ex.:
 * wsh(sortedmulti(2,@0/**,@1/**)), where @0 corresponds the first element in the keyRoots array.
 *
 * If this descriptor does not contain any key that can be signed with the ledger
 * (non-matching masterFingerprint), then this function returns null.
 *
 * This function takes into account all the considerations regarding Ledger
 * policy implementation details expressed in the header of this file.
 */
async function descriptorToLedgerFormat({ descriptor, ledgerClient, ledgerState }) {
    const expandedExpression = descriptor.expand().expandedExpression;
    const expansionMap = descriptor.expand().expansionMap;
    if (!expandedExpression || !expansionMap)
        throw new Error(`Error: invalid descriptor`);
    const ledgerMasterFingerprint = await getLedgerMasterFingerPrint({
        ledgerClient,
        ledgerState
    });
    //It's important to have keys sorted in ascii order. keys
    //are of this type: @0, @1, @2, ....  and they also appear in the expandedExpression
    //in ascending ascii order. Note that Object.keys(expansionMap ) does not ensure
    //that the order is respected and so we force it.
    const allKeys = Object.keys(expansionMap).sort();
    const ledgerKeys = allKeys.filter(key => {
        const masterFingerprint = expansionMap[key]?.masterFingerprint;
        return (masterFingerprint &&
            Buffer.compare(masterFingerprint, ledgerMasterFingerprint) === 0);
    });
    if (ledgerKeys.length === 0)
        return null;
    if (ledgerKeys.length > 1)
        throw new Error(`Error: descriptor ${expandedExpression} does not contain exactly 1 ledger key`);
    const ledgerKey = ledgerKeys[0];
    const masterFingerprint = expansionMap[ledgerKey].masterFingerprint;
    const originPath = expansionMap[ledgerKey].originPath;
    const keyPath = expansionMap[ledgerKey].keyPath;
    const bip32 = expansionMap[ledgerKey].bip32;
    if (!masterFingerprint || !originPath || !keyPath || !bip32) {
        throw new Error(`Error: Ledger key expression must have a valid masterFingerprint: ${masterFingerprint}, originPath: ${originPath}, keyPath: ${keyPath} and a valid bip32 node`);
    }
    if (!/^\/[01]\/\d+$/.test(keyPath))
        throw new Error(`Error: key paths must be /<1;0>/index, where change is 1 or 0 and index >= 0`);
    const keyRoots = [];
    let ledgerTemplate = expandedExpression;
    allKeys.forEach(key => {
        if (key !== ledgerKey) {
            //This block here only does data integrity assertions:
            const otherKeyInfo = expansionMap[key];
            if (!otherKeyInfo.bip32) {
                throw new Error(`Error: ledger only allows xpub-type key expressions`);
            }
            if (otherKeyInfo.originPath) {
                if (otherKeyInfo.originPath !== originPath) {
                    throw new Error(`Error: all originPaths must be the same for Ledger being able to sign. On the other hand, you can leave the origin info empty for external keys: ${otherKeyInfo.originPath} !== ${originPath}`);
                }
            }
            if (otherKeyInfo.keyPath !== keyPath) {
                throw new Error(`Error: all keyPaths must be the same for Ledger being able to sign: ${otherKeyInfo.keyPath} !== ${keyPath}`);
            }
        }
        ledgerTemplate = ledgerTemplate.replaceAll(key, `@${keyRoots.length}/**`);
        const keyInfo = expansionMap[key];
        if (keyInfo.masterFingerprint && keyInfo.originPath)
            keyRoots.push(`[${keyInfo.masterFingerprint?.toString('hex')}${keyInfo.originPath}]${keyInfo?.bip32?.neutered().toBase58()}`);
        else
            keyRoots.push(`${keyInfo?.bip32?.neutered().toBase58()}`);
    });
    return { ledgerTemplate, keyRoots };
}
exports.descriptorToLedgerFormat = descriptorToLedgerFormat;
/**
 * It registers a policy based on a descriptor. It stores it in ledgerState.
 *
 * If the policy was already registered, it does not register it.
 * If the policy is standard, it does not register it.
 *
 **/
async function registerLedgerWallet({ descriptor, ledgerClient, ledgerState, policyName }) {
    const { WalletPolicy, AppClient } = (await importAndValidateLedgerBitcoin(ledgerClient));
    if (!(ledgerClient instanceof AppClient))
        throw new Error(`Error: pass a valid ledgerClient`);
    const result = await descriptorToLedgerFormat({
        descriptor,
        ledgerClient,
        ledgerState
    });
    if (await ledgerPolicyFromStandard({ descriptor, ledgerClient, ledgerState }))
        return;
    if (!result)
        throw new Error(`Error: descriptor does not have a ledger input`);
    const { ledgerTemplate, keyRoots } = result;
    if (!ledgerState.policies)
        ledgerState.policies = [];
    let walletPolicy, policyHmac;
    //Search in ledgerState first
    const policy = await ledgerPolicyFromState({
        descriptor,
        ledgerClient,
        ledgerState
    });
    if (policy) {
        if (policy.policyName !== policyName)
            throw new Error(`Error: policy was already registered with a different name: ${policy.policyName}`);
        //It already existed. No need to register it again.
    }
    else {
        walletPolicy = new WalletPolicy(policyName, ledgerTemplate, keyRoots);
        let policyId;
        [policyId, policyHmac] = await ledgerClient.registerWallet(walletPolicy);
        const policy = {
            policyName,
            ledgerTemplate,
            keyRoots,
            policyId,
            policyHmac
        };
        ledgerState.policies.push(policy);
    }
}
exports.registerLedgerWallet = registerLedgerWallet;
/**
 * Retrieve a standard ledger policy or null if it does correspond.
 **/
async function ledgerPolicyFromStandard({ descriptor, ledgerClient, ledgerState }) {
    const result = await descriptorToLedgerFormat({
        descriptor,
        ledgerClient,
        ledgerState
    });
    if (!result)
        throw new Error(`Error: descriptor does not have a ledger input`);
    const { ledgerTemplate, keyRoots } = result;
    if (isLedgerStandard({
        ledgerTemplate,
        keyRoots,
        network: descriptor.getNetwork()
    }))
        return { ledgerTemplate, keyRoots };
    return null;
}
exports.ledgerPolicyFromStandard = ledgerPolicyFromStandard;
function compareKeyRoots(arr1, arr2) {
    if (arr1.length !== arr2.length) {
        return false;
    }
    for (let i = 0; i < arr1.length; i++) {
        if (arr1[i] !== arr2[i]) {
            return false;
        }
    }
    return true;
}
function comparePolicies(policyA, policyB) {
    return (compareKeyRoots(policyA.keyRoots, policyB.keyRoots) &&
        policyA.ledgerTemplate === policyB.ledgerTemplate);
}
exports.comparePolicies = comparePolicies;
/**
 * Retrieve a ledger policy from ledgerState or null if it does not exist yet.
 **/
async function ledgerPolicyFromState({ descriptor, ledgerClient, ledgerState }) {
    const result = await descriptorToLedgerFormat({
        descriptor,
        ledgerClient,
        ledgerState
    });
    if (!result)
        throw new Error(`Error: descriptor does not have a ledger input`);
    const { ledgerTemplate, keyRoots } = result;
    if (!ledgerState.policies)
        ledgerState.policies = [];
    //Search in ledgerState:
    const policies = ledgerState.policies.filter(policy => comparePolicies(policy, { ledgerTemplate, keyRoots }));
    if (policies.length > 1)
        throw new Error(`Error: duplicated policy`);
    if (policies.length === 1) {
        return policies[0];
    }
    else {
        return null;
    }
}
exports.ledgerPolicyFromState = ledgerPolicyFromState;
