"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createTx = void 0;
const tslib_1 = require("tslib");
const bs58check_1 = tslib_1.__importDefault(require("bs58check"));
const schema_utils_1 = require("@trezor/schema-utils");
const constants_1 = require("../../constants");
const tezos_1 = require("../../types/api/tezos");
const PREFIX = {
    B: new Uint8Array([1, 52]),
    tz1: new Uint8Array([6, 161, 159]),
    tz2: new Uint8Array([6, 161, 161]),
    tz3: new Uint8Array([6, 161, 164]),
    KT1: new Uint8Array([2, 90, 121]),
    edpk: new Uint8Array([13, 15, 37, 217]),
    sppk: new Uint8Array([3, 254, 226, 86]),
    p2pk: new Uint8Array([3, 178, 139, 127]),
};
const bs58checkDecode = (prefix, enc) => bs58check_1.default.decode(enc).slice(prefix.length);
const concatArray = (first, second) => {
    const result = new Uint8Array(first.length + second.length);
    result.set(first);
    result.set(second, first.length);
    return result;
};
const publicKeyHash2buffer = (publicKeyHash) => {
    switch (publicKeyHash.substring(0, 3)) {
        case 'tz1':
            return {
                originated: 0,
                hash: concatArray(new Uint8Array([0]), bs58checkDecode(PREFIX.tz1, publicKeyHash)),
            };
        case 'tz2':
            return {
                originated: 0,
                hash: concatArray(new Uint8Array([1]), bs58checkDecode(PREFIX.tz2, publicKeyHash)),
            };
        case 'tz3':
            return {
                originated: 0,
                hash: concatArray(new Uint8Array([2]), bs58checkDecode(PREFIX.tz3, publicKeyHash)),
            };
        case 'KT1':
            return {
                originated: 1,
                hash: concatArray(bs58checkDecode(PREFIX.KT1, publicKeyHash), new Uint8Array([0])),
            };
        default:
            throw constants_1.ERRORS.TypedError('Method_InvalidParameter', 'Wrong Tezos publicKeyHash address');
    }
};
const publicKey2buffer = (publicKey) => {
    switch (publicKey.substring(0, 4)) {
        case 'edpk':
            return concatArray(new Uint8Array([0]), bs58checkDecode(PREFIX.edpk, publicKey));
        case 'sppk':
            return concatArray(new Uint8Array([1]), bs58checkDecode(PREFIX.sppk, publicKey));
        case 'p2pk':
            return concatArray(new Uint8Array([2]), bs58checkDecode(PREFIX.p2pk, publicKey));
        default:
            throw constants_1.ERRORS.TypedError('Method_InvalidParameter', 'Wrong Tezos publicKey');
    }
};
const createTx = (address_n, branch, operation, chunkify) => {
    let message = {
        address_n,
        branch: bs58checkDecode(PREFIX.B, branch),
        chunkify: typeof chunkify === 'boolean' ? chunkify : false,
    };
    (0, schema_utils_1.Assert)(tezos_1.TezosOperation, operation);
    if (operation.reveal) {
        const { reveal } = operation;
        message = {
            ...message,
            reveal: {
                source: publicKeyHash2buffer(reveal.source).hash,
                public_key: publicKey2buffer(reveal.public_key),
                fee: reveal.fee,
                counter: reveal.counter,
                gas_limit: reveal.gas_limit,
                storage_limit: reveal.storage_limit,
            },
        };
    }
    if (operation.transaction) {
        const { transaction } = operation;
        message = {
            ...message,
            transaction: {
                source: publicKeyHash2buffer(transaction.source).hash,
                destination: {
                    tag: publicKeyHash2buffer(transaction.destination).originated,
                    hash: publicKeyHash2buffer(transaction.destination).hash,
                },
                amount: transaction.amount,
                counter: transaction.counter,
                fee: transaction.fee,
                gas_limit: transaction.gas_limit,
                storage_limit: transaction.storage_limit,
            },
        };
        if (Object.prototype.hasOwnProperty.call(transaction, 'parameters')) {
            message = {
                ...message,
                transaction: {
                    ...message.transaction,
                    parameters: transaction.parameters,
                },
            };
        }
        if (transaction.parameters_manager) {
            const { parameters_manager } = transaction;
            if (parameters_manager.set_delegate) {
                message = {
                    ...message,
                    transaction: {
                        ...message.transaction,
                        parameters_manager: {
                            set_delegate: publicKeyHash2buffer(parameters_manager.set_delegate)
                                .hash,
                        },
                    },
                };
            }
            if (Object.prototype.hasOwnProperty.call(parameters_manager, 'cancel_delegate')) {
                message = {
                    ...message,
                    transaction: {
                        ...message.transaction,
                        parameters_manager: {
                            cancel_delegate: true,
                        },
                    },
                };
            }
            if (parameters_manager.transfer) {
                const { transfer } = parameters_manager;
                message = {
                    ...message,
                    transaction: {
                        ...message.transaction,
                        parameters_manager: {
                            transfer: {
                                destination: {
                                    tag: publicKeyHash2buffer(transfer.destination).originated,
                                    hash: publicKeyHash2buffer(transfer.destination).hash,
                                },
                                amount: transfer.amount,
                            },
                        },
                    },
                };
            }
        }
    }
    if (operation.origination) {
        const { origination } = operation;
        message = {
            ...message,
            origination: {
                source: publicKeyHash2buffer(origination.source).hash,
                balance: origination.balance,
                fee: origination.fee,
                counter: origination.counter,
                gas_limit: origination.gas_limit,
                storage_limit: origination.storage_limit,
                script: origination.script,
            },
        };
        if (origination.delegate) {
            message = {
                ...message,
                origination: {
                    ...message.origination,
                    delegate: publicKeyHash2buffer(origination.delegate).hash,
                },
            };
        }
    }
    if (operation.delegation) {
        const { delegation } = operation;
        message = {
            ...message,
            delegation: {
                source: publicKeyHash2buffer(delegation.source).hash,
                delegate: publicKeyHash2buffer(delegation.delegate).hash,
                fee: delegation.fee,
                counter: delegation.counter,
                gas_limit: delegation.gas_limit,
                storage_limit: delegation.storage_limit,
            },
        };
    }
    return message;
};
exports.createTx = createTx;
//# sourceMappingURL=tezosSignTx.js.map